package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.ValvePosnState;
import org.lsst.ccs.subsystem.refrig.constants.ValveSetState;

/**
 *  Controls a heat exchanger valve.
 *
 *  @author Owen Saxton
 */
public class HexValve implements HasLifecycle {

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    ValveMaq20Device maqDevc;

    private String set;
    private String getOpen;
    private String getShut;

    private static final Logger LOG = Logger.getLogger(HexValve.class.getName());
    private int setIndex, setChan, getOpenIndex, getOpenChan, getShutIndex, getShutChan;

    /**
     *  Initializes the parameters.
     */
    @Override
    public void init()
    {
        if (maqDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 device", "not defined");
        }
        if (set == null) {
            ErrorUtils.reportConfigError(LOG, name, "set", "not specified");
        }
        int[] values = getChanParams("set", set);
        setIndex = values[0];
        setChan = values[1];
        if (!maqDevc.registerOut(setIndex, setChan)) {
            ErrorUtils.reportConfigError(LOG, name, "set", "is a duplicate or refers to incorrect module type");
        }
        if (getOpen == null) {
            ErrorUtils.reportConfigError(LOG, name, "getOpen", "not specified");
        }
        values = getChanParams("getOpen", getOpen);
        getOpenIndex = values[0];
        getOpenChan = values[1];
        if (!maqDevc.registerIn(getOpenIndex, getOpenChan)) {
            ErrorUtils.reportConfigError(LOG, name, "getOpen", "is a duplicate or refers to incorrect module type");
        }
        if (getShut == null) {
            ErrorUtils.reportConfigError(LOG, name, "getShut", "not specified");
        }
        values = getChanParams("getShut", getShut);
        getShutIndex = values[0];
        getShutChan = values[1];
        if (!maqDevc.registerIn(getShutIndex, getShutChan)) {
            ErrorUtils.reportConfigError(LOG, name, "getShut", "is a duplicate or refers to incorrect module type");
        }
    }


    /**
     *  Set the valve to open.
     */
    public void setOpen()
    {
        maqDevc.setValve(setIndex, setChan, false);
    }


    /**
     *  Set the valve to shut.
     */
    public void setShut()
    {
        maqDevc.setValve(setIndex, setChan, true);
    }


    /**
     *  Get the valve's set state.
     *
     *  This is determined from what was written to the discrete output module
     *
     *  @return  The set state
     */
    public ValveSetState getSetState()
    {
        Boolean outState = maqDevc.isValveSet(setIndex, setChan);
        return outState == null ? ValveSetState.OFFLINE : outState ? ValveSetState.SHUT : ValveSetState.OPEN;
    }


    /**
     *  Get the valve's position state.
     *
     *  This is determined from what is read from the discrete input module
     *
     *  @return  The read state
     */
    public ValvePosnState getPosnState()
    {
        Boolean openState = maqDevc.isPositionSet(getOpenIndex, getOpenChan);
        Boolean shutState = maqDevc.isPositionSet(getShutIndex, getShutChan);
        return openState == null || shutState == null ? ValvePosnState.OFFLINE :
               openState ? shutState ? ValvePosnState.BOTH : ValvePosnState.OPEN
                         : shutState ? ValvePosnState.SHUT : ValvePosnState.NONE;
    }


    /**
     *  Gets a channel's parameters from its string description.
     *
     *  @param  chanName  The name of the channel
     *  @param  chanDesc  The channel description
    */
    private int[] getChanParams(String chanName, String chanDesc)
    {
        String[] fields = chanDesc.split(":");
        if (fields.length != 2) {
            ErrorUtils.reportConfigError(LOG, name, chanName, "must have two colon-separated values");
        }
        int[] values = new int[2];
        for (int j = 0; j < 2; j++) {
            try {
                values[j] = Integer.decode(fields[j]);
            }
            catch (NumberFormatException e) {
                ErrorUtils.reportConfigError(LOG, name, chanName, "must specify integer values");
            }
        }
        if (values[0] < 0 || values[0] >= maqDevc.getModuleCount()) {
            ErrorUtils.reportConfigError(LOG, name, chanName, "has invalid module index");
        }
        if (values[1] < 0 || values[1] >= ValveMaq20Device.NUM_CHAN_DINP) {
            ErrorUtils.reportConfigError(LOG, name, chanName, "has invalid channel number");
        }
        return values;
    }

}
