package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.ColdState;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.AuxHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.ThermalAlert;
import org.lsst.ccs.subsystem.refrig.constants.ThermalConstants;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterOpState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;

/**
 *  Implements the refrigeration thermal control subsystem.
 *
 *  @author Owen Saxton
 */
public class ThermalMain implements HasLifecycle, AgentPresenceListener, StatusMessageListener, ClearAlertHandler {

    /**
     *  Inner class for holding trim heater parameters.
     */
    static class TrimHeater {

        TrimHeaterControl tempCtrl;
        PowerDevice powerDevc;
        int powerChan;

        TrimHeater(TrimHeaterControl tempCtrl, PowerDevice powerDevc, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Inner class for holding auxiliary heater parameters.
     */
    static class AuxHeater {

        PowerDevice powerDevc;
        int powerChan;

        AuxHeater(PowerDevice powerDevc, int powerChan)
        {
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Constants.
     */
    private static final String THERMAL_LIMITS = "ThermalLimits";

    private static final int
        UPDATE_STATE_INTVL = 1000;

    private static final List<Set<Integer>> validTrimHeaterSets = new ArrayList<>();
    static {
        Set<Integer> trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
        trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYE);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_C);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYE);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
        trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYE);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYC);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYC);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYE);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, TrimHeaterControl> tempControlMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, PowerDevice> powerDeviceMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, Channel> allChannels = new HashMap<>();

    @ConfigurationParameter(category=THERMAL_LIMITS, isFinal=true)
    private volatile double coldTempLimit = 40.0;

    private String[] auxPowerNames;       // Names of the auxiliary heater power objects (3)
    private String[] coldTempNames;       // Names of the cold plate temperature channels

    private static final Logger LOG = Logger.getLogger(ThermalMain.class.getName());
    private Set<Integer> trimHeaterSet;
    private final Set<Integer> auxHeaterSet = new HashSet<>();
    private final TrimHeater[] trimHeaters = new TrimHeater[ThermalState.NUM_TRIM_HEATERS];
    private final AuxHeater[] auxHeaters = new AuxHeater[ThermalState.NUM_AUX_HEATERS];
    private final List<Channel> coldTemps = new ArrayList<>();
    private final ThermalState thermalState = new ThermalState();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private int highTempCount = 0;
    private ColdState refrigColdState = ColdState.UNKNOWN;
    private String refrigGroup = null;


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the refrigeration state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("thermal-state", () -> updateThermalState()).withPeriod(Duration.ofMillis(UPDATE_STATE_INTVL));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the thermal subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a thermal subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.THERMAL_TYPE, ThermalMain.class.getCanonicalName());

        // Add an agent present listener
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);

        // General initialization
        thermalState.setTickMillis(getTickPeriod());
        refrigGroup = RefrigUtils.getGroupName(subsys.getAgentInfo());

        // Initialize trim heater control
        Set<Integer> heaters = new HashSet<>();
        for (TrimHeaterControl ctrl : tempControlMap.values()) {
            int htr = ctrl.getPowerChannel();
            heaters.add(htr);
            trimHeaters[htr] = new TrimHeater(ctrl, ctrl.getPowerDevice(), htr);
        }
        for (Set heaterSet : validTrimHeaterSets) {
            if (heaterSet.equals(heaters)) {
                trimHeaterSet = heaterSet;
                break;
            }
        }
        if (trimHeaterSet == null) {
            ErrorUtils.reportConfigError(LOG, name, "trim heater loop set", "is not a valid one");
        }
        thermalState.setTrimHeaters(trimHeaterSet.toArray(thermalState.getTrimHeaters()));

        // Initialize auxiliary heater control
        if (auxPowerNames == null) {
            LOG.info("No auxiliary heater devices specified");
        }
        else if (auxPowerNames.length > ThermalState.NUM_AUX_HEATERS) {
            ErrorUtils.reportConfigError(LOG, name, "auxPowerNames", "contains more than 3 items");
        }
        else {
            for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
                String desc = htr < auxPowerNames.length ? auxPowerNames[htr]: null;
                if (desc == null) {
                    LOG.log(Level.INFO, "Auxiliary heater {0} has not been specified", htr);
                }
                else {
                    String[] words = desc.split(":");
                    if (words.length > 2) {
                        ErrorUtils.reportConfigError(LOG, name, "auxPowerNames", "contains invalid item: " + desc);
                    }
                    PowerDevice power = powerDeviceMap.get(words[0]);
                    if (power == null) {
                        ErrorUtils.reportConfigError(LOG, name, "auxPowerNames", "contains non-power item: " + words[0]);
                    }
                    int channel = 0;
                    if (words.length == 2) {
                        try {
                            channel = Integer.decode(words[1]);
                        }
                        catch (NumberFormatException e) {
                            ErrorUtils.reportConfigError(LOG, name, "auxPowerNames", "contains invalid channel: " + words[1]);
                        }
                    }
                    auxHeaterSet.add(htr);
                    auxHeaters[htr] = new AuxHeater(power, channel);
                }
            }
        }
        thermalState.setAuxHeaters(auxHeaterSet.toArray(thermalState.getAuxHeaters()));

        // Initialize cold temperature channel list
        if (coldTempNames == null) {
            LOG.severe("No cold-plate temperature monitoring channels specified");
        }
        else {
            for (String coldName : coldTempNames) {
                Channel chan = allChannels.get(coldName);
                if (chan == null) {
                    ErrorUtils.reportConfigError(LOG, name, "coldTempNames", "contains non-channel item");
                }
                coldTemps.add(chan);
            }
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Set up initial states for the trim heaters
        for (int htr : trimHeaterSet) {
            TrimHeater heater = trimHeaters[htr];
            thermalState.setTrimHeaterState(htr, !heater.powerDevc.isOnline() ? TrimHeaterState.OFFLINE :
                                                 heater.powerDevc.isEnabled(heater.powerChan) ? TrimHeaterState.POWER
                                                                                              : TrimHeaterState.OFF);
            heater.tempCtrl.setPower(thermalState.getTrimHeaterPower(htr));
            heater.tempCtrl.setTemp(thermalState.getPlateTemperature(htr));
        }

        // Set up initial states for the aux heaters
        for (int htr : auxHeaterSet) {
            AuxHeater heater = auxHeaters[htr];
            if (heater != null) {
                thermalState.setAuxHeaterState(htr, !heater.powerDevc.isOnline() ? AuxHeaterState.OFFLINE :
                                                    heater.powerDevc.isEnabled(heater.powerChan) ? AuxHeaterState.ON
                                                                                                 : AuxHeaterState.OFF);
            }
        }

        // Announce the system
        LOG.info("Thermal control system started");
    }


    /**
     *  Listens for the arrival of the companion refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                if (refrigGroup.equals(RefrigUtils.getGroupName(agent))) {
                    subsys.getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName));
                    break;
                }
            }
        }
    }


    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE) && refrigGroup.equals(RefrigUtils.getGroupName(agent))) {
                subsys.getMessagingAccess().removeStatusMessageListener(this);
                //refrigColdState = ColdState.UNKNOWN;
            }
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        refrigColdState = msg.getState().getState(ColdState.class);
    }


    /**
     *  Gets the state of the thermal control system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=CommandType.QUERY, description="Get the thermal control state", level=0)
    public ThermalState getSystemState()
    {
        return thermalState;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        thermalState.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Sets a trim heater operation state.
     *
     *  @param  htr    The heater number
     *  @param  state  The heater operation state to set: OFF, POWER or TEMP.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater operation state")
    public void setTrimHeaterState(int htr, TrimHeaterOpState state) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                TrimHeaterOpState oldState = thermalState.getTrimHeaterOpState(htr);
                if (state != oldState & thermalState.getTrimHeaterState(htr) != TrimHeaterState.DISABLD) {
                    thermalState.setTrimHeaterOpState(htr, state);
                    heater.tempCtrl.setOpState(state);
                }
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets a trim heater power value.
     *
     *  @param  htr    The heater number
     *  @param  power  The load power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater power set point")
    public void setTrimHeaterPower(int htr, double power) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                thermalState.setTrimHeaterPower(htr, power);
                heater.tempCtrl.setPower(power);
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets a plate temperature value.
     *
     *  @param  htr   The heater number
     *  @param  temp  The plate temperature set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a plate temperature set point")
    public void setPlateTemperature(int htr, double temp) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                thermalState.setPlateTemperature(htr, temp);
                heater.tempCtrl.setTemp(temp);
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets an aux heater state.
     *
     *  @param  htr    The heater number
     *  @param  value  The enabled state value to set: 0 = off, ~0 = on.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set an aux heater power enabled state")
    public void setAuxHeaterState(int htr, int value) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                getAuxHeater(htr);
                AuxHeaterState oldState = thermalState.getAuxHeaterState(htr);
                if (oldState != AuxHeaterState.OFFLINE && oldState != AuxHeaterState.DISABLD) {
                    AuxHeaterState newState = (value != 0) ? AuxHeaterState.ON : AuxHeaterState.OFF;
                    if (newState != oldState) {
                        thermalState.setAuxHeaterState(htr, newState);
                        setAuxHeaterPower(htr);
                    }
                }
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets an aux heater power value.
     *
     *  @param  htr    The heater number
     *  @param  value  The power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set an aux heater power set point")
    public void setAuxHeaterPower(int htr, double value) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                getAuxHeater(htr);
                thermalState.setAuxHeaterPower(htr, value);
                setAuxHeaterPower(htr);
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Publishes the state of the thermal module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ThermalState.KEY, thermalState));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Updates the thermal state periodically.
     */
    private void updateThermalState()
    {
        boolean coldHigh = false;
        for (Channel chan : coldTemps) {
            coldHigh |= (chan.getValue() > coldTempLimit);
        }
        if (coldHigh) {
            coldHigh &= (++highTempCount >= 3);
        }
        else {
            highTempCount = 0;
        }
        synchronized(thermalState) {
            boolean changed = false;
            for (int htr : trimHeaterSet) {
                TrimHeater heater = trimHeaters[htr];
                TrimHeaterState oldState = thermalState.getTrimHeaterState(htr);
                TrimHeaterState state = coldHigh && htr != ThermalState.TRIM_HEATER_CRYO ? TrimHeaterState.DISABLD :
                                        !heater.powerDevc.isOnline() ? TrimHeaterState.OFFLINE :
                                        heater.tempCtrl.isOverTemp() ? TrimHeaterState.OVRTEMP :
                                        !heater.powerDevc.isEnabled(heater.powerChan) ? TrimHeaterState.OFF :
                                        heater.powerDevc.hasVoltError(heater.powerChan) ? TrimHeaterState.VOLTERR :
                                        heater.powerDevc.hasNoLoad(heater.powerChan) ? TrimHeaterState.NOLOAD :
                                        thermalState.getTrimHeaterOpState(htr) == TrimHeaterOpState.TEMP ? TrimHeaterState.TEMP
                                                                                                         : TrimHeaterState.POWER;
                if (state != oldState) {
                    thermalState.setTrimHeaterState(htr, state);
                    if (state == TrimHeaterState.DISABLD) {
                        thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.OFF);
                        heater.tempCtrl.setOpState(TrimHeaterOpState.OFF);
                    }
                    changed = true;
                }
            }
            boolean coldCompsOff = refrigColdState == ColdState.BOTH_OFF || refrigColdState == ColdState.UNKNOWN;
            for (int htr : auxHeaterSet) {
                AuxHeater heater = auxHeaters[htr];
                AuxHeaterState state = (coldHigh || coldCompsOff) ? AuxHeaterState.DISABLD :
                                       !heater.powerDevc.isOnline() ? AuxHeaterState.OFFLINE :
                                       !heater.powerDevc.isEnabled(heater.powerChan) ? AuxHeaterState.OFF :
                                       heater.powerDevc.hasVoltError(heater.powerChan) ? AuxHeaterState.VOLTERR : 
                                       heater.powerDevc.hasNoLoad(heater.powerChan) ? AuxHeaterState.NOLOAD :
                                       AuxHeaterState.ON;
                if (state != thermalState.getAuxHeaterState(htr)) {
                    thermalState.setAuxHeaterState(htr, state);
                    setAuxHeaterPower(htr);
                    changed = true;
                }
            }
            if (changed) {
                publishState();
                if (coldHigh) {
                    raiseAlert(ThermalAlert.COLD_TEMP_HIGH, AlertState.ALARM, "At least one cold-plate RTD is above " + coldTempLimit + " C");
                }
                else {
                    lowerAlert(ThermalAlert.COLD_TEMP_HIGH, "No cold-plate RTD is above " + coldTempLimit + " C");
                }
                if (coldCompsOff) {
                    raiseAlert(ThermalAlert.COLD_COMP_OFF, AlertState.ALARM, "Both cold compressors are turned off");
                }
                else {
                    lowerAlert(ThermalAlert.COLD_COMP_OFF, "At least one cold compressor is back on");
                }
            }
        }
    }


    /**
     *  Sets the power for an aux heater.
     */
    private void setAuxHeaterPower(int htr)
    {
        AuxHeater heater = auxHeaters[htr];
        if (ThermalConstants.auxOnStates.contains(thermalState.getAuxHeaterState(htr))) {
            heater.powerDevc.enableOutput(heater.powerChan, true);
            heater.powerDevc.setPower(heater.powerChan, thermalState.getAuxHeaterPower(htr));
        }
        else {
            heater.powerDevc.enableOutput(heater.powerChan, false);
        }
    }
    

    /**
     *  Raises an alert.
     *
     *  @param  alert  The thermal alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     */
    private void raiseAlert(ThermalAlert alert, AlertState state, String cond)
    {
        Boolean wasActive = activeAlertMap.put(alert.getId(), true);
        if (wasActive != Boolean.TRUE) {
            alertService.raiseAlert(alert.newAlert(), state, cond);
        }
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The thermal alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(ThermalAlert alert, String cond)
    {
        Boolean wasActive = activeAlertMap.put(alert.getId(), false);
        if (wasActive == Boolean.TRUE) {
            alertService.raiseAlert(alert.newAlert(), AlertState.NOMINAL, cond);
        }
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * Return a ClearAlertCode for the provided Alert.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     * 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }
    

    /**
     *  Checks a trim heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  RefrigException if the number is invalid or object is null
     */
    private TrimHeater getTrimHeater(int htr) throws RefrigException
    {
        if (!trimHeaterSet.contains(htr)) {
            throw new RefrigException("Invalid trim heater number: " + htr);
        }
        return trimHeaters[htr];
    }
    

    /**
     *  Checks an auxiliary heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  RefrigException if the number is invalid or object is null
     */
    private AuxHeater getAuxHeater(int htr) throws RefrigException
    {
        if (!auxHeaterSet.contains(htr)) {
            throw new RefrigException("Invalid auxiliary heater number: " + htr);
        }
        return auxHeaters[htr];
    }

}
