package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;

/**
 *  Device for controlling the heat exchanger valves.
 * 
 *  @author saxton
 */
public class ValveMaq20Device extends Maq20Device {

    private static final Logger LOG = Logger.getLogger(ValveMaq20Device.class.getName());
    private int[] setMask, getMask, posnMask, outMask, inMask;
    private boolean[] offline;
    private final List<Integer> outModules = new ArrayList<>();
    private final List<Integer> inModules = new ArrayList<>();


    @Override
    protected void initDevice()
    {
        super.initDevice();
        outMask = new int[getModuleCount()];
        inMask = new int[getModuleCount()];
        setMask = new int[getModuleCount()];
        getMask = new int[getModuleCount()];
        posnMask = new int[getModuleCount()];
        offline = new boolean[getModuleCount()];
    }


    @Override
    protected void initialize()
    {
        super.initialize();
        if (!isOnline()) return;
        for (int mod = 0; mod < getModuleCount(); mod++) {
            if (outMask[mod] != 0) {
                if (getModuleData(mod).maqDiscOut == null) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 module with index {1} cannot perform discrete output", new Object[]{name, mod});
                    offline[mod] = true;
                }
                else {
                    outModules.add(mod);
                    offline[mod] = false;
                }
            }
            if (inMask[mod] != 0) {
                if (getModuleData(mod).maqDiscIn == null) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 module with index {1} cannot perform discrete input", new Object[]{name, mod});
                    offline[mod] = true;
                }
                else {
                    inModules.add(mod);
                    offline[mod] = false;
                }
            }
        }
        getSetValves();
        System.arraycopy(getMask, 0, setMask, 0, getModuleCount());
        readPositions();
    }


    public void setValves()
    {
        for (int mod : outModules) {
            if (getModuleData(mod).maqDiscOut != null) {
                try {
                    getModuleData(mod).maqDiscOut.writeDiscAll(setMask[mod]);
                    offline[mod] = false;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 write error: {1}", new Object[]{name, e});
                    offline[mod] = true;
                    setOnline(false);
                }
            }
        }
    }


    public void getSetValves()
    {
        for (int mod : outModules) {
            if (getModuleData(mod).maqDiscOut != null) {
                try {
                    getMask[mod] = getModuleData(mod).maqDiscOut.readDiscAll();
                    offline[mod] = false;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 read error: {1}", new Object[]{name, e});
                    offline[mod] = true;
                    setOnline(false);
                }
            }
        }
    }


    public void readPositions()
    {
        for (int mod : inModules) {
            if (getModuleData(mod).maqDiscOut != null) {
                try {
                    posnMask[mod] = getModuleData(mod).maqDiscIn.readDiscAll();
                    offline[mod] = false;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 read error: {1}", new Object[]{name, e});
                    offline[mod] = true;
                    setOnline(false);
                }
            }
        }
    }


    public boolean registerOut(int index, int chan)
    {
        if ((outMask[index] & (1 << chan)) != 0 || inMask[index] != 0) return false;
        outMask[index] |= (1 << chan);
        return true;
    }


    public boolean registerIn(int index, int chan)
    {
        if ((inMask[index] & (1 << chan)) != 0 || outMask[index] != 0) return false;
        inMask[index] |= (1 << chan);
        return true;
    }


    public void setValve(int index, int chan, boolean on)
    {
        if (on) {
            setMask[index] |= (1 << chan);
        }
        else {
            setMask[index] &= ~(1 << chan);
        }
    }


    public Boolean isValveSet(int index, int chan)
    {
        return offline[index] ? null : (getMask[index] & (1 << chan)) != 0;
    }


    public Boolean isPositionSet(int index, int chan)
    {
        return offline[index] ? null : (posnMask[index] & (1 << chan)) != 0;
    }

}