package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.python.google.common.collect.HashBiMap;

/**
 *  Implements a temperature controller for the refrigeration system.
 *
 *  @author Owen Saxton
 */
public class HeaterPIControl implements HasLifecycle {

    /**
     *  Inner class to hold channel information
     */
    static class ChannelData {

        Channel channel;
        double  weight;

    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, PowerDevice> powerDevices = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();

    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput;         // maximum PID output (watts)
    private Double  awGain;            // anti-windup gain
    private Double  basePower;         // base power input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 0.0;   // minimum PID output (watts)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)
    private String  powerDevc;         // The power device name
    private Integer powerChan;         // The power device channel
    private String[] tempChans;        // The temperature channel names to use

    @ConfigurationParameter(category="Refrig")
    private final Map<String, Double> tempWeights = new HashMap<>();  // The map of temperature channel weights

    private static final Logger LOG = Logger.getLogger(HeaterPIControl.class.getName());
    private PIController pic;          // The PI controller
    private PowerDevice powerDevcC;    // The power controller object
    private final Map<String, ChannelData> channelData = new HashMap<>();
    private double lastPower;          // The last power value set
    private boolean active;            // True if loop is running


    /**
     *  Starts the temperature control task.
     */
    @Override
    public void build() {
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask(name + "-timer", () -> iterateLoop()).withPeriod(Duration.ofMillis(updateTime));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void postInit()
    {
        if (gain == null) {
            ErrorUtils.reportConfigError(LOG, name, "gain", "is missing");
        }
        if (timeConst == null) {
            ErrorUtils.reportConfigError(LOG, name, "timeConst", "is missing");
        }
        if (smoothTime == null) {
            ErrorUtils.reportConfigError(LOG, name, "smoothTime", "is missing");
        }
        if (maxOutput == null) {
            ErrorUtils.reportConfigError(LOG, name, "maxOutput", "is missing");
        }
        if (awGain == null) {
            ErrorUtils.reportConfigError(LOG, name, "awGain", "is missing");
        }
        if (basePower == null) {
            ErrorUtils.reportConfigError(LOG, name, "basePower", "is missing");
        }
        if (tolerance == null) {
            ErrorUtils.reportConfigError(LOG, name, "tolerance", "is missing");
        }
        if (updateTime == null) {
            ErrorUtils.reportConfigError(LOG, name, "updateTime", "is missing");
        }
        if (powerDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "powerDevc", "is missing");
        }
        powerDevcC = powerDevices.get(powerDevc);
        if (powerDevcC == null) {
            ErrorUtils.reportConfigError(LOG, name, powerDevc, "doesn't exist");
        }
        if (powerChan == null) {
            ErrorUtils.reportConfigError(LOG, name, "powerChan", "is missing");
        }
        checkTempWeights(tempWeights);
        if (tempChans == null) {
            ErrorUtils.reportConfigError(LOG, name, "tempChans", "is missing");
        }
        for (String cName : tempChans) {
            Channel chan = allChannels.get(cName);
            if (chan == null) {
                ErrorUtils.reportConfigError(LOG, name, "tempChans", "element " + cName + " doesn't exist");
            }
            ChannelData cData = new ChannelData();
            channelData.put(cName, cData);
            cData.channel = chan;
            if (tempWeights.isEmpty()) {
                cData.weight = 1.0;
            }
            else {
                Double value = tempWeights.get(cName);
                cData.weight = value == null ? 0.0 : value;
            }
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the temperature weights.
     *
     *  @param  weights  A map of channel names to weight factors
     */
    @ConfigurationParameterChanger
    public void setTempWeights(Map<String, Double> weights)
    {
        if (!channelData.isEmpty()) {
            checkTempWeights(weights);
            for (ChannelData cData : channelData.values()){
                cData.weight = weights.isEmpty() ? 1.0 : 0.0;
            }
            for (Map.Entry entry : weights.entrySet()) {
                ChannelData cData = channelData.get((String)entry.getKey());
                if (cData != null) {
                    cData.weight = (double)entry.getValue();
                }
            }
        }
        tempWeights.clear();
        tempWeights.putAll(weights);
    }


    /**
     *  Sets a temperature weight.
     *
     *  @param  cName  The name of the temperature channel
     *  @param  weight  The weight factor to set
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a temperature weight")
    public void setTempWeight(@Argument(description="Channel name") String cName,
                              @Argument(description="Channel weight") double weight) throws RefrigException
    {
        Map<String, Double> weights = new HashMap<>();
        weights.putAll(tempWeights);
        weights.put(cName, weight);
        configService.change(name, "tempWeights", weights);
    }


    /**
     *  Gets the power device,
     * 
     *  @return  The power device
     */
    public PowerDevice getPowerDevice()
    {
        return powerDevcC;
    }


    /**
     *  Gets the power channel,
     * 
     *  @return  The power channel
     */
    public int getPowerChannel()
    {
        return powerChan;
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  value  The temperature to set
     */
    public void setTemp(double value)
    {
        pic.setSetpoint(value);
   }


    /**
     *  Resets the controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Restarts the control loop.
     */
    public void restart()
    {
        start(lastPower);
    }


    /**
     *  Starts the control loop.
     *
     *  @param  power  The starting power value
     */
    public synchronized void start(double power)
    {
        if (active) return;
        active = true;
        pic.reset();
        pic.setIntegral(power - basePower);
    }


    /**
     *  Stops the control loop.
     */
    public synchronized void stop()
    {
        if (!active) return;
        active = false;
    }


    /**
     *  Timer method for control loop iteration.
     */
    private synchronized void iterateLoop()
    {
        if (!active) return;
        double temp = 0.0, weight = 0.0;
        for (ChannelData cData : channelData.values()) {
            double value = cData.channel.getValue();
            if (!Double.isNaN(value)) {
                temp += cData.weight * value;
                weight += cData.weight;
            }
        }
        if (weight > 0.0) {
            double avgTemp = temp / weight;
            LOG.log(Level.FINE, "{0} control loop average temperature = {1}", new Object[]{name, avgTemp});
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(new double[]{avgTemp}, tod);
            powerDevcC.setPower(powerChan, lastPower);
        }
        else {
            LOG.log(Level.SEVERE, "{0} control loop iteration failed: no valid temperatures available", name);
        }
    }


    /**
     *  Checks a temperature weight map for validity.
     */
    private void checkTempWeights(Map<String, Double> weights)
    {
        for (Map.Entry entry : weights.entrySet()) {
            String cName = (String)entry.getKey();
            if (allChannels.get(cName) == null) {
                ErrorUtils.reportConfigError(LOG, name, "tempWeights", "element " + cName + " doesn't exist");
            }
            if ((double)entry.getValue() < 0.0) {
                ErrorUtils.reportConfigError(LOG, name, "tempWeights", "element " + cName + " has negative value");
            }
        }
    }

}
