package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.ConfigurationParameterDescription;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterOpState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Implements a temperature controller for the refrigeration system.
 *
 *  @author Owen Saxton
 */
public class TrimHeaterControl implements HasLifecycle {

    /**
     *  Inner class to hold channel information
     */
    static class ChannelData {

        Channel channel;
        double  weight;
        boolean overTemp;

    }

    public static final int
        NUM_COLD_RTDS = 8,
        NUM_CRYO_RTDS = 20;

    private static final String PIC = "Pic";

    @LookupPath
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, PowerDevice> powerDevices = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();

    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  gain = Double.NaN;              // loop gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  timeConst = Double.NaN;         // integration time constant (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  smoothTime = Double.NaN;        // input smoothing time (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  awGain = Double.NaN;            // anti-windup gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  basePower = Double.NaN;         // base power input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  tolerance = Double.NaN;         // maximum on-target error (%)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minInput = -200.0;              // minimum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxInput = 100.0;               // maximum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minOutput = -200.0;             // minimum PID output (watts)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxOutput = Double.NaN;         // maximum PID output (watts)

    @ConfigurationParameter(category="Refrig")
    private volatile double tempLimit = Double.NaN;          // The temperature limit
    @ConfigurationParameter(category="Refrig")
    private volatile double tempDeadband = Double.NaN;       // The temperature dead band
    @ConfigurationParameter(category="Refrig", maxLength=NUM_CRYO_RTDS)
    private volatile Map<String, Double> tempWeights = new ConcurrentHashMap<>();  // The map of temperature channel weights

    private String  powerDevc;         // The power device name
    private Integer powerChan;         // The power device channel
    private String[] tempChans;        // The temperature channel names to use

    private static final Logger LOG = Logger.getLogger(TrimHeaterControl.class.getName());
    private PIController pic;            // The PI controller
    private PowerDevice powerDevcC;      // The power device object
    private final Map<String, ChannelData> channelData = new LinkedHashMap<>();
    private volatile double lastPower;          // The last power value set
    private volatile double power;              // The manual power value
    private volatile TrimHeaterOpState opState;  // The operation state (off, power or temp)
    private volatile boolean overTemp = false;   // Whether temperature is above the limit
    private volatile boolean loopTempError = false;  // Whether all loop temperatures are in error
    private volatile boolean limitTempError = false;  // Whether all limit temperatures are in error


    /**
     *  Starts the control loop iteration and temperature limit check tasks.
     */
    @Override
    public void build() {
        ConfigurationParameterDescription desc = new ConfigurationParameterDescription().withCategory(PIC).withName("updateTime");
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask(name + "-iterate",
                                   () -> iterateLoop()).withPeriod(Duration.ofMillis(10000)).withPeriodParameterDescription(desc);
        periodicTaskService.scheduleAgentPeriodicTask(pt);
        pt = new AgentPeriodicTask(name + "-tempCheck", () -> checkTempLimit()).withPeriod(Duration.ofMillis(2000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void postInit()
    {
        if (Double.isNaN(gain)) {
            ErrorUtils.reportConfigError(LOG, name, "gain", "is missing");
        }
        if (Double.isNaN(timeConst)) {
            ErrorUtils.reportConfigError(LOG, name, "timeConst", "is missing");
        }
        if (Double.isNaN(smoothTime)) {
            ErrorUtils.reportConfigError(LOG, name, "smoothTime", "is missing");
        }
        if (Double.isNaN(awGain)) {
            ErrorUtils.reportConfigError(LOG, name, "awGain", "is missing");
        }
        if (Double.isNaN(basePower)) {
            ErrorUtils.reportConfigError(LOG, name, "basePower", "is missing");
        }
        if (Double.isNaN(tolerance)) {
            ErrorUtils.reportConfigError(LOG, name, "tolerance", "is missing");
        }
        if (Double.isNaN(maxOutput)) {
            ErrorUtils.reportConfigError(LOG, name, "maxOutput", "is missing");
        }
        if (powerDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "powerDevc", "is missing");
        }
        powerDevcC = powerDevices.get(powerDevc);
        if (powerDevcC == null) {
            ErrorUtils.reportConfigError(LOG, name, powerDevc, "doesn't exist");
        }
        if (powerChan == null) {
            ErrorUtils.reportConfigError(LOG, name, "powerChan", "is missing");
        }
        //checkTempWeights(tempWeights);
        if (tempChans == null) {
            ErrorUtils.reportConfigError(LOG, name, "tempChans", "is missing");
        }
        for (String cName : tempChans) {
            Channel chan = allChannels.get(cName);
            if (chan == null) {
                ErrorUtils.reportConfigError(LOG, name, "tempChans", "element " + cName + " doesn't exist");
            }
            ChannelData cData = new ChannelData();
            channelData.put(cName, cData);
            cData.channel = chan;
            if (tempWeights.isEmpty()) {
                cData.weight = 1.0;
            }
            else {
                Double value = tempWeights.get(cName);
                cData.weight = value == null ? 0.0 : value;
            }
        }
        if (Double.isNaN(tempLimit)) {
            ErrorUtils.reportConfigError(LOG, name, "tempLimit", "is missing");
        }
        if (Double.isNaN(tempDeadband)) {
            ErrorUtils.reportConfigError(LOG, name, "tempDeadband", "is missing");
        }
        if (tempDeadband <= 0.0) {
            ErrorUtils.reportConfigError(LOG, name, "tempDeadband", "is non-positive");
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the temperature weights.
     *
     *  @param  weights  A map of channel names to weight factors
     */
    @ConfigurationParameterChanger
    public void setTempWeights(Map<String, Double> weights)
    {
        if (!channelData.isEmpty()) {
            checkTempWeights(weights);
            for (ChannelData cData : channelData.values()){
                cData.weight = weights.isEmpty() ? 1.0 : 0.0;
            }
            for (Map.Entry entry : weights.entrySet()) {
                ChannelData cData = channelData.get((String)entry.getKey());
                if (cData != null) {
                    cData.weight = (double)entry.getValue();
                }
            }
        }
        tempWeights.clear();
        tempWeights.putAll(weights);
    }


    @Command(type=Command.CommandType.QUERY, description="Get the list of temperature channel names")
    public String getTempChannelNames()
    {
        return channelData.keySet().toString();
    }


    /**
     *  Sets a temperature weight.
     *
     *  @param  cName  The name of the temperature channel
     *  @param  weight  The weight factor to set
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a temperature weight")
    public void setTempWeight(@Argument(description="Channel name") String cName,
                              @Argument(description="Channel weight") double weight) throws RefrigException
    {
        Map<String, Double> weights = new HashMap<>();
        weights.putAll(tempWeights);
        weights.put(cName, weight);
        configService.change(name, "tempWeights", weights);
    }


    /**
     *  Gets the power device,
     * 
     *  @return  The power device
     */
    public PowerDevice getPowerDevice()
    {
        return powerDevcC;
    }


    /**
     *  Gets the power channel,
     * 
     *  @return  The power channel
     */
    public int getPowerChannel()
    {
        return powerChan;
    }


    /**
     *  Sets the constant power value.
     *
     *  @param  value  The power value to set
     */
    public synchronized void setPower(double value)
    {
        power = value;
        if (opState == TrimHeaterOpState.POWER) {
            powerDevcC.setPower(powerChan, power);
            lastPower = power;
        }
   }


    /**
     *  Sets the target temperature.
     *
     *  @param  value  The temperature to set
     */
    public void setTemp(double value)
    {
        pic.setSetpoint(value);
   }


    /**
     *  Sets the operation state.
     *
     *  @param  state  The state to set (OFF, POWER or TEMP)
     */
    public synchronized void setOpState(TrimHeaterOpState state)
    {
        if (state == opState) return;
        switch (opState = state) {

        case OFF:
            powerDevcC.enableOutput(powerChan, false);
            break;

        case POWER:
            powerDevcC.enableOutput(powerChan, !overTemp);
            powerDevcC.setPower(powerChan, power);
            lastPower = power;
            break;

        case TEMP:
            pic.reset();
            pic.setIntegral(lastPower - basePower);
            powerDevcC.enableOutput(powerChan, !overTemp);
        }
    }


    /**
     *  Gets whether the temperature is over the limit.
     *
     *  @return  Whether over the limit
     */
    public boolean isOverTemp()
    {
        return overTemp;
    }


    /**
     *  Timer method for PI control loop iteration.
     */
    private synchronized void iterateLoop()
    {
        if (opState != TrimHeaterOpState.TEMP || overTemp) return;
        double temp = 0.0, weight = 0.0;
        for (ChannelData cData : channelData.values()) {
            double value = cData.channel.getValue();
            if (!Double.isNaN(value)) {
                temp += cData.weight * value;
                weight += cData.weight;
            }
        }
        if (weight > 0.0) {
            if (loopTempError) {
                LOG.log(Level.INFO, "{0} control loop iteration succeeded again", name);
                loopTempError = false;
            }
            double avgTemp = temp / weight;
            LOG.log(Level.FINE, "{0} control loop average temperature = {1}", new Object[]{name, avgTemp});
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(new double[]{avgTemp}, tod);
            powerDevcC.setPower(powerChan, lastPower);
        }
        else {
            if (!loopTempError) {
                LOG.log(Level.SEVERE, "{0} control loop iteration failed: no valid temperatures available", name);
                loopTempError = true;
            }
        }
    }


    /**
     *  Timer method for checking temperatures against the limits.
     */
    private synchronized void checkTempLimit()
    {
        int nValidChan = 0;
        boolean newOverTemp = false;
        for (ChannelData cData : channelData.values()) {
            double temp = cData.channel.getValue();
            if (!Double.isNaN(temp)) {
                nValidChan++;
                if (temp < tempLimit - tempDeadband) {
                    cData.overTemp = false;
                }
                else if (temp >= tempLimit) {
                    cData.overTemp = true;
                }
                if (cData.overTemp) {
                    newOverTemp = true;
                }
            }
        }
        if (nValidChan == 0) {
            if (!limitTempError) {
                LOG.log(Level.SEVERE, "{0} temperature check failed: no valid temperatures available", name);
                limitTempError = true;
            }
        }
        else {
            if (limitTempError) {
                LOG.log(Level.INFO, "{0} temperature check succeeded again", name);
                limitTempError = false;
            }
            if (newOverTemp ^ overTemp) {
                overTemp = newOverTemp;
                powerDevcC.enableOutput(powerChan, overTemp ? false : opState != TrimHeaterOpState.OFF);
            }
        }
    }


    /**
     *  Checks a temperature weight map for validity.
     */
    private void checkTempWeights(Map<String, Double> weights)
    {
        for (Map.Entry entry : weights.entrySet()) {
            String cName = (String)entry.getKey();
            if (channelData.get(cName) == null) {
                ErrorUtils.reportConfigError(LOG, name, "tempWeights", "element " + cName + " doesn't exist");
            }
            if ((double)entry.getValue() < 0.0) {
                ErrorUtils.reportConfigError(LOG, name, "tempWeights", "element " + cName + " has negative value");
            }
        }
    }

}
