package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Logger;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Simulated Yaskawa A1000 series variable frequency motor drive.
 *
 *  @author Owen Saxton
 */
public class SimA1000Device extends A1000Device {

    /**
     *  Constants
     */
    private static final Logger LOG = Logger.getLogger(SimA1000Device.class.getName());
    private double frequency;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Sets the frequency.
     * 
     *  @param  freq  The value to set
     */
    @Override
    public void setFrequency(double freq)
    {
        frequency = freq;
    }


    /**
     *  Gets the frequency.
     * 
     *  @return  The set frequency, or NaN if device is offline
     */
    @Override
    public double getFrequency()
    {
        return frequency;
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (online) {
            switch (hwChan) {
            case CHAN_FREQUENCY:
                value = frequency;
                break;
            case CHAN_CURRENT:
                value = 1.0;
                break;
            case CHAN_VOLTAGE:
                value = 380.0;
                break;
            }
        }
        return value;
    }

}
