package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.ConfigurationParameterDescription;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20PWMControl;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.PIController;

/**
 *  Implements a fan speed loop controller for the utility trunk system.
 *
 *  @author Owen Saxton
 */
public class FanControl implements HasLifecycle {

    private static final String PIC = "Pic";

    @ConfigurationParameter(category=PIC)
    private volatile double  gain = Double.NaN;              // loop gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  timeConst = Double.NaN;         // integration time constant (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  smoothTime = Double.NaN;        // input smoothing time (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  awGain = Double.NaN;            // anti-windup gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  baseDuty = Double.NaN;          // base duty cycle
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  tolerance = Double.NaN;         // maximum on-target error (%)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minInput = -50.0;               // minimum input (temp. difference)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxInput = 50.0;                // maximum input (temp. difference)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minOutput = 0.0;                // minimum PID output (duty cycle)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxOutput = 1.0;                // maximum PID output (duty cycle)
    @ConfigurationParameter(category=PIC)
    private volatile double setTemp = Double.NaN;            // Temperature set point
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile boolean useAbsTemp = false;             // use absolute temperature set point, not difference

    private String  ambientTemp;       // Ambient temperature channel name
    private String  cabinetTemp;       // Cabinet temperature channel name
    private String[] fanCntrl;         // The fan controller names

    @LookupPath
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, Channel> allChannels = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, Maq20PWMControl> allFans = new HashMap<>();
    
    private volatile Channel ambChannel;    // Ambient temperature channel
    private volatile Channel cabChannel;    // Cabinet temperature channel
    private List<Maq20PWMControl> maqControl = new ArrayList<>(); // Maq20 fan controllers
    //private volatile double setTemp = 0.5;  // Temperature difference set point

    private static final Logger LOG = Logger.getLogger(FanControl.class.getName());
    private PIController pic;               // The PI controller
    private volatile double lastDuty;       // The last inverse duty cycle value set
    private volatile boolean active;        // True if loop is active
    private boolean tempError = false;      // True if had error reading temperatures


    /**
     *  Sets up the fan speed control loop timer task.
     */
    @Override
    public void build() {
        ConfigurationParameterDescription desc = new ConfigurationParameterDescription().withCategory(PIC).withName("updateTime");
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask(name + "-iterate",
                                   () -> iterateLoop()).withPeriod(Duration.ofMillis(10000)).withPeriodParameterDescription(desc);
        pts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void postInit()
    {
        if (Double.isNaN(gain)) {
            ErrorUtils.reportConfigError(LOG, name, "gain", "is missing");
        }
        if (Double.isNaN(timeConst)) {
            ErrorUtils.reportConfigError(LOG, name, "timeConst", "is missing");
        }
        if (Double.isNaN(smoothTime)) {
            ErrorUtils.reportConfigError(LOG, name, "smoothTime", "is missing");
        }
        if (Double.isNaN(awGain)) {
            ErrorUtils.reportConfigError(LOG, name, "awGain", "is missing");
        }
        if (Double.isNaN(baseDuty)) {
            ErrorUtils.reportConfigError(LOG, name, "baseDuty", "is missing");
        }
        if (Double.isNaN(tolerance)) {
            ErrorUtils.reportConfigError(LOG, name, "tolerance", "is missing");
        }
        ambChannel = ambientTemp == null ? null : allChannels.get(ambientTemp);
        if (ambChannel == null) {
            ErrorUtils.reportConfigError(LOG, name, "ambientTemp", "is missing or is not a Channel");
        }
        cabChannel = cabinetTemp == null ? null : allChannels.get(cabinetTemp);
        if (cabChannel == null) {
            ErrorUtils.reportConfigError(LOG, name, "cabinetTemp", "is missing or is not a Channel");
        }
        if (Double.isNaN(setTemp)) {
            ErrorUtils.reportConfigError(LOG, name, "setTemp", "is missing");
        }
        if (fanCntrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "fanCntrl", "is missing");
        }
        for (String cName : fanCntrl) {
            if (cName == null) continue;
            Maq20PWMControl ctrl = allFans.get(cName);
            if (ctrl == null) {
                ErrorUtils.reportConfigError(LOG, name, "fanCntrl", "element (" + cName + ") is not a Maq20PWMControl");
            }
            maqControl.add(ctrl);
        }
        pic = new PIController(-gain, timeConst);  // Gain needs to be negative since set point < temp diff
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(baseDuty);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
        pic.setSetpoint(setTemp);
    }


    /**
     *  Sets the target temperature difference.
     *
     *  @param  temp  The temperature to set
     */
    @ConfigurationParameterChanger
    public void setSetTemp(double temp)
    {
        setTemp = temp;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setSetpoint(setTemp);
        }
    }


    /**
     *  Gets the target temperature difference.
     *
     *  @return  The set temperature
     */
    public double getSetTemp()
    {
        return setTemp;
   }


    /**
     *  Sets the gain.
     *
     *  @param  gain  The gain to set
     */
    @ConfigurationParameterChanger
    public void setGain(double gain)
    {
        this.gain = gain;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setPID(-gain, timeConst);
        }
    }


    /**
     *  Gets the gain.
     *
     *  @return  The gain
     */
    public double getGain()
    {
        return gain;
   }


    /**
     *  Sets the time constant.
     *
     *  @param  time  The time constant to set
     */
    public void setTimeConstant(double time)
    {
        timeConst = time;
        pic.setPID(-gain, timeConst);
    }


    /**
     *  Gets the time constant.
     *
     *  @return  The time constant
     */
    public double getTimeConstant()
    {
        return timeConst;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  duty  The initial duty cycle value
     */
    public synchronized void startLoop(double duty)
    {
        if (active) return;
        lastDuty = duty;
        startLoop();
    }


    /**
     *  Starts the control loop.
     */
    public synchronized void startLoop()
    {
        if (active) return;
        pic.reset();
        pic.setIntegral(lastDuty - baseDuty);
        active = true;
    }


    /**
     *  Stops the control loop.
     */
    public void stopLoop()
    {
        active = false;
        setDutyCycle(0.0);  // Minimum fan speed
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the PI controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Sets the fan speed (PWM duty cycle).
     *
     *  @param  duty  The duty cycle to set (0 - 1)
     */
    private void setDutyCycle(double duty)
    {
        try {
            for (Maq20PWMControl ctrl : maqControl) {
                ctrl.setDutyCycle1(duty);
            }
        }
        catch (Exception ex) {
            LOG.log(Level.SEVERE, "Error setting {0} fan speed: {1}", new Object[]{name, ex});
        }
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private synchronized void iterateLoop()
    {
        if (!active) return;
        double tempDiff = cabChannel.getValue() - (useAbsTemp ? 0.0 : ambChannel.getValue());
        if (!Double.isNaN(tempDiff)) {
            if (tempError) {
                LOG.log(Level.INFO, "{0} control loop iteration succeeded again", name);
                tempError = false;
            }
            double tod = (double)System.currentTimeMillis() / 1000;
            lastDuty = pic.performPI(new double[]{tempDiff}, tod);
            setDutyCycle(lastDuty);
            //System.out.format("Temp: %s, Inv. duty factor: %s, Set point: %s, Error: %s, Integral: %s\n",
            //                  temp / count, lastDuty, pic.getSetpoint(), pic.getError(), pic.getIntegral());
        }
        else {
            if (!tempError) {
                LOG.log(Level.SEVERE, "{0} control loop iteration failed: temperature values not available", name);
                tempError = true;
            }
        }
    }

}
