package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.CompAlert;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Controls a refrigeration compressor.
 *
 *  @author Owen Saxton
 */
public class Compressor implements HasLifecycle {

    public static interface SwitchDevice {

        public void setSwitchOn(int chan, boolean on) throws RefrigException;

        public Boolean isSwitchOn(int chan);

    }

    class AlertEnableSwitch implements SwitchDevice {

        @Override
        public void setSwitchOn(int chan, boolean on)
        {
            alertsEnabled = powerEnabled ? true: on;
        }

        @Override
        public Boolean isSwitchOn(int chan)
        {
            return alertsEnabled;
        }

    }

    public static class LimitData {

        private Channel channel = null;    // Monitor channel with the value: null if not used
        private Channel channel2 = null;   // Monitor channel with the second value, if applicable
        private boolean isLower = false;   // Is lower limit, not upper
        private boolean noShutoff = false; // Doesn't shut off the compressor
        private double immedLimit = 0.0;   // The immediate limit, or NaN if not applicable
        private double delayLimit = 0.0;   // The delayed limit, or NaN if not applicable
        private int delayTime = 0;         // The delay time (ms), or -1 if 8-hour on
        private double value = 0.0;        // The value that exceeded the limit
        private long endTime = 0;          // The computed end time
        private SwCondState swState = SwCondState.CLEAR; // The software condition state
        private SwCondState provSwState = SwCondState.CLEAR; // The provisional software condition state
        private int provCount = 0;         // Number of successive times provSwState has same value

    }

    /**
     *  Constants.
     */
    private static final String COMP_LIMITS = "CompLimits";
    private static final int MIN_SW_STATE_SITS = 5;

    private static final Map<Integer, CompAlert> ccsAlertMap = new HashMap<>();
    static {
        ccsAlertMap.put(CompSwConds.SWC_DISC_PRESS, CompAlert.DISC_PRESS_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_DISC_TEMP, CompAlert.DISC_TEMP_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_CMPR_POWER, CompAlert.COMP_POWER_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_LIQUID_TEMP, CompAlert.LIQD_TEMP_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_OIL_LEVEL, CompAlert.OIL_LEVEL_LOW);
        ccsAlertMap.put(CompSwConds.SWC_SUCT_TEMP, CompAlert.SUCT_TEMP_LOW);
        ccsAlertMap.put(CompSwConds.SWC_PHASE_SEP_TEMP, CompAlert.PHASE_TEMP_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_PLATE_TEMP, CompAlert.CRYO_TEMP_LOW);
        ccsAlertMap.put(CompSwConds.SWC_PRESS_DIFF, CompAlert.PRESS_DIFF_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_DISC_TEMP_LOW, CompAlert.DISC_TEMP_LOW);
    }
    private static final Map<Integer, CompAlert> plcAlertMap = new HashMap<>();
    static {
        plcAlertMap.put(CompLatches.LATCH_DISCHARGE_PRESS, CompAlert.DISC_PRESS_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_DISCHARGE_TEMP, CompAlert.DISC_TEMP_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_POWER, CompAlert.COMP_POWER_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_LIQUID_TEMP, CompAlert.LIQD_TEMP_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_OIL_LEVEL, CompAlert.OIL_LEVEL_LOW_PLC);
        plcAlertMap.put(CompLatches.LATCH_SUCTION_TEMP, CompAlert.SUCT_TEMP_LOW_PLC);
        plcAlertMap.put(CompLatches.LATCH_AFTER_COOLER, CompAlert.AFTER_TEMP_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_SENSORS_VALID, CompAlert.SENS_READ_BAD_PLC);
        plcAlertMap.put(CompLatches.LATCH_SMOKE_DETC, CompAlert.SMOKE_DETC_PLC);
        plcAlertMap.put(CompLatches.LATCH_EXT_PERMIT, CompAlert.EXT_PERMIT_PLC);
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private CompPlutoDevice plutoDevc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private CompMaq20Device maq20Devc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private A1000Device vfdDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private CompMaq20PWMControl pwmControl;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private FanControl fanControl;

    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double discPressImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double discPressDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile int discPressDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double discTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double discTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile int discTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double suctTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double cmprPowerImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double cmprPowerDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile int cmprPowerDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double phaseSepTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile int phaseSepTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double liquidTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double liquidTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile int liquidTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double coldTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double pressDiffImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double discTempLowImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double orificeOffPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double orificeOnPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    volatile double heaterTempLimit;

    protected String discPressChan, discTempChan, suctPressChan, suctTempChan, cmprPowerChan, phaseSepTempChan, liquidTempChan;

    private static final Logger LOG = Logger.getLogger(Compressor.class.getName());
    private final CompState state;
    private final SwitchDevice[] switchDevices;
    private final int[] switchChannels;
    private final int type;
    private Channel discPress, discTemp, suctPress, suctTemp, cmprPower, phaseSepTemp, liquidTemp;
    private int activeConds = 0;
    private boolean gotCommand = false;  // Forces status update after command processed
    private final LatchState[] savedLatchStates = new LatchState[CompLatches.NUM_LATCHES];
    private final LimitData[] limitData = new LimitData[CompSwConds.NUM_SW_CONDITIONS];
    private final SwCondState[] savedSwStates = new SwCondState[CompSwConds.NUM_SW_CONDITIONS];
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private final AlertEnableSwitch alertEnableSwitch = new AlertEnableSwitch();
    private boolean powerEnabled = false, alertsEnabled = false;


    /**
     *  Constructor.
     *
     *  @param  state  The compressor state object
     */
    public Compressor(CompState state)
    {
        this.state = state;
        type = state.getType();
        switchDevices = new SwitchDevice[CompSwitches.NUM_SWITCHES];
        switchChannels = new int[CompSwitches.NUM_SWITCHES];
        for (int j = 0; j < CompSwConds.NUM_SW_CONDITIONS; j++) {
            limitData[j] = new LimitData();
        }
    }


    /**
     *  Initializes the compressor control.
     */
    @Override
    public void postInit()
    {
        if (plutoDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "plutoDevc", "not specified");
        }
        if (maq20Devc == null) {
            ErrorUtils.reportConfigError(LOG, name, "maq20Devc", "not specified");
        }
        if (type == CompTypes.TYPE_COLD && vfdDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "vfdDevc", "not specified");
        }
        if (discTempChan != null) {
            discTemp = channelMap.get(discTempChan);
        }
        if (discTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "discTempChan", "not specified or not defined");
        }
        if (suctTempChan != null) {
            suctTemp = channelMap.get(suctTempChan);
        }
        if (suctTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "suctTempChan", "not specified or not defined");
        }
        if (discPressChan != null) {
            discPress = channelMap.get(discPressChan);
        }
        if (discPress == null) {
            ErrorUtils.reportConfigError(LOG, name, "discPressChan", "not specified or not defined");
        }
        if (cmprPowerChan != null) {
            cmprPower = channelMap.get(cmprPowerChan);
        }
        if (cmprPower == null) {
            ErrorUtils.reportConfigError(LOG, name, "cmprPowerChan", "not specified or not defined");
        }
        limitData[CompSwConds.SWC_DISC_PRESS].channel = discPress;
        limitData[CompSwConds.SWC_DISC_TEMP].channel = discTemp;
        limitData[CompSwConds.SWC_SUCT_TEMP].channel = suctTemp;
        limitData[CompSwConds.SWC_SUCT_TEMP].isLower = true;
        limitData[CompSwConds.SWC_SUCT_TEMP].delayLimit = Double.NaN;
        limitData[CompSwConds.SWC_CMPR_POWER].channel = cmprPower;
        switchDevices[CompSwitches.SW_ENABLE] = plutoDevc;
        switchChannels[CompSwitches.SW_ENABLE] = CompPlutoDevice.SW_ENABLE;
        switchDevices[CompSwitches.SW_LIGHTS] = plutoDevc;
        switchChannels[CompSwitches.SW_LIGHTS] = CompPlutoDevice.SW_LIGHTS;
        switchDevices[CompSwitches.SW_ENABLE_ALERTS] = alertEnableSwitch;
        if (type == CompTypes.TYPE_COLD) {
            if (liquidTempChan != null) {
                liquidTemp = channelMap.get(liquidTempChan);
            }
            if (liquidTemp == null) {
                ErrorUtils.reportConfigError(LOG, name, "liquidTempChan", "not specified or not defined");
            }
            limitData[CompSwConds.SWC_LIQUID_TEMP].channel = liquidTemp;
        }
        else {
            limitData[CompSwConds.SWC_DISC_PRESS].delayTime = -1;
            if (phaseSepTempChan != null) {
                phaseSepTemp = channelMap.get(phaseSepTempChan);
            }
            if (phaseSepTemp == null) {
                ErrorUtils.reportConfigError(LOG, name, "phaseSepTempChan", "not specified or not defined");
            }
            if (suctPressChan != null) {
                suctPress = channelMap.get(suctPressChan);
            }
            if (suctPress == null) {
                ErrorUtils.reportConfigError(LOG, name, "suctPressChan", "not specified or not defined");
            }
            limitData[CompSwConds.SWC_PHASE_SEP_TEMP].channel = phaseSepTemp;
            limitData[CompSwConds.SWC_PHASE_SEP_TEMP].immedLimit = Double.NaN;
            limitData[CompSwConds.SWC_DISC_TEMP_LOW].channel = discTemp;
            limitData[CompSwConds.SWC_DISC_TEMP_LOW].isLower = true;
            limitData[CompSwConds.SWC_DISC_TEMP_LOW].noShutoff = true;
            limitData[CompSwConds.SWC_DISC_TEMP_LOW].delayLimit = Double.NaN;
            limitData[CompSwConds.SWC_PRESS_DIFF].channel = discPress;
            limitData[CompSwConds.SWC_PRESS_DIFF].channel2 = suctPress;
            limitData[CompSwConds.SWC_PRESS_DIFF].noShutoff = true;
            limitData[CompSwConds.SWC_PRESS_DIFF].delayLimit = Double.NaN;
            switchDevices[CompSwitches.SW_HEATER] = plutoDevc;
            switchChannels[CompSwitches.SW_HEATER] = CompPlutoDevice.SW_HEATER;
            switchDevices[CompSwitches.SW_ORIFICE_VALVE] = maq20Devc;
            switchChannels[CompSwitches.SW_ORIFICE_VALVE] = CompMaq20Device.SW_ORIFICE_VALVE;
            switchDevices[CompSwitches.SW_COOLANT_VALVE] = maq20Devc;
            switchChannels[CompSwitches.SW_COOLANT_VALVE] = CompMaq20Device.SW_COOLANT_VALVE;
            switchDevices[CompSwitches.SW_BYPASS_VALVE] = maq20Devc;
            switchChannels[CompSwitches.SW_BYPASS_VALVE] = CompMaq20Device.SW_BYPASS_VALVE;
            switchDevices[CompSwitches.SW_SURGE_HEATER] = maq20Devc;
            switchChannels[CompSwitches.SW_SURGE_HEATER] = CompMaq20Device.SW_SURGE_HEATER;
        }
        state.setName(name);
        plutoDevc.setType(type);
        maq20Devc.setType(type);
    }


    /**
     *  Starts the compressor control.
     */
    @Override
    public void postStart()
    {
        if (fanControl != null) {
            LOG.log(Level.INFO, "Starting fan speed controller for compressor {0}", name);
            fanControl.startLoop();
        }
    }


    /**
     *  Sets the discharge pressure immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressImmedLimit(double value)
    {
        discPressImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_PRESS].immedLimit = value;
    }


    /**
     *  Sets the discharge pressure delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayLimit(double value)
    {
        discPressDelayLimit = value;
        limitData[CompSwConds.SWC_DISC_PRESS].delayLimit = value;
    }


    /**
     *  Sets the discharge pressure delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayTime(int value)
    {
        discPressDelayTime = value;
        if (type == CompTypes.TYPE_COLD) {
            limitData[CompSwConds.SWC_DISC_PRESS].delayTime = 1000 * value;
        }
    }


    /**
     *  Sets the discharge temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempImmedLimit(double value)
    {
        discTempImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP].immedLimit = value;
    }


    /**
     *  Sets the discharge temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempDelayLimit(double value)
    {
        discTempDelayLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP].delayLimit = value;
    }


    /**
     *  Sets the discharge temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempDelayTime(int value)
    {
        discTempDelayTime = value;
        limitData[CompSwConds.SWC_DISC_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the suction temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setSuctTempImmedLimit(double value)
    {
        suctTempImmedLimit = value;
        limitData[CompSwConds.SWC_SUCT_TEMP].immedLimit = value;
    }


    /**
     *  Sets the compressor power immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerImmedLimit(double value)
    {
        cmprPowerImmedLimit = value;
        limitData[CompSwConds.SWC_CMPR_POWER].immedLimit = value;
    }


    /**
     *  Sets the compressor power delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerDelayLimit(double value)
    {
        cmprPowerDelayLimit = value;
        limitData[CompSwConds.SWC_CMPR_POWER].delayLimit = value;
    }


    /**
     *  Sets the compressor power delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerDelayTime(int value)
    {
        cmprPowerDelayTime = value;
        limitData[CompSwConds.SWC_CMPR_POWER].delayTime = 1000 * value;
    }


    /**
     *  Sets the liquid temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempImmedLimit(double value)
    {
        liquidTempImmedLimit = value;
        if (type == CompTypes.TYPE_COLD) {
            limitData[CompSwConds.SWC_LIQUID_TEMP].immedLimit = value;
        }
    }


    /**
     *  Sets the liquid temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayLimit(double value)
    {
        liquidTempDelayLimit = value;
        if (type == CompTypes.TYPE_COLD) {
            limitData[CompSwConds.SWC_LIQUID_TEMP].delayLimit = value;
        }
    }


    /**
     *  Sets the liquid temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayTime(int value)
    {
        liquidTempDelayTime = value;
        if (type == CompTypes.TYPE_COLD) {
            limitData[CompSwConds.SWC_LIQUID_TEMP].delayTime = 1000 * value;
        }
    }


    /**
     *  Sets the phase separator temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayLimit(double value)
    {
        phaseSepTempDelayLimit = value;
        if (type == CompTypes.TYPE_CRYO) {
            limitData[CompSwConds.SWC_PHASE_SEP_TEMP].delayLimit = value;
        }
    }


    /**
     *  Sets the phase separator temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayTime(int value)
    {
        phaseSepTempDelayTime = value;
        if (type == CompTypes.TYPE_CRYO) {
            limitData[CompSwConds.SWC_PHASE_SEP_TEMP].delayTime = 1000 * value;
        }
    }


    /**
     *  Sets the discharge temperature low immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempLowImmedLimit(double value)
    {
        discTempLowImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].immedLimit = value;
    }


    /**
     *  Sets the discharge pressure difference immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPressDiffImmedLimit(double value)
    {
        pressDiffImmedLimit = value;
        limitData[CompSwConds.SWC_PRESS_DIFF].immedLimit = value;
    }


    /**
     *  Sets the compressor index.
     *
     *  @param  index  The index value
     */
    public void setIndex(int index)
    {
        state.setIndex(index);
    }


    /**
     *  Gets the compressor index.
     *
     *  @return  The index value
     */
    public int getIndex()
    {
        return state.getIndex();
    }


    /**
     *  Command to get the valid switch names.
     *
     *  @return  The index value
     */
    @Command(type=CommandType.QUERY, description="Get the valid switch names", level=0)
    public List<String> getSwitchNames()
    {
        List<String> names = new ArrayList<>();
        for (int swId : state.getValidSwitches()) {
            names.add(CompSwitches.ID_MAP.get(swId));
        }
        return names;
    }


    /**
     *  Command to turn a switch on or off..
     *
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a compressor's switch state")
    public synchronized void setSwitchOn(@Argument(description="Switch number") int sw,
                                         @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        if (!state.getValidSwitches().contains(sw)) {
            throw new RefrigException("Invalid switch number: " + sw);
        }
        procSwitchCmnd(sw, on);
    }


    /**
     *  Command to turn a named switch on or off..
     *
     *  @param  swName  The switch name
     *  @param  on      Whether to turn on
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a compressor's switch state")
    public synchronized void setNamedSwitchOn(@Argument(description="Switch name") String swName,
                                              @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        Integer swId = CompSwitches.NAME_MAP.get(swName);
        if (swId == null || !state.getValidSwitches().contains(swId)) {
            throw new RefrigException("Invalid switch name: " + swName);
        }
        procSwitchCmnd(swId, on);
    }


    /**
     *  Processes a switch command.
     *
     *  @param  swId  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    private void procSwitchCmnd(int swId, boolean on) throws RefrigException
    {
        gotCommand = true;
        if (swId != CompSwitches.SW_ENABLE || !on || activeConds == 0) {
            setSwitch(swId, on);
            if (swId == CompSwitches.SW_ENABLE) {
                powerEnabled = on;
                alertEnableSwitch.setSwitchOn(0, on);
            }
        }
    }


    /**
     *  Command to reset the latches.
     */
    @Command(type=CommandType.ACTION, description="Reset a compressor's latches")
    public synchronized void resetLatches()
    {
        gotCommand = true;
        plutoDevc.resetLatches();
    }


    /**
     *  Command to set the cold bypass valve position.
     *
     *  @param  posn  The valve position  (0 - 1)
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a cold compressor's bypass valve")
    public synchronized void setBypassValve(@Argument(description="Valve position") double posn) throws RefrigException
    {
        gotCommand = true;
        if (type != CompTypes.TYPE_COLD) {
            throw new RefrigException("Compressor type must be COLD");
        }
        maq20Devc.setBypassValve(posn);
    }


    /**
     *  Command to set the cold coolant valve position.
     *
     *  @param  posn  The valve position  (0 - 1)
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a cold compressor's coolant valve")
    public synchronized void setCoolantValve(@Argument(description="Valve position") double posn) throws RefrigException
    {
        gotCommand = true;
        if (type != CompTypes.TYPE_COLD) {
            throw new RefrigException("Compressor type must be COLD");
        }
        maq20Devc.setCoolantValve(posn);
    }


    /**
     *  Command to set the cold VFD frequency.
     *
     *  @param  freq  The frequency
     *  @throws  RefrigException
     */
     @Command(type=CommandType.ACTION, description="Set a cold compressor's VFD frequency")
    public synchronized void setVfdFrequency(@Argument(description="Valve position") double freq) throws RefrigException
    {
        gotCommand = true;
        if (type != CompTypes.TYPE_COLD) {
            throw new RefrigException("Compressor type must be COLD");
        }
        vfdDevc.setFrequency(freq);
    }


    /**
     *  Updates the compressor system.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any state value changed
     */
    public synchronized boolean updateSystem()
    {
        checkLimits();
        return updateState();
    }


    /**
     *  Gets the compressor state.
     *
     *  @return  The compressor state
     */
    public CompState getState()
    {
        return state;
    }    
            

    /**
     *  Checks compressor shut-off limits.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     */
    private synchronized void checkLimits()
    {
        for (int cond : state.getValidSwConditions()) {
            LimitData ld = limitData[cond];
            if (ld.channel == null) continue;
            double value = ld.channel.getValue();
            if (ld.channel2 != null) {
                value = Math.abs(value - ld.channel2.getValue());
            }
            if (Double.isNaN(value)) continue;
            long endTime = ld.endTime;
            SwCondState swState = ld.swState;
            if (!ld.isLower && value > ld.immedLimit || ld.isLower && value < ld.immedLimit) {
                endTime = -1;
                swState = SwCondState.ACTIVE;
            }
            else if (!ld.isLower && value > ld.delayLimit || ld.isLower && value < ld.delayLimit) {
                if (endTime == 0) {
                    endTime = System.currentTimeMillis() + ld.delayTime;
                    swState = SwCondState.DLYPEND;
                }
                if (endTime >= 0) {
                    if (ld.delayTime < 0 && state.getConditionState(CompConditions.COND_CMP_ON_8HRS) == ConditionState.YES
                          || ld.delayTime >= 0 && System.currentTimeMillis() > endTime) {
                        endTime = -1;
                        swState = SwCondState.DLYACTV;
                    }
                }
            }
            else {
                endTime = 0;
                swState = SwCondState.CLEAR;
            }
            if (swState != ld.provSwState) {
                ld.provCount = 0;
                ld.provSwState = swState;
            }
            if (++ld.provCount >= MIN_SW_STATE_SITS && ld.provSwState != ld.swState) {
                ld.value = value;
                ld.swState = ld.provSwState;
                ld.endTime = endTime;
                if (endTime < 0) {
                    setCondition(cond);
                }
                else {
                    clearCondition(cond);
                }
            }
        }
    }


    /**
     *  Updates the compressor state.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any value changed
     */
    private synchronized boolean updateState()
    {
        boolean changed = gotCommand;
        gotCommand = false;
        Boolean active = plutoDevc.isPLCActive();
        CompressorState compState = active == null ? CompressorState.OFFLINE : !active ? CompressorState.PLC_DEAD : null;
        for (int sw : state.getValidSwitches()) {
            Boolean on = switchDevices[sw].isSwitchOn(switchChannels[sw]);
            SwitchState swState = on == null ? SwitchState.OFFLINE : on ? SwitchState.ON : SwitchState.OFF;
            if (swState != state.getSwitchState(sw)) {
                state.setSwitchState(sw, swState);
                changed = true;
            }
        }
        powerEnabled = true;
        for (int cond : state.getValidConditions()) {
            ConditionState condState = plutoDevc.getConditionState(cond);
            if (cond == CompConditions.COND_CMP_POWERED && condState != ConditionState.YES) {
                powerEnabled = false;
            }
            if (condState != state.getConditionState(cond)) {
                state.setConditionState(cond, condState);
                changed = true;
            }
        }
        alertEnableSwitch.setSwitchOn(0, alertsEnabled);
        for (int cond : state.getValidLatches()) {
            LatchState latchState = plutoDevc.getLatchState(cond);
            LatchState oldLatchState = state.getLatchState(cond);
            if (latchState != oldLatchState) {
                state.setLatchState(cond, latchState);
                changed = true;
            }
            LatchState savedLatchState = savedLatchStates[cond];
            if (alertsEnabled) {
                if (savedLatchState != null && latchState != savedLatchState
                      || savedLatchState == null && latchState != oldLatchState) {
                    raisePlcAlert(cond, latchState, savedLatchState == null ? oldLatchState : savedLatchState);
                    savedLatchStates[cond] = null;
                }
            }
            else {
                if (savedLatchState == null && latchState != oldLatchState) {
                    savedLatchStates[cond] = oldLatchState;
                }
            }
        }
        for (int cond : state.getValidSwConditions()) {
            LimitData ld = limitData[cond];
            SwCondState swState = ld.noShutoff && powerEnabled ? SwCondState.CLEAR : ld.swState;
            if (compState == null && (swState == SwCondState.ACTIVE || swState == SwCondState.DLYACTV)) {
                compState = CompressorState.SW_DSAB;
            }
            SwCondState oldSwState = state.getSwConditionState(cond);
            if (swState != oldSwState) {
                state.setSwConditionState(cond, swState);
                changed = true;
            }
            SwCondState savedSwState = savedSwStates[cond];
            if (alertsEnabled) {
                if (savedSwState != null && swState != savedSwState
                      || savedSwState == null && swState != oldSwState) {
                    raiseCcsAlert(cond, ld, swState);
                    savedSwStates[cond] = null;
                }
            }
            else {
                if (savedSwState == null && swState != oldSwState) {
                    savedSwStates[cond] = oldSwState;
                }
            }
        }
        if (compState == null) {
            compState = state.getConditionState(CompConditions.COND_CMP_WAITING) == ConditionState.YES ? CompressorState.WAITING :
                        state.getConditionState(CompConditions.COND_CMP_POWERED) == ConditionState.YES ? CompressorState.RUNNING :
                        state.getConditionState(CompConditions.COND_LATCHES_CLEAR) == ConditionState.NO ? CompressorState.HW_DSAB :
                        CompressorState.STOPPED;
        }
        CompressorState oldCompState = state.getCompressorState();
        boolean stateChanged = compState != oldCompState;
        if (stateChanged) {
            if (compState == CompressorState.PLC_DEAD) {
                raiseAlert(CompAlert.PLC_DEAD, AlertState.ALARM, name + " PLC has died: error code = " + plutoDevc.getErrorCode());
            }
            else if (oldCompState == CompressorState.PLC_DEAD) {
                lowerAlert(CompAlert.PLC_DEAD, name + " PLC is alive");
            }
            state.setCompressorState(compState);
            changed = true;
        }
        if (type == CompTypes.TYPE_COLD) {
            double posn = maq20Devc.getBypassValve();
            if (!areEqual(posn, state.getBypassValve())) {
                state.setBypassValve(posn);
                changed = true;
            }
            double freq = vfdDevc.getFrequency();
            if (!areEqual(freq, state.getVfdFrequency())) {
                state.setVfdFrequency(freq);
                changed = true;
            }
            posn = maq20Devc.getCoolantValve();
            if (!areEqual(posn, state.getCoolantValve())) {
                state.setCoolantValve(posn);
                changed = true;
            }
        }
        if (type == CompTypes.TYPE_CRYO) {
            controlCryoValves(compState == CompressorState.RUNNING, stateChanged);
            controlCryoFan(compState == CompressorState.RUNNING, stateChanged);
            controlCryoHeater();
        }
        
        return changed;
    }


    /**
     *  Raise PLC latch state alert.
     *
     *  @param  cond           The PLC latch condition
     *  @param  latchState     The new PLC latch state
     *  @param  oldLatchState  The current PLC latch state
     */
    private void raisePlcAlert(int cond, LatchState latchState, LatchState oldLatchState)
    {
        CompAlert alert = plcAlertMap.get(cond);
        if (latchState == LatchState.ACTIVE) {
            raiseAlert(alert, AlertState.ALARM, name + " PLC error condition set");
        }
        else if (latchState == LatchState.WARNING) {
            raiseAlert(alert, AlertState.WARNING, name + " PLC warning condition set");
        }
        else {
            if (latchState == LatchState.LATCHED && oldLatchState == LatchState.CLEAR) {
                raiseAlert(alert, AlertState.ALARM, name + " PLC error condition set");
            }
            lowerAlert(alert, name + " PLC error condition cleared");
        }
    }


    /**
     *  Raise CCS error state alert.
     *
     *  @param  cond     The CCS error condition
     *  @oaram  ld       The associated limit data
     *  @param  swState  The new CCS software error state
     */
    private void raiseCcsAlert(int cond, LimitData ld, SwCondState swState)
    {
        CompAlert alert = ccsAlertMap.get(cond);
        String value = String.format("%.1f", ld.value);
        if (swState == SwCondState.ACTIVE) {
            raiseAlert(alert, AlertState.ALARM, name + " CCS immediate error condition set: value (" + value
                                                  + (ld.isLower ? ") < " : ") > ") + ld.immedLimit);
        }
        else if (swState == SwCondState.DLYACTV) {
            String timeText = ld.delayTime >= 0 ? " for " + ld.delayTime / 1000 + " sec" : " and compressor on for 8 hours";
            raiseAlert(alert, AlertState.ALARM, name + " CCS delayed error condition set: value (" + value
                                                  + (ld.isLower ? ") < " : ") > ") + ld.delayLimit + timeText);
        }
        else if (swState == SwCondState.DLYPEND) {
            raiseAlert(alert, AlertState.WARNING, name + " CCS delayed error condition pending: value (" + value
                                                    + (ld.isLower ? ") < " : ") > ") + ld.delayLimit);
        }
        else {
            lowerAlert(alert, name + " CCS error condition cleared: value = " + value);
        }
    }


    /**
     *  Controls the cryo valves.
     *
     *  @param  compOn   Whether compressor is on
     *  @param  changed  Whether compressor state has changed
     *  This should be called at regular (short) intervals to maintain the correct state
     */
    private void controlCryoValves(boolean compOn, boolean changed)
    {
        try {
            if (compOn) {
                double press = limitData[CompSwConds.SWC_DISC_PRESS].channel.getValue();
                if (press > orificeOffPress) {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, false);
                }
                else if (press < orificeOnPress) {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, true);
                }
            }
            if (changed) {
                if (compOn) {
                    setSwitch(CompSwitches.SW_BYPASS_VALVE, false);
                }
                else {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, false);
                }
                setSwitch(CompSwitches.SW_COOLANT_VALVE, !compOn);
            }
        }
        catch (RefrigException e) {
            LOG.log(Level.SEVERE, "Error operating {0} compressor valves: {1}", new Object[]{name, e});
        }
    }


    /**
     *  Controls the cryo heater.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     */
    private void controlCryoHeater()
    {
        double temp = discTemp.getValue();
        if (temp > heaterTempLimit && state.getSwitchState(CompSwitches.SW_HEATER) == SwitchState.ON) {
            LOG.log(Level.WARNING, "Turning {0} compressor heater off - discharge temperature ({1}) > {2}",
                    new Object[]{name, temp, heaterTempLimit});
            try {
                setSwitch(CompSwitches.SW_HEATER, false);
            }
            catch (RefrigException e) {
                LOG.log(Level.SEVERE, "Error operating {0} compressor heater: {1}", new Object[]{name, e});
            }
        }
    }


    /**
     *  Controls the cryo fan.
     *
     *  @param  on  Whether compressor is on
     */
    private void controlCryoFan(boolean compOn, boolean changed)
    {
        if (changed && pwmControl != null) {
            if (compOn) {
                pwmControl.enable();
            }
            else {
                pwmControl.disable();
            }
        }
    }


    /**
     *  Turns a switch on or off..
     *
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    private void setSwitch(int sw, boolean on) throws RefrigException
    {
        switchDevices[sw].setSwitchOn(switchChannels[sw], on);
    }


    /**
     *  Sets a condition that disables the compressor
     *
     *  @param  cond  The condition number
     */
    private void setCondition(int cond)
    {
        LOG.log(Level.FINE, "{0} compressor encountered \"{1}\" condition", new Object[]{name, CompSwConds.DESCS[cond]});
        if (!limitData[cond].noShutoff) {
            try {
                setSwitchOn(CompSwitches.SW_ENABLE, false);
            }
            catch (RefrigException e) {
                LOG.log(Level.SEVERE, "Error shutting off {0} compressor: {1}", new Object[]{name, e});
            }
        }
        activeConds |= 1 << cond;
    }


    /**
     *  Clears a condition that disabled the compressor
     *
     *  @param  cond  The condition number
     */
    private void clearCondition(int cond)
    {
        activeConds &= ~(1 << cond);
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The refrigeration alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     */
    private void raiseAlert(CompAlert alert, AlertState state, String cond)
    {
        activeAlertMap.put(alert.getId(name), true);
        alertService.raiseAlert(alert.newAlert(name), state, cond);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The refrigeration alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(CompAlert alert, String cond)
    {
        if (activeAlertMap.put(alert.getId(name), false) == Boolean.TRUE) {
            alertService.raiseAlert(alert.newAlert(name), AlertState.NOMINAL, cond);
        }
    }


    /**
     *  Enables an alert to be cleared.
     * 
     *  @param  alert  The alert
     *  @return  Action code
     */
    public ClearAlertCode canClearAlert(Alert alert)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Compares two numbers for equality, accounting for possible NaNs.
     *
     *  @param  value1  First value
     *  @param  value2  Second value
     *  @return  Whether the values are equal
     */
    private boolean areEqual(double value1, double value2)
    {
        return value1 == value2 || Double.isNaN(value1) && Double.isNaN(value2);
    }

}
