package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.yaskawa.A1000;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Handles a Yaskawa A1000 series variable frequency motor drive.
 *
 *  @author Owen Saxton
 */
public class A1000Device extends Device {

    /**
     *  Constants
     */
    public static final int
        CHAN_FREQUENCY = 0,
        CHAN_CURRENT = 1,
        CHAN_VOLTAGE = 2,
        NUM_CHANS = 3;
    private static final double
        INPUT_VOLTAGE = 380.0;

    /**
     *  Data fields.
     */
    @ConfigurationParameter(category = "Device", isFinal = true)
    private String  devcName;      // Device name of the serial line
    @ConfigurationParameter(category = "Device", isFinal = true)
    private int     baudRate = 0;  // The serial line baud rate (0 = driver default)
    @ConfigurationParameter(category = "Device", isFinal = true)
    private int     address = 0;   // RS-485 address of VFD controller (0 = driver default)

    private static final Logger LOG = Logger.getLogger(A1000Device.class.getName());
    private final A1000 a1000 = new A1000();  // Associated A1000 object
    private boolean initError = false;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        super.configure(mon);
        if (devcName == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "devcName", "is missing");
        }
        fullName = "A1000 VFD (" + devcName + ":" + address + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            a1000.open(devcName, address, baudRate);
            try {
                initParams();
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error initializing {0} parameters: {1}", new Object[]{fullName, e});
                LOG.log(Level.SEVERE, "Make sure that VFD is in remote mode.  Use initParams command to re-initialize");
            }
            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {  // Avoid reporting consecutive errors
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                a1000.close();
            }
            catch (DriverException ce) {
                // Will happen if open was unsuccessful
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            a1000.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Sets the frequency.
     * 
     *  @param  freq  The value to set
     *  @throws RefrigException 
     */
    public void setFrequency(double freq) throws RefrigException
    {
        try {
            a1000.setFrequency(freq);
        }
        catch (DriverException e) {
            setOnline(false);
            throw new RefrigException("Error setting frequency for " + getPath() + ": " + e);
        }
    }


    /**
     *  Gets the frequency.
     * 
     *  @return  The set frequency, or NaN if device is offline
     */
    public double getFrequency()
    {
        double value = Double.NaN;
        if (isOnline()) {
            try {
                value = a1000.getFrequencyApp();
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error getting frequency for {0}: {1}", new Object[]{getPath(), e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        String quantity = null;
        if (online) {
            try {
                switch (hwChan) {
                case CHAN_FREQUENCY:
                    quantity = "frequency";
                    value = a1000.readFrequency();
                    break;
                case CHAN_CURRENT:
                    quantity = "current";
                    value = a1000.readCurrent();
                    break;
                case CHAN_VOLTAGE:
                    quantity = "voltage";
                    value = a1000.readVoltage();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading VFD {0}: {1}", new Object[]{quantity, e});
                setOnline(false);
            }
        }

        return value;
    }


    @Command(description="Initialize parameters")
    public void initParams() throws DriverException
    {
        a1000.setAccessLevel(A1000.AccessLevel.FULL);
        a1000.setRunCmndFwdRev(true);
        a1000.setRunProg(A1000.RunProg.RUN_ONLY);
        a1000.setStopMethod(A1000.StopMethod.COAST);
        a1000.setControlMode(A1000.ControlMode.VF);
        a1000.setErrorAction(A1000.ErrorAction.ALARM_ONLY);
        a1000.setFaultDetected(false);
        a1000.setInputVoltage(INPUT_VOLTAGE);
        a1000.setVfSelection(A1000.VfSelection.HZ50);
        a1000.setVoltageUnits(A1000.VoltageUnits.TENTHS);
        a1000.setEnterRequired(false);
        a1000.setReversePermitted(false);
        a1000.setRunCmndSrc(A1000.RunCmndSrc.MODBUS);
        a1000.setFreqRefSrc(A1000.FreqRefSrc.MODBUS);
        a1000.enterParameters();
        a1000.runForward();  // Always switch VFD on
        a1000.runForward();  // Needs to be done twice if switching from local mode
    }

    /* Commands to aid in debugging */
    
    private final int MAX_READ = 16;

    @Command(description="Read a register")
    public String readRegister(@Argument(description="Register to read") int number) throws DriverException
    {
        return String.format("%04x: %04x", number, a1000.readRegister((short)number));
    }


    @Command(description="Read a set of registers")
    public String readRegisters(@Argument(description="First register to read") int number,
                                @Argument(description="Number of registers to read") int count)
    {
        StringBuilder text = new StringBuilder();
        int lineSize = 8;
        int nRead = 0;
        try {
            while (nRead < count) {
                short[] reply = a1000.readRegisters((short)number, (short)Math.min(count - nRead, MAX_READ));
                for (int j = 0; j < reply.length; j++, nRead++, number++) {
                    if ((nRead % lineSize) == 0) {
                        if (nRead != 0) {
                            text.append("\n");
                        }
                        text.append(String.format("%04x:", number));
                    }
                    text.append(String.format(" %04x", reply[j]));
                }
            }
        }
        catch (DriverException e) {
            if (text.length() > 0) {
                text.append("\n");
            }
            text.append("***** Got read error: ").append(e.getMessage());
        }
        return text.toString();
    }


    @Command(description = "Show node address")
    public int showNodeAddress() throws DriverException
    {
        return a1000.getNodeAddress();
    }


    @Command(description = "Show baud rate")
    public String showBaudRate() throws DriverException
    {
        return a1000.getBaudRate().toString().substring(1);
    }


    @Command(description = "Show parity")
    public A1000.Parity showParity() throws DriverException
    {
        return a1000.getParity();
    }


    @Command(description = "Show access level")
    public A1000.AccessLevel showAccessLevel() throws DriverException
    {
        return a1000.getAccessLevel();
    }


    @Command(description = "Show control mode")
    public A1000.ControlMode showControlMode() throws DriverException
    {
        return a1000.getControlMode();
    }


    @Command(description = "Show whether enter required")
    public boolean isEnterRequired() throws DriverException
    {
        return a1000.isEnterRequired();
    }


    @Command(description = "Show communications error action")
    public A1000.ErrorAction showErrorAction() throws DriverException
    {
        return a1000.getErrorAction();
    }


    @Command(description = "Show whether fault detected")
    public boolean isFaultDetected() throws DriverException
    {
        return a1000.isFaultDetected();
    }


    @Command(description = "Show frequency reference source")
    public A1000.FreqRefSrc showFreqRefSrc() throws DriverException
    {
        return a1000.getFreqRefSrc();
    }


    @Command(description = "Show input voltage")
    public double showInputVoltage() throws DriverException
    {
        return a1000.getInputVoltage();
    }


    @Command(description = "Show whether reverse permitted")
    public boolean isReversePermitted() throws DriverException
    {
        return a1000.isReversePermitted();
    }


    @Command(description = "Show whether run command style is forward/reverse")
    public boolean isRunCmndFwdRev() throws DriverException
    {
        return a1000.isRunCmndFwdRev();
    }


    @Command(description = "Show run command source")
    public A1000.RunCmndSrc showRunCmndSrc() throws DriverException
    {
        return a1000.getRunCmndSrc();
    }


    @Command(description = "Show run/program interaction")
    public A1000.RunProg showRunProg() throws DriverException
    {
        return a1000.getRunProg();
    }


    @Command(description = "Show the stop method")
    public A1000.StopMethod showStopMethod() throws DriverException
    {
        return a1000.getStopMethod();
    }


    @Command(description = "Show V/f profile selection")
    public A1000.VfSelection showVfSelection() throws DriverException
    {
        return a1000.getVfSelection();
    }


    @Command(description = "Show voltge units")
    public A1000.VoltageUnits showVoltageUnits() throws DriverException
    {
        return a1000.getVoltageUnits();
    }


    @Command(description = "Show the reference frequency")
    public double showFrequency() throws DriverException
    {
        return a1000.getFrequency();
    }


    @Command(description = "Show the applied reference frequency")
    public double showFrequencyApp() throws DriverException
    {
        return a1000.getFrequencyApp();
    }


    @Command(description = "Read the frequency")
    public double readFrequency() throws DriverException
    {
        return a1000.readFrequency();
    }


    @Command(description = "Read the current")
    public double readCurrent() throws DriverException
    {
        return a1000.readCurrent();
    }


    @Command(description = "Read the voltage")
    public double readVoltage() throws DriverException
    {
        return a1000.readVoltage();
    }


    @Command(description = "Read the bus voltage")
    public double readBusVoltage() throws DriverException
    {
        return a1000.readBusVoltage();
    }


    @Command(level=50, description="Write a set of registers")
    public void writeRegisters(@Argument(description="First register to write") int number,
                               @Argument(description="Values to write") int... values) throws DriverException
    {
        short count = (short)values.length;
        if (count == 0) {
            throw new DriverException("No values supplied");
        }
        else {
            short[] sValues = new short[count];
            for (int j = 0; j < count; j++) {
                sValues[j] = (short)values[j];
            }
            a1000.writeRegisters((short)number, sValues);
        }
    }


    @Command(level=50, description = "Enter parameters")
    public void enterParameters() throws DriverException
    {
        a1000.enterParameters();
    }


    @Command(level=50, description = "Save parameters")
    public void saveParameters() throws DriverException
    {
        a1000.saveParameters();
    }


    @Command(level=50, description = "Set node address")
    public void setNodeAddress(@Argument(description = "Node address") int address) throws DriverException
    {
        a1000.setNodeAddress(address);
    }


    @Command(level=50, description = "Set baud rate")
    public void setBaudRate(@Argument(description = "Baud rate enum") A1000.BaudRate baudRate) throws DriverException
    {
        a1000.setBaudRate(baudRate);
    }


    @Command(level=50, description = "Set parity")
    public void setParity(@Argument(description = "Parity enum") A1000.Parity parity) throws DriverException
    {
        a1000.setParity(parity);
    }


    @Command(level=50, description = "Set access level")
    public void setAccessLevel(@Argument(description = "Access level enum") A1000.AccessLevel level) throws DriverException
    {
        a1000.setAccessLevel(level);
    }


    @Command(level=50, description = "Set control mode")
    public void setControlMode(@Argument(description = "Control mode enum") A1000.ControlMode mode) throws DriverException
    {
        a1000.setControlMode(mode);
    }


    @Command(level=50, description = "Set whether enter required")
    public void setEnterRequired(@Argument(description = "Whether enter required") boolean required) throws DriverException
    {
        a1000.setEnterRequired(required);
    }


    @Command(level=50, description = "Set communications error action")
    public void setErrorAction(@Argument(description = "Error action enum") A1000.ErrorAction action) throws DriverException
    {
        a1000.setErrorAction(action);
    }


    @Command(level=50, description = "Set whether fault detected")
    public void setFaultDetected(@Argument(description = "Whether fault detected") boolean detect) throws DriverException
    {
        a1000.setFaultDetected(detect);
    }


    @Command(level=50, description = "Set frequency reference source")
    public void setFreqRefSrc(@Argument(description = "Freq refc source enum") A1000.FreqRefSrc source) throws DriverException
    {
        a1000.setFreqRefSrc(source);
    }


    @Command(level=50, description = "Set the input voltage")
    public void setInputVoltage(@Argument(description = "The voltage") double voltage) throws DriverException
    {
        a1000.setInputVoltage(voltage);
    }


    @Command(level=50, description = "Set whether reverse permitted")
    public void setReversePermitted(@Argument(description = "Whether reverse permitted") boolean permit) throws DriverException
    {
        a1000.setReversePermitted(permit);
    }


    @Command(level=50, description = "Set whether run command style is forward/reverse")
    public void setRunCmndFwdRev(@Argument(description = "Whether forward/reverse") boolean fwdRev) throws DriverException
    {
        a1000.setRunCmndFwdRev(fwdRev);
    }


    @Command(level=50, description = "Set run command source")
    public void setRunCmndSrc(@Argument(description = "Run cmnd source enum") A1000.RunCmndSrc source) throws DriverException
    {
        a1000.setRunCmndSrc(source);
    }


    @Command(level=50, description = "Set run/program interaction")
    public void setRunProg(@Argument(description = "Run/program enum") A1000.RunProg runProg) throws DriverException
    {
        a1000.setRunProg(runProg);
    }


    @Command(level=50, description = "Set the stop method")
    public void setStopMethod(@Argument(description = "Stop method enum") A1000.StopMethod method) throws DriverException
    {
        a1000.setStopMethod(method);
    }


    @Command(level=50, description = "Set V/f profile selection")
    public void setVfSelection(@Argument(description = "V/f selection enum") A1000.VfSelection select) throws DriverException
    {
        a1000.setVfSelection(select);
    }



    //@Command(description = "Set the reference frequency")
    //public void setFrequency(@Argument(description = "The frequency") double freq) throws DriverException
    //{
    //    a1000.setFrequency(freq);
    //}


    @Command(level=50, description = "Set voltage units")
    public void setVoltageUnits(@Argument(description = "Voltage units enum") A1000.VoltageUnits units) throws DriverException
    {
        a1000.setVoltageUnits(units);
    }


    @Command(level=50, description = "Run the motor forwards")
    public void runForward() throws DriverException
    {
        a1000.runForward();
    }


    @Command(level=50, description = "Run the motor in reverse")
    public void runReverse() throws DriverException
    {
        a1000.runReverse();
    }


    @Command(level=50, description = "Stop the motor")
    public void stop() throws DriverException
    {
        a1000.stop();
    }

}
