package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.HexState;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;
import org.lsst.ccs.subsystem.refrig.data.UpdatePeriod;

/**
 *  Implements the refrigeration heat exchanger monitoring subsystem.
 *
 *  @author Owen Saxton
 */
public class HexMain extends Subsystem implements HasLifecycle, AgentPresenceListener, StatusMessageListener {

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    private static final Logger LOG = Logger.getLogger(HexMain.class.getName());
    private final HexState state = new HexState();
    private String refrigGroup;


    /**
     *  Constructor.
     */
    public HexMain() {
        super("hex", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Check that "use full paths" has been set correctly
        String useFull = propertiesService.getAgentProperty("org.lsst.ccs.use.full.paths");
        if (useFull == null || !useFull.equals("true")) {
            ErrorUtils.reportConfigError(LOG, name, "org.lsst.ccs.use.full.paths", "not defined as true");
        }
    }


    /**
     *  Initializes the subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a heat exchanger subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.HEX_TYPE, HexMain.class.getCanonicalName());
        // Add an agent present listener
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        refrigGroup = RefrigUtils.getGroupName(subsys.getAgentInfo());
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Heat exchanger subsystem started");
    }


    /**
     *  Listens for the arrival of the companion refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                if (RefrigUtils.getGroupName(agent).equals(refrigGroup)) {
                    getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                                                   && msg instanceof StatusSubsystemData);
                    break;
                }
            }
        }
    }


    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE) && RefrigUtils.getGroupName(agent).equals(refrigGroup)) {
                getMessagingAccess().removeStatusMessageListener(this);
            }
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  Sets the tick period from the refrig system value.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData sd = (StatusSubsystemData)msg;
        if (sd.getDataKey().equals(UpdatePeriod.KEY)) {
            setTickPeriod(((UpdatePeriod)((KeyValueData)sd.getSubsystemData()).getValue()).getTickMillis());
        }
    }


    /**
     *  Gets the state of the heat exchanger module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the heat exchanger state", level=0)
    public HexState getSystemState()
    {
        state.setTickMillis(getTickPeriod());
        return state;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        publishState();
    }


    /**
     *  Publishes the state of the heat exchanger module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        state.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(HexState.KEY, state));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
