package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.auxelex.HeaterPS;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Handles a SLAC heater power supply.
 *
 *  @author Owen Saxton
 */
public class HeaterPsDevice extends PowerDevice {

    /**
     *  Constants.
     */
    public static final int
        CHANSET_COLD = 0,
        CHANSET_CRYO = 1,

        TYPE_VOLTAGE     = 0,
        TYPE_CURRENT     = 1,
        TYPE_POWER       = 2,
        TYPE_TOTAL_POWER = 3,
        TYPE_TEMP        = 4,
        TYPE_MAIN_VOLTS  = 5,
        TYPE_MAIN_CURR   = 6,
        TYPE_MAIN_POWER  = 7,
        TYPE_MAIN_TEMP   = 8,
        TYPE_MAIN_STATUS = 9;

    private static final int
        N_HW_CHANS = HeaterPS.NUM_HEATERS,
        N_SET_CHANS = N_HW_CHANS / 2,
        CRYO_CHANNEL_FIRST = 0,
        COLD_CHANNEL_FIRST = N_SET_CHANS,
        COLD_CHANNEL_MINUS_Y = COLD_CHANNEL_FIRST,
        COLD_CHANNEL_PLUS_Y  = COLD_CHANNEL_FIRST + N_SET_CHANS - 1;
    private static final String
        NODE   = "node";

    /**
     *  Private lookup maps.
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("VOLTAGE", TYPE_VOLTAGE);
        typeMap.put("CURRENT", TYPE_CURRENT);
        typeMap.put("POWER", TYPE_POWER);
        typeMap.put("TOTALPOWER", TYPE_TOTAL_POWER);
        typeMap.put("TEMP", TYPE_TEMP);
        typeMap.put("MAINVOLTS", TYPE_MAIN_VOLTS);
        typeMap.put("MAINCURR", TYPE_MAIN_CURR);
        typeMap.put("MAINPOWER", TYPE_MAIN_POWER);
        typeMap.put("MAINTEMP", TYPE_MAIN_TEMP);
        typeMap.put("MAINSTAT", TYPE_MAIN_STATUS);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category="Refrig", isFinal=true)
    private Integer node;      // Heater PS node number
    private int[] cryoChannels;
    private int[] coldChannels;

    private static final Logger LOG = Logger.getLogger(HeaterPsDevice.class.getName());
    private final HeaterPS htr = new HeaterPS();  // Associated heater PS object
    private final Set<Integer> coldChannelSet = new HashSet<>();
    private final Set<Integer> dsabColdChannelSet = new HashSet<>();
    private final Set<Integer> cryoChannelSet = new HashSet<>();
    private final Set<Integer>[] channelSets = new Set[2];
    private final double[] currents = new double[N_HW_CHANS];  // Read currents 
    private final double[] voltages = new double[N_HW_CHANS];  // Read voltages
    private final boolean[] outputs = new boolean[N_HW_CHANS]; // Read output states


    /**
     *  Constructor.
     */
    public HeaterPsDevice()
    {
        super("Heater PS", null, 0, 0, 0);
        connType = DriverConstants.ConnType.NET;
        devcId = "";
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "node", "is missing");
        }
        if (coldChannels != null) {
            for (int chan : coldChannels) {
                if (chan < 0 || chan >= N_HW_CHANS) {
                    ErrorUtils.reportConfigError(LOG, getName(), "coldChannel " + chan, " is invalid");
                }
                if (chan < COLD_CHANNEL_FIRST) {
                    ErrorUtils.reportConfigError(LOG, getName(), "coldChannel " + chan, " is a cryo channel");
                }
                if (!coldChannelSet.add(chan)) {
                    ErrorUtils.reportConfigError(LOG, getName(), "coldChannel " + chan, " is a duplicate");
                }
            }
        }
        else {
            for (int chan = COLD_CHANNEL_FIRST; chan < COLD_CHANNEL_FIRST + N_SET_CHANS; chan++) {
                coldChannelSet.add(chan);
            }
        }
        if (cryoChannels != null) {
            for (int chan : cryoChannels) {
                if (chan < 0 || chan >= N_HW_CHANS) {
                    ErrorUtils.reportConfigError(LOG, getName(), "cryoChannel " + chan, " is invalid");
                }
                if (chan >= COLD_CHANNEL_FIRST) {
                    ErrorUtils.reportConfigError(LOG, getName(), "cryoChannel " + chan, " is a cold channel");
                }
                if (!cryoChannelSet.add(chan)) {
                    ErrorUtils.reportConfigError(LOG, getName(), "cryoChannel " + chan, " is a duplicate");
                }
            }
        }
        else {
            for (int chan = CRYO_CHANNEL_FIRST; chan < CRYO_CHANNEL_FIRST + N_SET_CHANS; chan++) {
                cryoChannelSet.add(chan);
            }
        }
        channelSets[CHANSET_COLD] = coldChannelSet;
        channelSets[CHANSET_CRYO] = cryoChannelSet;
        fullName = "Heater power supply (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            htr.open(node);
            htr.setMainPowerOn(true);
            htr.setSwitchPeriod(500);
            setOnline(true);
            LOG.log(Level.INFO, "Connected to {0}", fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
            }
            if (htr != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            htr.close();
        }
        catch (DriverException e) {
        }
        for (int j = 0; j < N_HW_CHANS; j++) {
            currents[j] = 0;
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string (not used)
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        int numChan = (iType == TYPE_VOLTAGE || iType == TYPE_CURRENT || iType == TYPE_POWER) ? N_HW_CHANS :
                      (iType == TYPE_TOTAL_POWER) ? ThermalState.NUM_TRIM_HEATERS : 1;
        if (hwChan < 0 || hwChan >= numChan) {
            ErrorUtils.reportChannelError(LOG, name, "HW channel", hwChan);
        }
        return new int[]{iType, 0};
    }


    /**
     *  Reads all heater channels as a group.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            for (int chan = 0; chan < N_HW_CHANS; chan++) {
                currents[chan] = htr.readCurrent(chan);
                voltages[chan] = htr.getVoltage(chan);
                outputs[chan] = htr.getOutput(chan);
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading currents & voltages from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (online) {
            try {
                switch (type) {
                case TYPE_VOLTAGE:
                    value = outputs[hwChan] ? voltages[hwChan] : 0.0;
                    break;
                case TYPE_CURRENT:
                    value = currents[hwChan];
                    break;
                case TYPE_POWER:
                    value = outputs[hwChan] ? voltages[hwChan] * currents[hwChan] : 0.0;
                    break;
                case TYPE_TOTAL_POWER:
                    value = 0.0;
                    int firstChan = (hwChan == CHANSET_COLD) ? COLD_CHANNEL_FIRST : CRYO_CHANNEL_FIRST;
                    for (int chan = firstChan; chan < firstChan + N_SET_CHANS; chan++) {
                        value += outputs[chan] ? voltages[chan] * currents[chan] : 0.0;
                    }
                    break;
                case TYPE_TEMP:
                    value = htr.readBoardTemperature();
                    break;
                case TYPE_MAIN_VOLTS:
                    value = htr.readMainVoltage();
                    break;
                case TYPE_MAIN_CURR:
                    value = htr.readMainCurrent();
                    break;
                case TYPE_MAIN_POWER:
                    value = htr.readMainVoltage() * htr.readMainCurrent();
                    break;
                case TYPE_MAIN_TEMP:
                    value = htr.readMainTemperature();
                    break;
                case TYPE_MAIN_STATUS:
                    value = htr.readMainStatus();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading value from {0}: {1}", new Object[]{fullName, e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Sets the enabled state of the outer cold channels.
     *
     *  @param  section  The cold channel section (y-plus or y-minus)
     *  @param  enable   Whether or not to enable
     */
    public void enableColdSection(int section, boolean enable)
    {
        int chan = section == ThermalState.COLD_SECTION_MINUS_Y ? COLD_CHANNEL_MINUS_Y : COLD_CHANNEL_PLUS_Y;
        if (coldChannelSet.contains(chan)) {
            if (enable) {
                dsabColdChannelSet.remove(chan);
            }
            else {
                dsabColdChannelSet.add(chan);
            }
        }
    }


    /**
     *  Sets the output state of a set of channels.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @param  value    The output state to set, true or false
     *  @throws  DriverException
     */
    @Override
    public void setOutput(int chanSet, boolean value) throws DriverException
    {
        for (int chan : channelSets[chanSet]) {
            htr.setOutput(chan, dsabColdChannelSet.contains(chan) ? false : value);
        }
    }


    /**
     *  Gets the output state.
     *
     *  @param  chanSet   The hardware channel set (cold or cryo)
     *  @return  The output state (true if any channel has output true)
     *  @throws  DriverException
     */
    @Override
    public boolean getOutput(int chanSet) throws DriverException
    {
        boolean state = false;
        for (int chan : channelSets[chanSet]) {
            state |= htr.getOutput(chan);
        }
        return state;
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @param  value    The voltage to set
     *  @throws  DriverException
     */
    @Override
    public void setVoltage(int chanSet, double value) throws DriverException
    {
        for (int chan : channelSets[chanSet]) {
            htr.setVoltage(chan, value);
        }
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @return  The read voltage (average of all channels in the set)
     *  @throws  DriverException
     */
    @Override
    public double readVoltage(int chanSet) throws DriverException
    {
        double voltage = 0.0;
        int count = 0;
        for (int chan : channelSets[chanSet]) {
            if (htr.getOutput(chan)) {
                voltage += htr.getVoltage(chan);
                count++;
            }
        }
        return count > 0 ? voltage / count : 0.0;
    }


    /**
     *  Sets the current.
     *
     *  This is a no-op.
     * 
     *  @param  chan   The hardware channel (not used)
     *  @param  value  The current to set
     */
    @Override
    public void setCurrent(int chan, double value)
    {
    }


    /**
     *  Reads the current.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @return  The read current (the sum over all channels in the set)
     *  @throws  DriverException
     */
    @Override
    public double readCurrent(int chanSet) throws DriverException
    {
        double current = 0;
        for (int chan : channelSets[chanSet]) {
            current += htr.readCurrent(chan);
        }
        return current;
    }

}
