package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Simulates the compressor Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class CompSimPlutoDevice extends CompPlutoDevice {

    /**
     *  Constants
     */
    private static final Map<String, Integer> latchMap = new LinkedHashMap<>();
    static {
        latchMap.put("AfterTmp", CompState.LATCH_AFTER_COOLER);
        latchMap.put("DischPrs", CompState.LATCH_DISCHARGE_PRESS);
        latchMap.put("DischTmp", CompState.LATCH_DISCHARGE_TEMP);
        latchMap.put("ExtPermit", CompState.LATCH_EXT_PERMIT);
        latchMap.put("LiquidTmp", CompState.LATCH_LIQUID_TEMP);
        latchMap.put("OilLevel", CompState.LATCH_OIL_LEVEL);
        latchMap.put("Power", CompState.LATCH_POWER);
        latchMap.put("SensValid", CompState.LATCH_SENSORS_VALID);
        latchMap.put("SmokeDetc", CompState.LATCH_SMOKE_DETC);
        latchMap.put("SuctnTmp", CompState.LATCH_SUCTION_TEMP);
    }
    private static final Map<String, Integer> conditionMap = new LinkedHashMap<>();
    static {
        conditionMap.put("CmpEnabled", CompState.COND_CMP_ENABLED);
        conditionMap.put("CmpOn6Hrs", CompState.COND_CMP_ON_6HRS);
        conditionMap.put("CmpPowered", CompState.COND_CMP_POWERED);
        conditionMap.put("CmpWaiting", CompState.COND_CMP_WAITING);
        conditionMap.put("CurrentValid", CompState.COND_CURRENT_VALID);
        conditionMap.put("CurrSensErr", CompState.COND_CURR_SENSOR_ERR);
        conditionMap.put("DiscPrsValid", CompState.COND_DISC_PRESS_VALID);
        conditionMap.put("DiscTmpValid", CompState.COND_DISC_TEMP_VALID);
        conditionMap.put("KeySwitchOn", CompState.COND_KEYSWITCH_ON);
        conditionMap.put("LatchesClear", CompState.COND_LATCHES_CLEAR);
        conditionMap.put("LiqdTmpValid", CompState.COND_LIQD_TEMP_VALID);
        conditionMap.put("OilLvlValid", CompState.COND_OIL_LEVEL_VALID);
        conditionMap.put("PowerLedOn", CompState.COND_POWER_LED);
        conditionMap.put("SuctPrsValid", CompState.COND_SUCT_PRESS_VALID);
        conditionMap.put("SuctTmpValid", CompState.COND_SUCT_TEMP_VALID);
        conditionMap.put("VoltageValid", CompState.COND_VOLTAGE_VALID);
    }

    /**
     *  Data fields.
     */
    private final boolean[] coldSwitches = new boolean[NUM_COLD_SWITCHES];
    private final boolean[] cryoSwitches = new boolean[NUM_CRYO_SWITCHES];
    private final LatchState[] latches = new LatchState[CompState.NUM_LATCHES];
    private final ConditionState[] conditions = new ConditionState[CompState.NUM_CONDITIONS];
    private boolean[] switches;
    private long powerTime = 0;
    private final Map<String, Integer> channelMap = new LinkedHashMap<>();
    private final List<Double> channelValues = new ArrayList<>();


    /**
     *  Constructor.
     */
    public CompSimPlutoDevice()
    {
        Arrays.fill(latches, LatchState.CLEAR);
        Arrays.fill(conditions, ConditionState.YES);
        conditions[CompState.COND_CMP_ENABLED] = ConditionState.NO;
        conditions[CompState.COND_CMP_WAITING] = ConditionState.NO;
        conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
    {
        int index = channelValues.size();
        channelMap.put(name, index);
        channelValues.add((double)index);
        return new int[]{index, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return channelValues.get(type);
    }


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    @Override
    public void setType(int type)
    {
        switches = type == CompState.TYPE_COLD ? coldSwitches : cryoSwitches;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw] = on;
        if (sw == SW_ENABLE) {
            if (on) {
                powerTime = System.currentTimeMillis() + 30000;
                conditions[CompState.COND_CMP_ENABLED] = ConditionState.YES;
                if (areLatchesClear()) {
                    conditions[CompState.COND_CMP_WAITING] = ConditionState.YES;
                    conditions[CompState.COND_CMP_POWERED] = ConditionState.YES;
                }
            }
            else {
                powerTime = 0;
                conditions[CompState.COND_CMP_ENABLED] = ConditionState.NO;
                conditions[CompState.COND_CMP_WAITING] = ConditionState.NO;
                conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
            }
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        if (powerTime != 0 && System.currentTimeMillis() >= powerTime) {
            conditions[CompState.COND_CMP_WAITING] = ConditionState.NO;
            powerTime = 0;
        }
        return switches[sw];
    }


    /**
     *  Gets the state of a latched condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    @Override
    public LatchState getLatchState(int cond)
    {
        return latches[cond];
    }


    /**
     *  Gets the state of a condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    @Override
    public ConditionState getConditionState(int cond)
    {
        return conditions[cond];
    }


    /**
     *  Resets all latches.
     */
    @Override
    public void resetLatches()
    {
        for (int cond = 0; cond < latches.length; cond++) {
            LatchState state = latches[cond];
            latches[cond] = state == LatchState.LATCHED ? LatchState.CLEAR : state;
        }
        if (switches[SW_ENABLE] && areLatchesClear()) {
            conditions[CompState.COND_CMP_POWERED] = ConditionState.YES;
        }
    }


    private boolean areLatchesClear()
    {
        boolean clear = true;
        for (LatchState state : latches) {
            if (state != LatchState.CLEAR) {
                clear = false;
                break;
            }
        }
        return clear;
    }


    @Command(type=Command.CommandType.QUERY, description="Get the list of latched condition names")
    public String getLatchNames()
    {
        return latchMap.keySet().toString();
    }

    @Command(type=Command.CommandType.ACTION, description="Make a latched condition active")
    public void setLatchActive(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[getLatch(cond)] = LatchState.ACTIVE;
        conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
    }

    @Command(type=Command.CommandType.ACTION, description="Make a latched condition latched")
    public void setLatchLatched(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[getLatch(cond)] = LatchState.LATCHED;
        conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
    }

    @Command(type=Command.CommandType.ACTION, description="Put a latched condition into warning state")
    public void setLatchWarning(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[getLatch(cond)] = LatchState.WARNING;
    }

    private int getLatch(String name) throws RefrigException {
        Integer cond = latchMap.get(name);
        if (cond == null) {
            throw new RefrigException("Unknown latched condition name: " + name);
        }
        return cond;
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of condition names")
    public String getConditionNames()
    {
        return conditionMap.keySet().toString();
    }

    @Command(type=Command.CommandType.ACTION, description="Set the state of a condition")
    public void setCondition(@Argument(description="Running condition name") String cond,
                             @Argument(description="Condition on state") boolean set) throws RefrigException {
        Integer condNum = conditionMap.get(cond);
        if (condNum == null) {
            throw new RefrigException("Unknown condition name: " + cond);
        }
        if (condNum != CompState.COND_CMP_POWERED) {
            conditions[condNum] = set ? ConditionState.YES : ConditionState.NO;
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws RefrigException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new RefrigException("Invalid channel name");
        }
        channelValues.set(index, value);
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
