package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.Maq20Control;
import org.lsst.ccs.subsystem.common.devices.Maq20Device;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Handles a compressor MAQ20 DIOL module.
 *
 *  @author Owen Saxton
 */
public class CompMaq20Device extends Maq20Device implements Compressor.SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        NUM_COLD_SWITCHES  = 0,
        SW_COOLANT_VALVE   = 1,
        SW_ORIFICE_VALVE   = 2,
        SW_BYPASS_VALVE    = 3,
        SW_SURGE_HEATER    = 4,
        NUM_CRYO_SWITCHES  = 4;

    /**
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Maq20Control ctrl;
    
    private static final Logger LOG = Logger.getLogger(CompMaq20Device.class.getName());
    private int numSwitches = 0;


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    public void setType(int type)
    {
        if (type == CompState.TYPE_CRYO && ctrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 discrete control", "not defined");
        }
        numSwitches = type == CompState.TYPE_CRYO ? NUM_CRYO_SWITCHES : NUM_COLD_SWITCHES;
    }


    /**
     *  Sets a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     *  @throws  RefrigException
     */
    @Override
    public void setSwitchOn(int sw, boolean on) throws RefrigException
    {
        try {
            ctrl.setLineOn(sw, on);
        }
        catch (DriverException e) {
            throw new RefrigException(e.getMessage());
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return ctrl.isLineOn(sw);
    }

}
