package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Line;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.RefrigState1;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the basic refrigeration subsystem.
 *
 *  @author Owen Saxton
 */
public class RefrigMain1 implements HasLifecycle, Monitor.AlarmHandler {

    /**
     *  Constants.
     */
    public static final int
        ALARM_HEATER = RefrigState1.MAX_COMPRESSORS,
        ALARM_ALL = ALARM_HEATER + 1;
    private static final int
        POWER_SET_INTVL = 5000;

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, HeaterControl> tempControls = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, Channel> allChannels = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, Line> allLines = new HashMap<>();

    private static final String REFRIG = "Refrig";
    private static final Logger LOG = Logger.getLogger(RefrigMain1.class.getName());

    private String[][] cmprPowerLines = new String[0][0];  // Names of the lines controlling the compressor power
    private String[] loadPowerLines = new String[0];  // Names of the lines controlling the load power
    private String[] alarmDsabChans = new String[0];  // Names of the channels whose alarms can be disabled
    private String tempCtrl;          // Name of the temperature control object

    private boolean coldStart;
    private final List<List<Line>> cmprPowerLinesL = new ArrayList<>();
    private final List<Line> loadPowerLinesL = new ArrayList<>();
    private final List<Channel> alarmDsabChansL = new ArrayList<>();

    private PowerDevice loadPowerDevcD;
    private HeaterControl tempCtrlC;
    private final RefrigState1 state = new RefrigState1();
    private Double startPower = 0.0;


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("heater-power", () -> setHeaterPower()).withPeriod(Duration.ofMillis(POWER_SET_INTVL));
        pts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the refrigeration subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a refrigetion subsystem.
        agentPropertiesService.setAgentProperty(RefrigAgentProperties.REFRIG_TYPE, RefrigMain1.class.getCanonicalName());

        // General initialization
        state.setTickMillis(getTickPeriod());

        // Get cold start option
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");

        // Initialize component references
        for (String[] cLines : cmprPowerLines) {
            List<Line> lines = new ArrayList<>();
            for (String lName : cLines) {
                Line line = allLines.get(lName);
                if (line != null) {
                    lines.add(line);
                }
            }
            cmprPowerLinesL.add(lines);
        }
        state.setNumCmprs(Math.min(RefrigState1.MAX_COMPRESSORS, cmprPowerLinesL.size()));
        if (cmprPowerLinesL.isEmpty()) {
            LOG.error("No valid compressors specified");
        }
        else if (state.getNumCmprs() != cmprPowerLines.length) {
            LOG.error("Some compressors are invalid");
        }
        for (int j = 0; j < state.getNumCmprs(); j++) {
            List<Line> lines = cmprPowerLinesL.get(j);
            if (lines.size() != cmprPowerLines[j].length) {
                LOG.error("Some (or all) control lines are invalid for compressor " + j);
            }
            else if (lines.isEmpty()) {
                LOG.error("No control lines specified for compressor " + j);
            }
        }

        for (String lName : loadPowerLines) {
            Line line = allLines.get(lName);
            if (line != null) {
                loadPowerLinesL.add(line);
            }
        }
        if (loadPowerLinesL.size() != loadPowerLines.length) {
            LOG.error("Some (or all) heater PS control lines are invalid");
        }
        else if (loadPowerLinesL.isEmpty()) {
            LOG.warn("No heater PS control lines specified: assume always on");
        }

        for (String cName : alarmDsabChans) {
            Channel chan = allChannels.get(cName);
            if (chan != null) {
                alarmDsabChansL.add(chan);
            }
        }
        if (alarmDsabChansL.size() != alarmDsabChans.length) {
            LOG.error("Some alarm disable channels are invalid");
        }

        tempCtrlC = tempControls.get(tempCtrl);
        if (tempCtrlC == null) {
            LOG.error("No valid temperature controller (or heater control) specified");
            state.setHeaterControlState(RefrigState1.CONTROL_NONE);   // Indicate no heater control
        }
        else {
            loadPowerDevcD = tempCtrlC.getPowerDevice();
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Set power control lines' warm start option
        for (List<Line> lines : cmprPowerLinesL) {
            for (Line line : lines) {
                line.setWarm(!coldStart);
            }
        }
        for (Line line : loadPowerLinesL) {
            line.setWarm(!coldStart);
        }

        // Turn off the power control lines' warm start option
        for (List<Line> lines : cmprPowerLinesL) {
            for (Line line : lines) {
                line.setWarm(false);
            }
        }
        for (Line line : loadPowerLinesL) {
            line.setWarm(false);
        }

        // Make sure state and power control lines are consistent
        for (int cmpr = 0; cmpr < state.getNumCmprs(); cmpr++) {
            setCmprPowerEnable(cmpr, isCmprPowerOn(cmpr) ? 1 : 0);
        }
        setHeaterPowerEnable(isHeaterPowerOn() ? 1 : 0);

        // Announce startup
        LOG.info("Refrigeration subsystem (" + name + ") started");
        //publishState();
    }


    /**
     *  Gets the full state of the refrigeration module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The full refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the full refrigeration state")
    public RefrigState1 getSystemState()
    {
        return state;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        state.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Handles alarm events.
     *
     *  @param  event  The event type
     *  @param  parm   The event parameter
     *  @param  cause  The alarm cause
     *  @param  name   The alarm name
     *  @return  Whether to process as an alarm
     */
    @Override
    public boolean processAlarm(int event, int parm, String cause, String name)
    {
        if ((parm >= 0 && parm < state.getNumCmprs()) || parm == ALARM_ALL) {
            int first = (parm == ALARM_ALL) ? 0 : parm;
            int last = (parm == ALARM_ALL) ? state.getNumCmprs() : parm + 1;
            for (int j = first; j < last; j++) {
                if (event == Alarm.EVENT_TRIP) {
                    if (state.getCmprPowerState(j) != RefrigState1.POWER_TRIPPED) {
                        state.setCmprPowerState(j, RefrigState1.POWER_TRIPPED);
                        setCmprPowerSwitch(j);
                        publishState();
                    }
                }
                else if (event == Alarm.EVENT_RESET) {
                    if (state.getCmprPowerState(j) == RefrigState1.POWER_TRIPPED) {
                        state.setCmprPowerState(j, RefrigState1.POWER_OFF);
                        publishState();
                    }
                }
            }
        }
        if (parm == ALARM_HEATER || parm == ALARM_ALL) {
            if (event == Alarm.EVENT_TRIP) {
                if (state.getHeaterPowerState() != RefrigState1.POWER_TRIPPED) {
                    state.setHeaterPowerState(RefrigState1.POWER_TRIPPED);
                    setHeaterPowerSwitch();
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if (state.getHeaterPowerState() == RefrigState1.POWER_TRIPPED) {
                    state.setHeaterPowerState(RefrigState1.POWER_OFF);
                    publishState();
                }
            }
        }
        return false;
    }


    /**
     *  Sets the power state of a compressor.
     *
     *  @param  cmpr   The compressor number
     *  @param  value  The enabled state value to set: 0 = off, ~0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set compressor power enabled state")
    public void setCmprPowerEnable(int cmpr, int value)
    {
        if (cmpr < 0 || cmpr >= state.getNumCmprs()) {
            LOG.error("Invalid compressor number");
            return;
        }
        int oldState = state.getCmprPowerState(cmpr);
        int newState = (oldState == RefrigState1.POWER_TRIPPED) ? oldState
                         : (value == 0) ? RefrigState1.POWER_OFF : RefrigState1.POWER_ON;
        if (newState != oldState) {
            state.setCmprPowerState(cmpr, newState);
            setCmprPowerSwitch(cmpr);
            publishState();
        }
    }


    /**
     *  Sets the heater power enable state.
     *
     *  @param  value    The enabled state value to set: 0 = off, ~0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set the heater power enabled state")
    public void setHeaterPowerEnable(int value)
    {
        int oldState = state.getHeaterPowerState();
        int newState = (oldState == RefrigState1.POWER_TRIPPED) ? oldState
                         : (value == 0) ? RefrigState1.POWER_OFF : RefrigState1.POWER_ON;
        if (newState != oldState) {
            state.setHeaterPowerState(newState);
            setHeaterPowerSwitch();
            publishState();
        }
    }


    /**
     *  Sets the heater control state.
     *
     *  @param  value  The heater power state to set: 0 = off; &gt; 0 = manual;
     *                 &lt; 0 = automatic (temp loop).
     */
    @Command(type=CommandType.ACTION, description="Set the heater control state")
    public void setHeaterControl(int value)
    {
        int oldState = state.getHeaterControlState();
        if (oldState != RefrigState1.CONTROL_NONE) {
            int newState = (value == 0) ? RefrigState1.CONTROL_OFF
                            : (value > 0) ? RefrigState1.CONTROL_ON
                            : tempCtrlC != null ? RefrigState1.CONTROL_AUTO
                            : RefrigState1.CONTROL_ON;
            if (newState != oldState) {
                state.setHeaterControlState(newState);
                if (newState == RefrigState1.CONTROL_AUTO) {
                    tempCtrlC.setTemp(state.getLoadTemp());
                    startTempControl();
                }
                else {
                    stopTempControl();
                    setHeaterPower();
                }
                publishState();
            }
        }
    }


    /**
     *  Sets the alarm disabled state on or off.
     *
     *  @param  value  The alarm disabled state value to set: 0 = off; ~0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set the load trip enabled state")
    public void setAlarmDisabled(int value)
    {
        boolean disable = (value != 0);
        state.setAlarmDisabled(disable);
        for (Channel chan : alarmDsabChansL) {
            chan.enableAlarm(false, !disable);
        }
        publishState();
    }


    /**
     *  Sets the heater power value.
     *
     *  @param  value  The load power set point.
     */
    @Command(type=CommandType.ACTION, description="Set the heater power set point")
    public void setHeaterPower(double value)
    {
        state.setHeaterPower(value);
        if (startPower != null) {
            startPower = value;
        }
        setHeaterPower();
        publishState();
    }


    /**
     *  Sets the load temperature value.
     *
     *  @param  value  The load temperature set point.
     */
    @Command(type=CommandType.ACTION, description="Set the load temperature set point")
    public void setLoadTemp(double value)
    {
        state.setLoadTemp(value);
        if (tempCtrlC != null) {
            tempCtrlC.setTemp(value);
        }
        publishState();
    }


    /**
     *  Saves a named refrigeration configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        sce.saveChangesForCategoriesAs(REFRIG + ":" + name);
    }


    /**
     *  Loads a named refrigeration configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Load a named configuration")
    public void loadNamedConfig(String name) throws IOException
    {
        sce.loadCategories(REFRIG + ":" + name);
    }

  
    /**
     *  Publishes the state of the refrigeration module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(RefrigState1.KEY, state));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Powers a compressor on or off, according to its state.
     */
    private void setCmprPowerSwitch(int cmpr)
    {
        boolean on = state.getCmprPowerState(cmpr) == RefrigState1.POWER_ON;
        for (Line line : cmprPowerLinesL.get(cmpr)) {
            line.set(on);
        }
    }


    /**
     *  Sets the heater power on or off, according to the state.
     */
    private void setHeaterPowerSwitch()
    {
        boolean on = state.getHeaterPowerState() == RefrigState1.POWER_ON;
        for (Line line : loadPowerLinesL) {
            line.set(on);
        }
        if (loadPowerDevcD != null) {
            if (on) {
                loadPowerDevcD.enable();
                if (loadPowerLinesL.isEmpty()) {
                    loadPowerDevcD.enableOutput(0, true);
                }
                if (state.getHeaterControlState() == RefrigState1.CONTROL_AUTO) {
                    startTempControl();
                }
            }
            else {
                stopTempControl();
                if (loadPowerLinesL.isEmpty()) {
                    loadPowerDevcD.enableOutput(0, false);
                }
                else {
                    loadPowerDevcD.disable();
                }
            }
        }
    }
            

    /**
     *  Sets the heater power.
     */
    private void setHeaterPower()
    {
        if (loadPowerDevcD == null || !loadPowerDevcD.isOnline()) return;
        if (state.getHeaterPowerState() != RefrigState1.POWER_ON) return;
        loadPowerDevcD.enableOutput(0, state.getHeaterControlState() != RefrigState1.CONTROL_OFF);
        if (state.getHeaterControlState() == RefrigState1.CONTROL_ON) {
            double loadPower = state.getHeaterPower();
            loadPowerDevcD.setPower(0, loadPower);
            startPower = loadPower;
        }
    }
    

    /**
     *  Gets the power state of a hardware compressor
     *
     *  @param  cmpr  The compressor number
     */
    private boolean isCmprPowerOn(int cmpr)
    {
        if (cmprPowerLinesL.isEmpty() || cmprPowerLinesL.get(cmpr).isEmpty()) return false;
        for (Line line : cmprPowerLinesL.get(cmpr)) {
            if (!line.isSet()) return false;
        }
        return true;
    }
            

    /**
     *  Gets the hardware heater power state.
     */
    private boolean isHeaterPowerOn()
    {
        if (loadPowerLinesL.isEmpty()) return true;
        for (Line line : loadPowerLinesL) {
            if (!line.isSet()) return false;
        }
        return true;
    }
    

    /**
     *  Starts the temperature control loop.
     */
    private void startTempControl()
    {
        if (tempCtrlC != null) {
            if (startPower == null) {
                tempCtrlC.restart();
            }
            else {
                tempCtrlC.start(startPower);
                startPower = null;
            }
        }
    }
    

    /**
     *  Stops the temperature control loop.
     */
    private void stopTempControl()
    {
        if (tempCtrlC != null) {
            tempCtrlC.stop();
        }
    }

}
