package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.ColdState;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.data.UpdatePeriod;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the refrigeration compressor subsystem.
 *
 *  @author Owen Saxton
 */
public class RefrigMain extends Subsystem implements HasLifecycle, ClearAlertHandler, AlertListener {

    /**
     *  Constants.
     */
    private static final ColdState[] coldStates = {ColdState.BOTH_OFF, ColdState.ONE_ON, ColdState.TWO_ON, ColdState.BOTH_ON};

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService peopertiesService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, ColdCompressor> coldCompMap = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, CryoCompressor> cryoCompMap = new LinkedHashMap<>();

    private static final Logger LOG = Logger.getLogger(RefrigMain.class.getName());
    private final Compressor[] coldComps = new Compressor[RefrigState.MAX_COLD_COMPRESSORS];
    private final Compressor[] cryoComps = new Compressor[RefrigState.MAX_CRYO_COMPRESSORS];
    private final RefrigState refrigState = new RefrigState();


    /**
     *  Constructor.
     */
    public RefrigMain() {
        super("refrig", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Check that "use full paths" has been set correctly
        String useFull = peopertiesService.getAgentProperty("org.lsst.ccs.use.full.paths");
        if (useFull == null || !useFull.equals("true")) {
            MonitorLogUtils.reportConfigError(LOG, name, "org.lsst.ccs.use.full.paths", "not defined as true");
        }

        // Create and schedule an AgentPeriodicTask to publish the refrigeration state when updated
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("compressor-state", () -> publishUpdatedRefrigState()).withPeriod(Duration.ofMillis(2000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Create and schedule an AgentPeriodicTask to publish the monitor update period - used by hex subsystem
        pt = new AgentPeriodicTask("update-time", () -> publishUpdateTime()).withPeriod(Duration.ofMillis(5000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Initialize the cold compressor state
        stateService.registerState(ColdState.class, "Overall cold compressor state", this);
        stateService.updateAgentState(ColdState.UNKNOWN);
    }


    /**
     *  Initializes the refrigeration subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a compressor subsystem.
        peopertiesService.setAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE, RefrigMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        // Initialize component references
        if (coldCompMap.isEmpty() && cryoCompMap.isEmpty()) {
            MonitorLogUtils.reportConfigError(LOG, name, "No compressors", "defined");
        }
        int nComp = coldCompMap.size();
        if (nComp > RefrigState.MAX_COLD_COMPRESSORS) {
            LOG.warning("Too many (" + nComp + ") cold compressors specified: using only the first "
                           + RefrigState.MAX_COLD_COMPRESSORS);
            List<String> keys = new ArrayList(coldCompMap.keySet());
            for (int j = RefrigState.MAX_COLD_COMPRESSORS; j < nComp; j++) {
                coldCompMap.remove(keys.get(j));
            }
        }
        nComp = 0;
        for (Compressor comp : coldCompMap.values()) {
            coldComps[nComp] = comp;
            comp.setIndex(nComp++);
        }
        refrigState.setNumColdComps(nComp);
        nComp = cryoCompMap.size();
        if (nComp > RefrigState.MAX_CRYO_COMPRESSORS) {
            LOG.warning("Too many (" + nComp + ") cryo compressors specified: using only the first "
                           + RefrigState.MAX_CRYO_COMPRESSORS);
            List<String> keys = new ArrayList(cryoCompMap.keySet());
            for (int j = RefrigState.MAX_CRYO_COMPRESSORS; j < nComp; j++) {
                cryoCompMap.remove(keys.get(j));
            }
        }
        nComp = 0;
        for (Compressor comp : cryoCompMap.values()) {
            cryoComps[nComp] = comp;
            comp.setIndex(nComp++);
        }
        refrigState.setNumCryoComps(nComp);
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Refrigeration subsystem started with " + coldCompMap.size() + " cold compressors and "
                    + cryoCompMap.size() + " cryo compressors");
    }


    /**
     *  Gets the state of the refrigeration module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the refrigeration state")
    public RefrigState getSystemState()
    {
        synchronized (refrigState) {
            refrigState.clearCompStates();
            refrigState.setTickMillis(getTickPeriod());
            for (Compressor comp : coldCompMap.values()) {
                refrigState.addColdState(comp.getState());
            }
            for (Compressor comp : cryoCompMap.values()) {
                refrigState.addCryoState(comp.getState());
            }
            return refrigState;
        }
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        synchronized (refrigState) {
            setTickPeriod(value);
            refrigState.clearCompStates();
            publishState();
        }
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     * 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        String id = alert.getAlertId();
        for (String compName : coldCompMap.keySet()) {
            if (id.startsWith(compName)) {
                return coldCompMap.get(compName).canClearAlert(alert);
            }
        }
        for (String compName : cryoCompMap.keySet()) {
            if (id.startsWith(compName)) {
                return cryoCompMap.get(compName).canClearAlert(alert);
            }
        }
        return ClearAlertCode.UNKNOWN_ALERT;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            if (id.endsWith("PLC")) {
                boolean reset = false;
                for (String compName : coldCompMap.keySet()) {
                    if (id.startsWith(compName)) {
                        coldCompMap.get(compName).resetLatches();
                        reset = true;
                        break;
                    }
                }
                if (reset) continue;
                for (String compName : cryoCompMap.keySet()) {
                    if (id.startsWith(compName)) {
                        cryoCompMap.get(compName).resetLatches();
                        break;
                    }
                }
            }
        }
    }


    /**
     *  Publishes the updated refrigeration state.
     *
     *  This is called regularly as a timer task.
     */
    private void publishUpdatedRefrigState()
    {
        synchronized (refrigState) {
            boolean changed = false;
            refrigState.clearCompStates();
            int running = 0;
            for (Compressor comp : coldCompMap.values()) {
                comp.checkLimits();
                if (comp.updateState()) {
                    changed = true;
                    refrigState.addColdState(comp.getState());
                }
                CompressorState compState = comp.getState().getCompressorState();
                if (compState == CompressorState.RUNNING || compState == CompressorState.WAITING) {
                    running |= (1 << comp.getIndex());
                }
            }
            stateService.updateAgentState(coldStates[running]);
            for (Compressor comp : cryoCompMap.values()) {
                comp.controlValves();
                comp.checkLimits();
                if (comp.updateState()) {
                    changed = true;
                    refrigState.addCryoState(comp.getState());
                }
            }
            if (changed) {
                publishState();
            }
        }
    }


    /**
     *  Publishes the update time.
     *
     *  This is called regularly as a timer task.
     */
    private void publishUpdateTime()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(UpdatePeriod.KEY, new UpdatePeriod(getTickPeriod())));
    }


    /**
     *  Publishes the state of the refrigeration module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        refrigState.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(RefrigState.KEY, refrigState));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
