package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements a temperature controller for the refrigeration system.
 *
 *  @author Owen Saxton
 */
public class HeaterControl implements HasLifecycle {

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, PowerDevice> powerDevices = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();

    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput;         // maximum PID output (watts)
    private Double  awGain;            // anti-windup gain
    private Double  basePower;         // base power input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 0.0;   // minimum PID output (watts)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)
    private String  powerDevc;         // The power device name
    private Integer powerChan;         // The power device channel
    private String[] tempChans;        // The temperature channel names to use

    private static final Logger LOG = Logger.getLogger(HeaterControl.class.getName());
    private PIController pic;          // The PI controller
    private PowerDevice powerDevcC;    // The power controller object
    private final List<Channel>tempChansL = new ArrayList<>();  // Temperature channels to use
    private double lastPower;          // The last power value set
    private boolean active;            // True if loop is running


    /**
     *  Start temperature control task.
     */
    @Override
    public void build() {
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask(name + "-timer", () -> iterateLoop()).withPeriod(Duration.ofMillis(updateTime));
        pts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void postInit()
    {
        if (gain == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "gain", "is missing");
        }
        if (timeConst == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "timeConst", "is missing");
        }
        if (smoothTime == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "smoothTime", "is missing");
        }
        if (maxOutput == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "maxOutput", "is missing");
        }
        if (awGain == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "awGain", "is missing");
        }
        if (basePower == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "basePower", "is missing");
        }
        if (tolerance == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "tolerance", "is missing");
        }
        if (updateTime == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "updateTime", "is missing");
        }
        if (powerDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "powerDevc", "is missing");
        }
        powerDevcC = powerDevices.get(powerDevc);
        if (powerDevcC == null) {
            MonitorLogUtils.reportConfigError(LOG, name, powerDevc, "doesn't exist");
        }
        if (powerChan == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "powerChan", "is missing");
        }
        if (tempChans == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "tempChans", "is missing");
        }
        for (String cName : tempChans) {
            Channel chan = allChannels.get(cName);
            if (chan == null) {
                MonitorLogUtils.reportConfigError(LOG, name, cName, "doesn't exist");
            }
            tempChansL.add(chan);
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Gets the power device,
     * 
     *  @return  The power device
     */
    public PowerDevice getPowerDevice()
    {
        return powerDevcC;
    }


    /**
     *  Gets the power channel,
     * 
     *  @return  The power channel
     */
    public int getPowerChannel()
    {
        return powerChan;
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  value  The temperature to set
     */
    public void setTemp(double value)
    {
        pic.setSetpoint(value);
   }


    /**
     *  Resets the controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Restarts the control loop.
     */
    public void restart()
    {
        start(lastPower);
    }


    /**
     *  Starts the control loop.
     *
     *  @param  power  The starting power value
     */
    public synchronized void start(double power)
    {
        if (active) return;
        active = true;
        pic.reset();
        pic.setIntegral(power - basePower);
    }


    /**
     *  Stops the control loop.
     */
    public synchronized void stop()
    {
        if (!active) return;
        active = false;
    }


    /**
     *  Timer method for control loop iteration.
     */
    private synchronized void iterateLoop()
    {
        if (!active) return;
        int count = 0;
        double temp = 0.0;
        for (Channel tempChan : tempChansL) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                temp += value;
                count++;
            }
        }
        if (count > 0) {
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(new double[]{temp / count}, tod);
            powerDevcC.setPower(powerChan, lastPower);
        }
        else {
            LOG.error("Control loop iteration failed: no valid temperature values available");
        }
    }

}
