package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.devices.Maq20PWMControl;
import org.lsst.ccs.subsystem.common.PIController;

/**
 *  Implements a fan speed loop controller for the utility trunk system.
 *
 *  @author Owen Saxton
 */
public class FanControl implements HasLifecycle {

    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput = 10.0;  // maximum PID output (inverse duty cycle)
    private Double  awGain;            // anti-windup gain
    private Double  baseDuty;          // base duty cycle input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 1.0;   // minimum PID output (inverse duty cycle)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)
    private String  ambientTemp;       // Ambient temperature channel name
    private String  cabinetTemp;       // Cabinet temperature channel name
    private String  fanCntrl;          // The fan controller name

    @LookupPath
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, Channel> allChannels = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, Maq20PWMControl> allFans = new HashMap<>();
    
    private static final Logger LOG = Logger.getLogger(FanControl.class.getName());
    private Channel ambChannel;         // Ambient temperature channel
    private Channel cabChannel;         // Cabinet temperature channel
    private Maq20PWMControl maqControl; // Maq20 fan controller
    private PIController pic;           // The PI controller
    private double lastDuty;            // The last inverse duty cycle value set
    private boolean active;             // True if loop is active
    private double setTemp = 1.0;       // Temperature difference set point


    /**
     *  Sets up the fan speed control loop timer task.
     */
    @Override
    public void build() {
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask(name + "-timer", () -> iterateLoop()).withPeriod(Duration.ofMillis(updateTime));
        pts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void postInit()
    {
        if (gain == null) {
            reportConfigError("gain", "is missing");
        }
        if (timeConst == null) {
            reportConfigError("timeConst", "is missing");
        }
        if (smoothTime == null) {
            reportConfigError("smoothTime", "is missing");
        }
        if (maxOutput == null) {
            reportConfigError("maxOutput", "is missing");
        }
        if (awGain == null) {
            reportConfigError("awGain", "is missing");
        }
        if (baseDuty == null) {
            reportConfigError("baseDuty", "is missing");
        }
        if (tolerance == null) {
            reportConfigError("tolerance", "is missing");
        }
        if (updateTime == null) {
            reportConfigError("updateTime", "is missing");
        }
        ambChannel = ambientTemp == null ? null : allChannels.get(ambientTemp);
        if (ambChannel == null) {
            reportConfigError("ambientTemp", "is missing or is not a Channel");
        }
        cabChannel = cabinetTemp == null ? null : allChannels.get(cabinetTemp);
        if (cabChannel == null) {
            reportConfigError("cabinetTemp", "is missing or is not a Channel");
        }
        maqControl = fanCntrl == null ? null : allFans.get(fanCntrl);
        if (maqControl == null) {
            reportConfigError("fanCtrl", "is missing or is not a Maq20PWMControl");
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(baseDuty);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
        pic.setSetpoint(setTemp);
    }


    /**
     *  Sets the fan speed (PWM duty cycle).
     *
     *  @param  duty  The duty cycle to set (0 - 1)
     */
    private void setDutyCycle(double duty)
    {
        try {
            maqControl.setDutyCycle1(duty);
        }
        catch (Exception ex) {
            LOG.log(Level.SEVERE, "Error setting {0} fan speed: {1}", new Object[]{name, ex});
        }
    }


    /**
     *  Sets the target temperature difference.
     *
     *  @param  temp  The temperature to set
     */
    public void setTemperature(double temp)
    {
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature difference.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Sets the gain.
     *
     *  @param  gain  The gain to set
     */
    public void setGain(double gain)
    {
        this.gain = gain;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setPID(gain, timeConst);
        }
    }


    /**
     *  Gets the gain.
     *
     *  @return  The gain
     */
    public double getGain()
    {
        return gain;
   }


    /**
     *  Sets the time constant.
     *
     *  @param  time  The time constant to set
     */
    public void setTimeConstant(double time)
    {
        timeConst = time;
        pic.setPID(gain, timeConst);
    }


    /**
     *  Gets the time constant.
     *
     *  @return  The time constant
     */
    public double getTimeConstant()
    {
        return timeConst;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  duty  The initial duty cycle value
     */
    public synchronized void startLoop(double duty)
    {
        if (active) return;
        lastDuty = duty == 0 ? maxOutput : Math.min(1.0 / duty, maxOutput);
        startLoop();
    }


    /**
     *  Starts the control loop.
     */
    public synchronized void startLoop()
    {
        if (active) return;
        pic.reset();
        pic.setIntegral(lastDuty - baseDuty);
        active = true;
    }


    /**
     *  Stops the control loop.
     */
    public void stopLoop()
    {
        active = false;
        //setDutyCycle(1.0);  // Maximum fan speed
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the PI controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private synchronized void iterateLoop()
    {
        if (!active) return;
        double tempDiff = cabChannel.getValue() - ambChannel.getValue();
        if (!Double.isNaN(tempDiff)) {
            double tod = (double)System.currentTimeMillis() / 1000;
            lastDuty = pic.performPI(new double[]{tempDiff}, tod);
            setDutyCycle(1.0 / lastDuty);
            //System.out.format("Temp: %s, Inv. duty factor: %s, Set point: %s, Error: %s, Integral: %s\n",
            //                  temp / count, lastDuty, pic.getSetpoint(), pic.getError(), pic.getIntegral());
        }
        else {
            LOG.log(Level.SEVERE, "{0} control loop iteration failed: temperature values not available", name);
        }
    }


    /**
     *  Reports configuration error and terminates.
     */
    private void reportConfigError(String param, String desc)
    {
        LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{name, param, desc});
        throw new RuntimeException("Fatal configuration error");
    }

}
