package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.CompAlert;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Controls a refrigeration compressor.
 *
 *  @author Owen Saxton
 */
public class Compressor implements HasLifecycle {

    public static interface SwitchDevice {

        public void setSwitchOn(int chan, boolean on) throws RefrigException;

        public Boolean isSwitchOn(int chan);

    }

    public static class LimitData {

        private Channel channel = null;    // Monitor channel with the value
        private boolean isLower = false;   // Is lower limit, not upper
        private double immedLimit = 0.0;   // The immediate limit, or NaN if not applicable
        private double delayLimit = 0.0;   // The delayed limit, or NaN if not applicable
        private int delayTime = 0;         // The delay time (ms), or -1 if 6-hour on
        private double value = 0.0;        // The value that exceeded the limit
        private long endTime;              // The computed end time

    }

    /**
     *  Constants.
     */
    private static final String COMP_LIMITS = "CompLimits";

    private static final Map<Integer, CompAlert> ccsAlertMap = new HashMap<>();
    static {
        ccsAlertMap.put(CompState.SWC_DISC_PRESS, CompAlert.DISC_PRESS_HIGH);
        ccsAlertMap.put(CompState.SWC_DISC_TEMP, CompAlert.DISC_TEMP_HIGH);
        ccsAlertMap.put(CompState.SWC_CMPR_POWER, CompAlert.COMP_POWER_HIGH);
        ccsAlertMap.put(CompState.SWC_LIQUID_TEMP, CompAlert.LIQD_TEMP_HIGH);
        ccsAlertMap.put(CompState.SWC_OIL_LEVEL, CompAlert.OIL_LEVEL_LOW);
        ccsAlertMap.put(CompState.SWC_SUCT_TEMP, CompAlert.SUCT_TEMP_LOW);
        ccsAlertMap.put(CompState.SWC_PHASE_SEP_TEMP, CompAlert.PHASE_TEMP_HIGH);
        ccsAlertMap.put(CompState.SWC_PLATE_TEMP, CompAlert.CRYO_TEMP_LOW);
    }
    private static final Map<Integer, CompAlert> plcAlertMap = new HashMap<>();
    static {
        plcAlertMap.put(CompState.LATCH_DISCHARGE_PRESS, CompAlert.DISC_PRESS_HIGH_PLC);
        plcAlertMap.put(CompState.LATCH_DISCHARGE_TEMP, CompAlert.DISC_TEMP_HIGH_PLC);
        plcAlertMap.put(CompState.LATCH_POWER, CompAlert.COMP_POWER_HIGH_PLC);
        plcAlertMap.put(CompState.LATCH_LIQUID_TEMP, CompAlert.LIQD_TEMP_HIGH_PLC);
        plcAlertMap.put(CompState.LATCH_OIL_LEVEL, CompAlert.OIL_LEVEL_LOW_PLC);
        plcAlertMap.put(CompState.LATCH_SUCTION_TEMP, CompAlert.SUCT_TEMP_LOW_PLC);
        plcAlertMap.put(CompState.LATCH_AFTER_COOLER, CompAlert.AFTER_TEMP_HIGH_PLC);
        plcAlertMap.put(CompState.LATCH_SENSORS_VALID, CompAlert.SENS_READ_BAD_PLC);
        plcAlertMap.put(CompState.LATCH_SMOKE_DETC, CompAlert.SMOKE_DETC_PLC);
        plcAlertMap.put(CompState.LATCH_EXT_PERMIT, CompAlert.EXT_PERMIT_PLC);
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private CompPlutoDevice plutoDevc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private CompMaq20Device maq20Devc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private FanControl fanControl;

    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double discPressImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double discPressDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    int discPressDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double discTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double discTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    int discTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double suctTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double cmprPowerImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double cmprPowerDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    int cmprPowerDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double phaseSepTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    int phaseSepTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double liquidTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double liquidTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    int liquidTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double coldTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double orificeOffPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    double orificeOnPress;

    protected String discPressChan, discTempChan, suctTempChan, cmprPowerChan, phaseSepTempChan, liquidTempChan;

    private static final Logger LOG = Logger.getLogger(Compressor.class.getName());
    private final CompState state;
    private final SwitchDevice[] switchDevices;
    private final int[] switchChannels;
    private final int type;
    private Channel discPress, discTemp, suctTemp, cmprPower, phaseSepTemp, liquidTemp;
    private int activeConds = 0;
    private final LimitData[] limitData = new LimitData[CompState.NUM_SW_CONDITIONS];
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();


    /**
     *  Constructor.
     *
     *  @param  state  The compressor state object
     */
    public Compressor(CompState state)
    {
        this.state = state;
        type = state.getType();
        switchDevices = new SwitchDevice[CompState.NUM_SWITCHES];
        switchChannels = new int[CompState.NUM_SWITCHES];
        for (int j = 0; j < CompState.NUM_SW_CONDITIONS; j++) {
            limitData[j] = new LimitData();
        }
    }


    /**
     *  Initializes the compressor control.
     */
    @Override
    public void postInit()
    {
        if (plutoDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "plutoDevc", "not specified");
        }
        if (maq20Devc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "maq20Devc", "not specified");
        }
        if (discTempChan != null) {
            discTemp = channelMap.get(discTempChan);
        }
        if (discTemp == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "discTempChan", "not specified or not defined");
        }
        if (suctTempChan != null) {
            suctTemp = channelMap.get(suctTempChan);
        }
        if (suctTemp == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "suctTempChan", "not specified or not defined");
        }
        if (discPressChan != null) {
            discPress = channelMap.get(discPressChan);
        }
        if (discPress == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "discPressChan", "not specified or not defined");
        }
        if (cmprPowerChan != null) {
            cmprPower = channelMap.get(cmprPowerChan);
        }
        if (cmprPower == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "cmprPowerChan", "not specified or not defined");
        }
        limitData[CompState.SWC_DISC_PRESS].channel = discPress;
        limitData[CompState.SWC_DISC_TEMP].channel = discTemp;
        limitData[CompState.SWC_SUCT_TEMP].channel = suctTemp;
        limitData[CompState.SWC_SUCT_TEMP].isLower = true;
        limitData[CompState.SWC_SUCT_TEMP].delayLimit = Double.NaN;
        limitData[CompState.SWC_CMPR_POWER].channel = cmprPower;
        if (type == CompState.TYPE_COLD) {
            limitData[CompState.SWC_DISC_PRESS].channel = discPress;
            if (liquidTempChan != null) {
                liquidTemp = channelMap.get(liquidTempChan);
            }
            if (liquidTemp == null) {
                MonitorLogUtils.reportConfigError(LOG, name, "liquidTempChan", "not specified or not defined");
            }
            limitData[CompState.SWC_LIQUID_TEMP].channel = liquidTemp;
            switchDevices[CompState.SW_ENABLE] = plutoDevc;
            switchChannels[CompState.SW_ENABLE] = CompPlutoDevice.SW_ENABLE;
            switchDevices[CompState.SW_LIGHTS] = plutoDevc;
            switchChannels[CompState.SW_LIGHTS] = CompPlutoDevice.SW_LIGHTS;
        }
        else {
            limitData[CompState.SWC_DISC_PRESS].delayTime = -1;
            if (phaseSepTempChan != null) {
                phaseSepTemp = channelMap.get(phaseSepTempChan);
            }
            if (phaseSepTemp == null) {
                MonitorLogUtils.reportConfigError(LOG, name, "phaseSepTempChan", "not specified or not defined");
            }
            limitData[CompState.SWC_PHASE_SEP_TEMP].channel = phaseSepTemp;
            limitData[CompState.SWC_PHASE_SEP_TEMP].immedLimit = Double.NaN;
            switchDevices[CompState.SW_ENABLE] = plutoDevc;
            switchChannels[CompState.SW_ENABLE] = CompPlutoDevice.SW_ENABLE;
            switchDevices[CompState.SW_LIGHTS] = plutoDevc;
            switchChannels[CompState.SW_LIGHTS] = CompPlutoDevice.SW_LIGHTS;
            switchDevices[CompState.SW_HEATER] = plutoDevc;
            switchChannels[CompState.SW_HEATER] = CompPlutoDevice.SW_HEATER;
            switchDevices[CompState.SW_ORIFICE_VALVE] = maq20Devc;
            switchChannels[CompState.SW_ORIFICE_VALVE] = CompMaq20Device.SW_ORIFICE_VALVE;
            switchDevices[CompState.SW_COOLANT_VALVE] = maq20Devc;
            switchChannels[CompState.SW_COOLANT_VALVE] = CompMaq20Device.SW_COOLANT_VALVE;
            switchDevices[CompState.SW_BYPASS_VALVE] = maq20Devc;
            switchChannels[CompState.SW_BYPASS_VALVE] = CompMaq20Device.SW_BYPASS_VALVE;
            switchDevices[CompState.SW_SURGE_HEATER] = maq20Devc;
            switchChannels[CompState.SW_SURGE_HEATER] = CompMaq20Device.SW_SURGE_HEATER;
        }
        state.setName(name);
        plutoDevc.setType(type);
        maq20Devc.setType(type);
        if (fanControl != null) {
            LOG.info("Compressor " + name + " has an associated fan speed controller");
        }
    }


    /**
     *  Starts the fan control loop..
     */
    @Override
    public void postStart()
    {
        if (fanControl != null) {
            fanControl.startLoop(0.0);
        }
    }


    /**
     *  Sets the discharge pressure immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressImmedLimit(double value)
    {
        discPressImmedLimit = value;
        limitData[CompState.SWC_DISC_PRESS].immedLimit = value;
    }


    /**
     *  Sets the discharge pressure delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayLimit(double value)
    {
        discPressDelayLimit = value;
        limitData[CompState.SWC_DISC_PRESS].delayLimit = value;
    }


    /**
     *  Sets the discharge pressure delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayTime(int value)
    {
        discPressDelayTime = value;
        if (type == CompState.TYPE_COLD) {
            limitData[CompState.SWC_DISC_PRESS].delayTime = 1000 * value;
        }
    }


    /**
     *  Sets the discharge temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempImmedLimit(double value)
    {
        discTempImmedLimit = value;
        limitData[CompState.SWC_DISC_TEMP].immedLimit = value;
    }


    /**
     *  Sets the discharge temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempDelayLimit(double value)
    {
        discTempDelayLimit = value;
        limitData[CompState.SWC_DISC_TEMP].delayLimit = value;
    }


    /**
     *  Sets the discharge temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempDelayTime(int value)
    {
        discTempDelayTime = value;
        limitData[CompState.SWC_DISC_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the suction temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setSuctTempImmedLimit(double value)
    {
        suctTempImmedLimit = value;
        limitData[CompState.SWC_DISC_TEMP].immedLimit = value;
    }


    /**
     *  Sets the compressor power immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerImmedLimit(double value)
    {
        cmprPowerImmedLimit = value;
        limitData[CompState.SWC_CMPR_POWER].immedLimit = value;
    }


    /**
     *  Sets the compressor power delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerDelayLimit(double value)
    {
        cmprPowerDelayLimit = value;
        limitData[CompState.SWC_CMPR_POWER].delayLimit = value;
    }


    /**
     *  Sets the compressor power delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerDelayTime(int value)
    {
        cmprPowerDelayTime = value;
        limitData[CompState.SWC_CMPR_POWER].delayTime = 1000 * value;
    }


    /**
     *  Sets the liquid temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempImmedLimit(double value)
    {
        liquidTempImmedLimit = value;
        if (type == CompState.TYPE_COLD) {
            limitData[CompState.SWC_LIQUID_TEMP].immedLimit = value;
        }
    }


    /**
     *  Sets the liquid temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayLimit(double value)
    {
        liquidTempDelayLimit = value;
        if (type == CompState.TYPE_COLD) {
            limitData[CompState.SWC_LIQUID_TEMP].delayLimit = value;
        }
    }


    /**
     *  Sets the liquid temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayTime(int value)
    {
        liquidTempDelayTime = value;
        if (type == CompState.TYPE_COLD) {
            limitData[CompState.SWC_LIQUID_TEMP].delayTime = 1000 * value;
        }
    }


    /**
     *  Sets the phase separator temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayLimit(double value)
    {
        phaseSepTempDelayLimit = value;
        if (type == CompState.TYPE_CRYO) {
            limitData[CompState.SWC_PHASE_SEP_TEMP].delayLimit = value;
        }
    }


    /**
     *  Sets the phase separator temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayTime(int value)
    {
        phaseSepTempDelayTime = value;
        if (type == CompState.TYPE_CRYO) {
            limitData[CompState.SWC_PHASE_SEP_TEMP].delayTime = 1000 * value;
        }
    }


    /**
     *  Sets the compressor index.
     *
     *  @param  index  The index value
     */
    public void setIndex(int index)
    {
        state.setIndex(index);
    }


    /**
     *  Gets the compressor index.
     *
     *  @return  The index value
     */
    public int getIndex()
    {
        return state.getIndex();
    }


    /**
     *  Command to turn a switch on or off..
     *
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a compressor's switch state")
    public synchronized void setSwitchOn(@Argument(description="Switch number") int sw,
                                         @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        if (!state.getValidSwitches().contains(sw)) {
            throw new RefrigException("Invalid switch number: " + sw);
        }
        if (sw == CompState.SW_ENABLE && on && activeConds != 0) return;
        setSwitch(sw, on);
        if (sw == CompState.SW_ENABLE) {
            setValves(on);
        }
    }


    /**
     *  Resets the latches.
     */
    @Command(type=CommandType.ACTION, description="Reset a compressor's latches")
    public void resetLatches()
    {
        plutoDevc.resetLatches();
    }


    /**
     *  Updates the compressor state.
     *
     *  @return  Whether any value changed
     */
    public synchronized boolean updateState()
    {
        boolean changed = false;
        CompressorState compState = null;
        for (int sw : state.getValidSwitches()) {
            Boolean on = switchDevices[sw].isSwitchOn(switchChannels[sw]);
            SwitchState swState = on == null ? SwitchState.OFFLINE : on ? SwitchState.ON : SwitchState.OFF;
            if (swState != state.getSwitchState(sw)) {
                state.setSwitchState(sw, swState);
                changed = true;
            }
        }
        for (int cond : state.getValidLatches()) {
            LatchState latchState = plutoDevc.getLatchState(cond);
            if (latchState != state.getLatchState(cond)) {
                state.setLatchState(cond, latchState);
                CompAlert alert = plcAlertMap.get(cond);
                if (latchState == LatchState.ACTIVE) {
                    raiseAlert(alert, AlertState.ALARM, name + " PLC error condition set");
                }
                else if (latchState == LatchState.WARNING) {
                    raiseAlert(alert, AlertState.WARNING, name + " PLC warning condition set");
                }
                else {
                    lowerAlert(alert, name + " PLC error condition cleared");
                }
                changed = true;
            }
        }
        boolean poweredOff = false;
        for (int cond : state.getValidConditions()) {
            ConditionState condState = plutoDevc.getConditionState(cond);
            if (condState == ConditionState.OFF) {
                compState = CompressorState.OFFLINE;
            }
            if (condState != state.getConditionState(cond)) {
                if (cond == CompState.COND_CMP_POWERED && condState != ConditionState.YES) {
                    poweredOff = true;
                }
                state.setConditionState(cond, condState);
                changed = true;
            }
        }
        if (poweredOff && state.getConditionState(CompState.COND_CMP_ENABLED) == ConditionState.YES) {
            setValves(false);
        }
        for (int cond : state.getValidSwConditions()) {
            LimitData ld = limitData[cond];
            SwCondState swCondState = ld.endTime == 0 ? SwCondState.CLEAR :
                                      ld.endTime > 0 ? SwCondState.DLYPEND :
                                      ld.endTime == -1 ? SwCondState.DLYACTV : SwCondState.ACTIVE;
            if (swCondState != state.getSwConditionState(cond)) {
                state.setSwConditionState(cond, swCondState);
                CompAlert alert = ccsAlertMap.get(cond);
                String value = String.format("%.1f", ld.value);
                if (swCondState == SwCondState.ACTIVE) {
                    raiseAlert(alert, AlertState.ALARM, name + " CCS immediate error condition set: value (" + value
                                                          + (ld.isLower ? ") < " : ") > ") + ld.immedLimit);
                }
                else if (swCondState == SwCondState.DLYACTV) {
                    String timeText = ld.delayTime >= 0 ? " for " + ld.delayTime / 1000 + " sec" : " and compressor on for 6 hours";
                    raiseAlert(alert, AlertState.ALARM, name + " CCS delayed error condition set: value (" + value
                                                          + (ld.isLower ? ") < " : ") > ") + ld.delayLimit + timeText);
                }
                else if (swCondState == SwCondState.DLYPEND) {
                    raiseAlert(alert, AlertState.WARNING, name + " CCS delayed error condition pending: value (" + value
                                                            + (ld.isLower ? ") < " : ") > ") + ld.delayLimit);
                }
                else {
                    lowerAlert(alert, name + " CCS error condition cleared: value = " + value);
                }
                changed = true;
            }
        }
        if (compState == null) {
            if (activeConds != 0) {
                compState = CompressorState.SW_DSAB;
            }
        }
        if (compState == null) {
            compState = state.getConditionState(CompState.COND_CMP_WAITING) == ConditionState.YES ? CompressorState.WAITING :
                        state.getConditionState(CompState.COND_CMP_POWERED) == ConditionState.YES ? CompressorState.RUNNING :
                        state.getConditionState(CompState.COND_LATCHES_CLEAR) == ConditionState.NO ? CompressorState.HW_DSAB :
                        CompressorState.STOPPED;
        }
        if (compState != state.getCompressorState()) {
            state.setCompressorState(compState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Gets the compressor state.
     *
     *  @return  The compressor state
     */
    public CompState getState()
    {
        return state;
    }    
            

    /**
     *  Checks compressor shut-off limits.
     */
    public synchronized void checkLimits()
    {
        for (int cond = 0; cond < CompState.NUM_SW_CONDITIONS; cond++) {
            LimitData item = limitData[cond];
            if (item.channel == null) continue;
            item.value = item.channel.getValue();
            long endTime = item.endTime;
            if (!item.isLower && item.value > item.immedLimit 
                  || item.isLower && item.value < item.immedLimit) {
                endTime = -2;
            }
            else if (!item.isLower && item.value > item.delayLimit
                       || item.isLower && item.value < item.delayLimit) {
                if (endTime == 0) {
                    endTime = System.currentTimeMillis() + item.delayTime;
                }
                if (endTime >= 0) {
                    if (item.delayTime < 0) {
                        if (state.getConditionState(CompState.COND_CMP_ON_6HRS) == ConditionState.YES) {
                            endTime = -1;
                        }
                    }
                    else if (System.currentTimeMillis() > endTime) {
                        endTime = -1;
                    }
                }
            }
            else {
                endTime = 0;
            }
            if (endTime != item.endTime) {
                item.endTime = endTime;
                if (endTime < 0) {
                    setCondition(cond);
                }
                else {
                    clearCondition(cond);
                }
            }
        }
    }


    /**
     *  Control the cryo valves.
     */
    public synchronized void controlValves()
    {
        if (type != CompState.TYPE_CRYO) return;
        if (plutoDevc.getConditionState(CompState.COND_CMP_POWERED) != ConditionState.YES) return;
        double press = limitData[CompState.SWC_DISC_PRESS].channel.getValue();
        try {
            if (press > orificeOffPress) {
                setSwitch(CompState.SW_ORIFICE_VALVE, false);
            }
            else if (press < orificeOnPress) {
                setSwitch(CompState.SW_ORIFICE_VALVE, true);
            }
        }
        catch (RefrigException e) {
            LOG.severe("Error operating " + name + " orifice valve: " + e);
        }
    }


    /**
     *  Turns a switch on or off..
     *
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    private void setSwitch(int sw, boolean on) throws RefrigException
    {
        switchDevices[sw].setSwitchOn(switchChannels[sw], on);
    }


    /**
     *  Sets a condition that disables the compressor
     *
     *  @param  cond  The condition number
     */
    private void setCondition(int cond)
    {
        if (activeConds == 0) {
            try {
                setSwitch(CompState.SW_ENABLE, false);
                setValves(false);
            }
            catch (RefrigException e) {
                LOG.severe("Error shutting off " + name + " compressor: " + e);
            }
        }
        activeConds |= (1 << cond);
    }


    /**
     *  Clears a condition that disabled the compressor
     *
     *  @param  cond  The condition number
     */
    private void clearCondition(int cond)
    {
        activeConds &= ~(1 << cond);
    }


    /**
     *  Sets the state of the cryo compressor valves.
     *
     *  @param  on  Whether to activate the orifice valve
     */
    private void setValves(boolean on)
    {
        if (type == CompState.TYPE_CRYO) {
            try {
                setSwitch(CompState.SW_BYPASS_VALVE, false);
                setSwitch(CompState.SW_ORIFICE_VALVE, on);
            }
            catch (RefrigException e) {
                LOG.severe("Error operating " + name + " compressor valves: " + e);
            }
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The refrigeration alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     */
    private void raiseAlert(CompAlert alert, AlertState state, String cond)
    {
        activeAlertMap.put(alert.getId(name), true);
        alertService.raiseAlert(alert.newAlert(name), state, cond);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The refrigeration alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(CompAlert alert, String cond)
    {
        Boolean active = activeAlertMap.put(alert.getId(name), false);
        if (active != null && active) {
            alertService.raiseAlert(alert.newAlert(name), AlertState.NOMINAL, cond);
        }
    }


    /**
     *  Enables an alert to be cleared.
     * 
     *  @param  alert  The alert
     *  @return  Action code
     */
    public ClearAlertCode canClearAlert(Alert alert)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }

}
