package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Defines a power supply interface.
 *
 *  @author Owen Saxton
 */
public class PowerDevice extends Device {

    protected static final int
        NETWORK_CONN_INVALID = 0x01,
        CONN_TYPE_NOT_CONFIG = 0x02,
        DEVC_ID_NOT_CONFIG   = 0x04,
        DEVC_PARM_NOT_CONFIG = 0x08;

    protected static final int
        MON_TYPE_POWER = 0,
        MON_TYPE_VOLTAGE = 1,
        MON_TYPE_CURRENT = 2;

    private final static double
        MIN_VOLTAGE = 4,
        MAX_CURRENT = 3,
        TIMEOUT = 2.0;

    private static final String
        REFRIG = "Refrig",
        CONN_TYPE = "connType",
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm";

    protected static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("POWER", MON_TYPE_POWER);
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

    private final Map<Integer, PowerState> states = new HashMap<>();

    @ConfigurationParameter(name=CONN_TYPE, category=REFRIG, isFinal=true)
    protected DriverConstants.ConnType connType;
    @ConfigurationParameter(name=DEVC_ID, category=REFRIG, isFinal=true)
    protected String devcId;
    @ConfigurationParameter(name=DEVC_PARM, category=REFRIG, isFinal=true)
    protected int devcParm = 0;

    protected double[] maxCurrent;   // Current limits
    protected double timeout = TIMEOUT;

    private static final Logger LOG = Logger.getLogger(PowerDevice.class.getName());
    private final String devcName;
    protected PowerSupplyDriver psd;
    private final int options, minChan, maxChan;
    private final double voltages[], currents[];
    
    /**
     *  Inner class for saving the power state of a channel.
     */
    public static class PowerState {
        
        private double resistance, power;
        private boolean enabled;

    }


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  psd       The power supply device driver object
     *  @param  options   The options word
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    public PowerDevice(String name, PowerSupplyDriver psd, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
        maxCurrent = new double[maxChan - minChan + 1];
        Arrays.fill(maxCurrent, MAX_CURRENT);
        voltages = new double[maxChan - minChan + 1];
        currents = new double[maxChan - minChan + 1];
    }


    /**
     *  Gets the minimum channel number.
     *
     *  @return  The minimum channel number
     */
    public int getMinChannel()
    {
        return minChan;
    }


    /**
     *  Gets the maximum channel number.
     *
     *  @return  The maximum channel number
     */
    public int getMaxChannel()
    {
        return maxChan;
    }

   
    /**
     *  Gets the connection type.
     *
     *  @return  The connection type enum
     */
    @Command(type=Command.CommandType.QUERY, description="Gets the connection type")
    public DriverConstants.ConnType getConnType()
    {
        return connType;
    }


    /**
     *  Gets the device ID.
     *
     *  @return  The device identification string
     */
    @Command(type=Command.CommandType.QUERY, description="Gets the device ID")
    public String getDevcId()
    {
        return devcId;
    }


    /**
     *  Gets the device parameter.
     *
     *  @return  The device parameter
     */
    @Command(type=Command.CommandType.QUERY, description="Gets the device parameter")
    public int getDevcParm()
    {
        return devcParm;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (connType == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        if (connType == DriverConstants.ConnType.NET && (options & NETWORK_CONN_INVALID)  != 0) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "connType", "is invalid: " + connType);
        }
        if (devcId == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = devcName + (devcId.isEmpty() ? "" : " (" + devcId + ")");
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            psd.open(connType, devcId, devcParm);
            setOnline(true);
            initSensors();
            //psd.setTimeout(timeout);
            for (int j = 0; j < maxCurrent.length; j++) {
                psd.setCurrent(maxCurrent[j], minChan + j);
            }
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            psd.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < minChan || hwChan > maxChan) {
            MonitorLogUtils.reportError(LOG, name, "HW channel", hwChan);
        }
        Integer iType = mTypeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(LOG, name, "type", type);
        }
        return new int[]{iType, 0};
    }


    /**
     *  Initializes voltages and currents as unread.
     */
    @Override
    protected void readChannelGroup()
    {
        Arrays.fill(voltages, Double.NaN);
        Arrays.fill(currents, Double.NaN);
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double power = Double.NaN;
        if (online) {
            try {
                switch (type) {
                case MON_TYPE_CURRENT:
                    power = currents[hwChan - minChan] = psd.readCurrent(hwChan);
                    break;

                case MON_TYPE_VOLTAGE:
                    power = voltages[hwChan - minChan] = psd.readVoltage(hwChan);
                    break;

                case MON_TYPE_POWER:
                    power = voltages[hwChan - minChan] * currents[hwChan - minChan];
                    if (Double.isNaN(power)) {
                        power = psd.readVoltage(hwChan) * psd.readCurrent(hwChan);
                    }
                    break;
                }
            }
            catch (DriverException e) {
                LOG.severe("Error reading from " + fullName + ": " + e);
                setOnline(false);
            }
        }
        return power;
    }


    /**
     *  Enables the output.
     *
     *  @param  chan    The hardware channel
     *  @param  enable  The enabled state to set
     */
    public void enableOutput(int chan, boolean enable)
    {
        getPowerState(chan).enabled = enable;
        setOutput(chan, enable);
    }


    /**
     *  Sets the power.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The power to set
     */
    public void setPower(int chan, double value)
    {
        PowerState state = getPowerState(chan);
        if (!state.enabled) return;
        List<Float> voltages = new ArrayList<>();
        state.power = value;
        if (state.resistance == 0) {
            boolean okay = false;
            double ohms = 0; 
            for (int j = 0; j < 5 && !okay; j++) {
                double prevOhms = ohms;
                setVoltage(chan, MIN_VOLTAGE);
                double amps = readCurrent(chan);
                ohms = (amps <= 0) ? 0 : readVoltage(chan) / amps;
                okay = areClose(ohms, prevOhms);
            }
            if (okay && ohms != 0) {
                state.resistance = ohms;
            }
            else {
                LOG.severe("Unable to determine load resistance");
                return;
            }
        }
        boolean okay = false;
        double ohms = 0;
        for (int j = 0; j < 5 && !okay; j++) {
            double prevOhms = ohms;
            double volts = Math.sqrt(state.power * state.resistance);
            setVoltage(chan, volts);
            voltages.add((float)volts);
            double amps = readCurrent(chan);
            ohms = (amps <= 0) ? 0 : readVoltage(chan) / amps;
            okay = areClose(ohms, state.resistance) || areClose(ohms, prevOhms);
        }
        if (okay) {
            if (ohms != 0) {
                state.resistance = ohms;
                double volts = Math.sqrt(state.power * state.resistance);
                setVoltage(chan, volts);
                voltages.add((float)volts);
            }
        }
        else {
            LOG.severe("Unable to set power consistently");
        }
        //if (LOG.getLevel().intValue() <= Level.FINE.intValue()) {
            StringBuilder msg = new StringBuilder("Setting power to ");
            msg.append((float)value).append(": voltages =");
            for (Float v : voltages) {
                msg.append(" ").append(v);
            }
            LOG.fine(msg.toString());
        //}
    }


    private boolean areClose(double val1, double val2)
    {
        if (val2 == 0) {
            return val1 == 0;
        }
        return Math.abs((val1 - val2) / val2) < 0.05;
    }

   
    /**
     *  Gets the enabled state.
     *
     *  @param  chan   The hardware channel
     *  @return  Whether enabled
     */
    public boolean isEnabled(int chan)
    {
        return getPowerState(chan).enabled;
    }


    /**
     *  Sets the online state.
     *
     *  If coming on line, set previously-cached value.
     *  @param  online  The online state to set: true or false
     */
    @Override
    protected void setOnline(boolean online)
    {
        super.setOnline(online);
        if (online) {
            for (Integer chan : states.keySet()) {
                setPower(chan, states.get(chan).power);
            }
        }
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The voltage to set
     */
    public void setVoltage(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            psd.setVoltage(value, chan);
        }
        catch (DriverException e) {
            LOG.severe("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Sets the current.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The current to set
     */
    public void setCurrent(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            psd.setCurrent(value, chan);
        }
        catch (DriverException e) {
            LOG.severe("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Sets the output state.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The output state to set
     */
    public void setOutput(int chan, boolean value)
    {
        if (!testOnline()) return;
        try {
            psd.setOutput(value, chan);
        }
        catch (DriverException e) {
            LOG.severe("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chan  The hardware channel
     *  @return  The voltage
     */
    public double readVoltage(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return psd.readVoltage(chan);
        }
        catch (DriverException e) {
            LOG.severe("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }


    /**
     *  Reads the current.
     *
     *  @param  chan  The hardware channel
     *  @return  The current
     */
    public double readCurrent(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return psd.readCurrent(chan);
        }
        catch (DriverException e) {
            LOG.severe("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }


    /**
     *  Gets the output state.
     *  @param  chan   The hardware channel
     *  @return  The output state
     */
    public boolean getOutput(int chan)
    {
        if (!testOnline()) return false;
        try {
            return psd.getOutput(chan);
        }
        catch (DriverException e) {
            LOG.severe("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return false;
        }
    }


    /**
     *  Gets the power state for a channel.
     *
     *  @param  chan   The hardware channel
     *  @return  The power state
     */
    private PowerState getPowerState(int chan)
    {
        PowerState state = states.get(chan);
        if (state == null) {
            state = new PowerState();
            states.put(chan, state);
        }

        return state;
    }

}
