package org.lsst.ccs.subsystem.refrig;

import java.util.Arrays;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;

/**
 *  Handles the vacuum system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class CompSimPlutoDevice extends CompPlutoDevice {

    /**
     *  Constants
     */

    /**
     *  Data fields.
     */
    private final boolean[] coldSwitches = new boolean[NUM_COLD_SWITCHES];
    private final boolean[] cryoSwitches = new boolean[NUM_CRYO_SWITCHES];
    private final LatchState[] latches = new LatchState[CompState.NUM_LATCHES];
    private final ConditionState[] conditions = new ConditionState[CompState.NUM_CONDITIONS];
    private boolean[] switches;
    private long powerTime = 0;


    /**
     *  Constructor.
     */
    public CompSimPlutoDevice()
    {
        Arrays.fill(latches, LatchState.CLEAR);
        Arrays.fill(conditions, ConditionState.YES);
        conditions[CompState.COND_CMP_ENABLED] = ConditionState.NO;
        conditions[CompState.COND_CMP_WAITING] = ConditionState.NO;
        conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
    {
        return new int[]{0, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return 0.0;
    }


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    @Override
    public void setType(int type)
    {
        switches = type == CompState.TYPE_COLD ? coldSwitches : cryoSwitches;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw] = on;
        if (sw == SW_ENABLE) {
            if (on) {
                powerTime = System.currentTimeMillis() + 30000;
                conditions[CompState.COND_CMP_ENABLED] = ConditionState.YES;
                if (areLatchesClear()) {
                    conditions[CompState.COND_CMP_WAITING] = ConditionState.YES;
                    conditions[CompState.COND_CMP_POWERED] = ConditionState.YES;
                }
            }
            else {
                powerTime = 0;
                conditions[CompState.COND_CMP_ENABLED] = ConditionState.NO;
                conditions[CompState.COND_CMP_WAITING] = ConditionState.NO;
                conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
            }
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        if (powerTime != 0 && System.currentTimeMillis() >= powerTime) {
            conditions[CompState.COND_CMP_WAITING] = ConditionState.NO;
            powerTime = 0;
        }
        return switches[sw];
    }


    /**
     *  Gets the state of a latched condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    @Override
    public LatchState getLatchState(int cond)
    {
        return latches[cond];
    }


    /**
     *  Gets the state of a condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    @Override
    public ConditionState getConditionState(int cond)
    {
        return conditions[cond];
    }


    /**
     *  Resets all latches.
     */
    @Override
    public void resetLatches()
    {
        for (int cond = 0; cond < latches.length; cond++) {
            LatchState state = latches[cond];
            latches[cond] = state == LatchState.LATCHED ? LatchState.CLEAR : state;
        }
        if (switches[SW_ENABLE] && areLatchesClear()) {
            conditions[CompState.COND_CMP_POWERED] = ConditionState.YES;
        }
    }


    private boolean areLatchesClear()
    {
        boolean clear = true;
        for (LatchState state : latches) {
            if (state != LatchState.CLEAR) {
                clear = false;
                break;
            }
        }
        return clear;
    }


    @Command(type = Command.CommandType.ACTION)
    public void setLatchActive(@Argument(description="Latched condition number") int cond) {
        latches[cond] = LatchState.ACTIVE;
        conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setLatchLatched(@Argument(description="Latched condition number") int cond) {
        latches[cond] = LatchState.LATCHED;
        conditions[CompState.COND_CMP_POWERED] = ConditionState.NO;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setLatchWarning(@Argument(description="Latched condition number") int cond) {
        latches[cond] = LatchState.WARNING;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setCondition(@Argument(description="Running condition number") int cond,
                             @Argument(description="Condition on state") boolean set) {
        if (cond != CompState.COND_CMP_POWERED) {
            conditions[cond] = set ? ConditionState.YES : ConditionState.NO;
        }
    }

}
