package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a WattsUp power meter.
 *
 *  @author Owen Saxton
 */
public class WUDevice extends Device implements WattsUp.Listener {

    /**
     *  Constants.
     */
    private static final int
        CHAN_WATTS  = 0,
        CHAN_VOLTS  = 1,
        CHAN_FREQ   = 2,
        N_DEV_CHANS = 3,
        LOG_PERIOD  = 1;

    /**
     *  Data fields.
     */
    private String   node;        // Name of the node attached to the meter
    private String   serial;      // The serial number of the meter

    private static final Logger LOG = Logger.getLogger(WUDevice.class.getName());
    private WattsUp  wtu;         // WattsUp hardware object
    private final double[] value = new double[N_DEV_CHANS];  // Current values


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        super.configure(mon);
        fullName = "WattsUp meter (" + (node == null ? "local" : node)
                     + (serial == null ? "" : ":" + serial) + ")";
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            if (!inited || wtu == null) {
                wtu = new WattsUp();
                wtu.addListener(this);
            }
            wtu.openFtdi(serial, 0);
            wtu.restart();
            setOnline(true);
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
        }
        finally {
            inited = true;
        }
    }


    /**
     *  Closes the connection.
     *
     *  This is a no-op
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            MonitorLogUtils.reportError(LOG, name, "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  chan  The hardware channel number.
     *  @param  type  The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int chan, int type)
    {
        return online ? value[chan] : super.readChannel(chan, type);
    }


    /**
     *  Changes the WattsUp? meter powered state.
     *
     *  Performs meter setup upon power-on.
     *
     *  @param  on  Whether the meter is powered on or not
     */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields((1 << WattsUp.FLD_WATTS) |
                                    (1 << WattsUp.FLD_VOLTS) |
                                    (1 << WattsUp.FLD_FREQUENCY));
                wtu.setExternalLogging(LOG_PERIOD);
            }
            catch (DriverException e) {
                LOG.error("Error configuring " + fullName + ": " + e);
            }
        }
        else {
            for (int j = 0; j < N_DEV_CHANS; j++) {
                value[j] = 0F;
            }
        }
    }


    /**
     *  Changes the WattsUp? meter open state.
     */
    @Override
    public void setClosed()
    {
        setOnline(false);
    }


    /**
     *  Receives data periodically from the WattsUp? meter.
     *
     *  @param  data  The array of data from the meter.
     */
    @Override
    public void processData(double[] data)
    {
        value[CHAN_WATTS] = data[WattsUp.FLD_WATTS];
        value[CHAN_VOLTS] = data[WattsUp.FLD_VOLTS];
        value[CHAN_FREQ]  = data[WattsUp.FLD_FREQUENCY];
    }

}
