package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.ColdState;
import org.lsst.ccs.subsystem.refrig.constants.HeaterControlState;
import org.lsst.ccs.subsystem.refrig.constants.HeaterPowerState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.ThermalAlert;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the refrigeration thermal control subsystem.
 *
 *  @author Owen Saxton
 */
public class ThermalMain implements HasLifecycle, AgentPresenceListener, StatusMessageListener, ClearAlertHandler {

    /**
     *  Inner class for holding trim heater parameters.
     */
    static class TrimHeater {

        TempControl tempCtrl;
        PowerDevice powerDevc;
        int powerChan;
        Double startPower = 0.0;

        TrimHeater(TempControl tempCtrl, PowerDevice powerDevc, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Inner class for holding auxiliary heater parameters.
     */
    static class AuxHeater {

        PowerDevice powerDevc;
        int powerChan;

        AuxHeater(PowerDevice powerDevc, int powerChan)
        {
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Constants.
     */
    private static final String THERMAL_LIMITS = "ThermalLimits";

    private static final int
        POWER_SET_INTVL = 5000;

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService as;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, TempControl> tempControlMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, PowerDevice> powerDeviceMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, Channel> allChannels = new HashMap<>();

    @ConfigurationParameter(category=THERMAL_LIMITS, isFinal=true)
    private double coldTempLimit = 40.0;

    private String coldTempCtrl;          // Name of the cold plate temperature control object
    private String cryoTempCtrl;          // Name of the cryo plate temperature control object
    private String[] auxPowerNames;       // Names of the auxiliary heater power objects (3)
    private String[] coldTempNames;       // Names of the cold plate temperature channels

    private static final Logger LOG = Logger.getLogger(ThermalMain.class.getName());
    private final TrimHeater[] trimHeaters = new TrimHeater[ThermalState.NUM_TRIM_HEATERS];
    private final AuxHeater[] auxHeaters = new AuxHeater[ThermalState.NUM_AUX_HEATERS];
    private final List<Channel> coldTemps = new ArrayList<>();
    private final ThermalState thermalState = new ThermalState();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private int highTempCount = 0;
    private ColdState refrigColdState = ColdState.UNKNOWN;


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the refrigeration state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("heater-power", () -> setHeaterPower()).withPeriod(Duration.ofMillis(POWER_SET_INTVL));
        pts.scheduleAgentPeriodicTask(pt);
        pt = new AgentPeriodicTask("thermal-state", () -> updateThermalState()).withPeriod(Duration.ofMillis(1000));
        pts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the thermal subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a thermal subsystem.
        aps.setAgentProperty(RefrigAgentProperties.THERMAL_TYPE, ThermalMain.class.getCanonicalName());

        // Add an agent present listener
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);

        // General initialization
        thermalState.setTickMillis(getTickPeriod());

        // Initialize component references
        for (int htr = 0; htr < ThermalState.NUM_TRIM_HEATERS; htr++) {
            TempControl ctrl = tempControlMap.get(htr == ThermalState.TRIM_HEATER_COLD ? coldTempCtrl : cryoTempCtrl);
            if (ctrl == null) {
                String type = htr == ThermalState.TRIM_HEATER_COLD ? "cold" : "cryo";
                LOG.error("No " + type + "-plate temperature controller (or heater control) specified");
            }
            else {
                trimHeaters[htr] = new TrimHeater(ctrl, ctrl.getPowerDevice(), ctrl.getPowerChannel());
            }
        }

        if (auxPowerNames == null) {
            LOG.info("No auxiliary heater devices specified");
        }
        else if (auxPowerNames.length > ThermalState.NUM_AUX_HEATERS) {
            MonitorLogUtils.reportConfigError(LOG, name, "auxHeaterPower", "contains more than 3 items");
        }
        else {
            for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
                String name = htr < auxPowerNames.length ? auxPowerNames[htr]: null;
                if (name == null) {
                    LOG.info("Auxiliary heater " + htr + " has not been specified");
                }
                else {
                    PowerDevice power = powerDeviceMap.get(name);
                    if (power == null) {
                        MonitorLogUtils.reportConfigError(LOG, name, "auxHeaterPower", "contains non-power item");
                    }
                    auxHeaters[htr] = new AuxHeater(power, 0);
                }
            }
        }

        if (coldTempNames == null) {
            LOG.severe("No cold-plate temperature monitoring channels specified");
        }
        else {
            for (String coldName : coldTempNames) {
                Channel chan = allChannels.get(coldName);
                if (chan == null) {
                    MonitorLogUtils.reportConfigError(LOG, name, "coldTempNames", "contains non-channel item");
                }
                coldTemps.add(chan);
            }
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        for (int htr = 0; htr < ThermalState.NUM_TRIM_HEATERS; htr++) {
            TrimHeater heater = trimHeaters[htr];
            if (heater != null) {
                thermalState.setTrimHeaterState(htr, !heater.powerDevc.isOnline() ? HeaterPowerState.OFFLINE :
                                                     heater.powerDevc.getOutput(heater.powerChan) ? HeaterPowerState.ON
                                                                                                  : HeaterPowerState.OFF);
                thermalState.setTrimHeaterCtrlState(htr, HeaterControlState.OFF);
            }
        }
        for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
            AuxHeater heater = auxHeaters[htr];
            if (heater != null) {
                thermalState.setAuxHeaterState(htr, !heater.powerDevc.isOnline() ? HeaterPowerState.OFFLINE :
                                                    heater.powerDevc.getOutput(heater.powerChan) ? HeaterPowerState.ON
                                                                                                 : HeaterPowerState.OFF);
            }
        }
        LOG.info("Thermal control system started");
    }


    /**
     *  Listens for the arrival of the refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                subsys.getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName));
                break;
            }
        }
    }


    /**
     *  Listens for the departure of the refrigeration subsystem.
     *
     *  @param  agent  Agent going absent
     */
    @Override
    public void disconnecting(AgentInfo agent) {
        if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
            subsys.getMessagingAccess().removeStatusMessageListener(this);
            //refrigColdState = ColdState.UNKNOWN;
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        refrigColdState = msg.getState().getState(ColdState.class);
    }


    /**
     *  Gets the state of the thermal control system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=CommandType.QUERY, description="Get the thermal control state")
    public ThermalState getSystemState()
    {
        return thermalState;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        thermalState.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Enables/disables a cold trim heater section.
     *
     *  @param  section  The cold section (y-minus or y-plus)
     *  @param  enable   Whether to enable
     */
    @Command(type=CommandType.ACTION, description="Enables/disables a cold trim heater section")
    public void enableColdSection(int section, boolean enable)
    {
        TrimHeater heater = trimHeaters[ThermalState.TRIM_HEATER_COLD];
        if (heater != null) {
            synchronized(heater) {
                thermalState.enableColdSection(section, enable);
                ((HeaterPsDevice)heater.powerDevc).enableColdSection(section, enable);
                setHeaterPower();
            }
        }
        publishState();
    }


    /**
     *  Sets a trim heater power enable state.
     *
     *  @param  htr    The heater number
     *  @param  value  The enabled state value to set: 0 = off, ~0 = on.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater power enabled state")
    public void setTrimHeaterPowerEnable(int htr, int value) throws RefrigException
    {
        try {
            synchronized(getTrimHeater(htr)) {
                HeaterPowerState oldState = thermalState.getTrimHeaterState(htr);
                if (oldState != HeaterPowerState.OFFLINE && oldState != HeaterPowerState.DISABLD) {
                    HeaterPowerState newState = (value != 0) ? HeaterPowerState.ON : HeaterPowerState.OFF;
                    if (newState != oldState) {
                        thermalState.setTrimHeaterState(htr, newState);
                        setTrimHeaterPowerSwitch(htr);
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a trim heater control state.
     *
     *  @param  htr    The heater number
     *  @param  value  The heater control state to set: 0 = off; &gt; 0 = manual;
     *                  &lt; 0 = automatic (temp loop).
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater control state")
    public void setTrimHeaterControl(int htr, int value) throws RefrigException
    {
        try {
            TrimHeater heater = getTrimHeater(htr);
            synchronized(heater) {
                HeaterControlState oldState = thermalState.getTrimHeaterCtrlState(htr);
                if (oldState != HeaterControlState.OFFLINE) {
                    HeaterControlState newState = (value == 0) ? HeaterControlState.OFF :
                                                  (value > 0) ? HeaterControlState.POWER : HeaterControlState.TEMP;
                    if (newState != oldState) {
                        thermalState.setTrimHeaterCtrlState(htr, newState);
                        if (newState == HeaterControlState.TEMP) {
                            heater.tempCtrl.setTemp(thermalState.getPlateTemperature(htr));
                            startTempControl(htr);
                        }
                        else {
                            stopTempControl(htr);
                            setTrimHeaterPower(htr);
                        }
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a trim heater power value.
     *
     *  @param  htr    The heater number
     *  @param  value  The load power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater power set point")
    public void setTrimHeaterPower(int htr, double value) throws RefrigException
    {
        try {
            TrimHeater heater = getTrimHeater(htr);
            synchronized(heater) {
                thermalState.setTrimHeaterPower(htr, value);
                if (heater.startPower != null) {
                    heater.startPower = value;
                }
                setTrimHeaterPower(htr);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a plate temperature value.
     *
     *  @param  htr    The heater number
     *  @param  value  The plate temperature set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a plate temperature set point")
    public void setPlateTemperature(int htr, double value) throws RefrigException
    {
        try {
            TrimHeater heater = getTrimHeater(htr);
            synchronized(heater) {
                thermalState.setPlateTemperature(htr, value);
                heater.tempCtrl.setTemp(value);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets an aux heater power enable state.
     *
     *  @param  htr    The heater number
     *  @param  value  The enabled state value to set: 0 = off, ~0 = on.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set an aux heater power enabled state")
    public void setAuxHeaterPowerEnable(int htr, int value) throws RefrigException
    {
        try {
            synchronized(getAuxHeater(htr)) {
                HeaterPowerState oldState = thermalState.getAuxHeaterState(htr);
                if (oldState != HeaterPowerState.OFFLINE && oldState != HeaterPowerState.DISABLD) {
                    HeaterPowerState newState = (value != 0) ? HeaterPowerState.ON : HeaterPowerState.OFF;
                    if (newState != oldState) {
                        thermalState.setAuxHeaterState(htr, newState);
                        setAuxHeaterPower(htr);
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets an aux heater power value.
     *
     *  @param  htr    The heater number
     *  @param  value  The power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set an aux heater power set point")
    public void setAuxHeaterPower(int htr, double value) throws RefrigException
    {
        try {
            synchronized(getAuxHeater(htr)) {
                thermalState.setAuxHeaterPower(htr, value);
                setAuxHeaterPower(htr);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Publishes the state of the thermal module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ThermalState.KEY, thermalState));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Updates the thermal state periodically.
     */
    private void updateThermalState()
    {
        boolean coldHigh = false;
        for (Channel chan : coldTemps) {
            coldHigh |= (chan.getValue() > coldTempLimit);
        }
        if (coldHigh) {
            coldHigh &= (++highTempCount >= 3);
        }
        else {
            highTempCount = 0;
        }
        boolean changed = false;
        for (int htr = 0; htr < ThermalState.NUM_TRIM_HEATERS; htr++) {
            TrimHeater heater = trimHeaters[htr];
            if (heater == null) continue;
            synchronized(heater) {
                HeaterPowerState state = coldHigh && htr == ThermalState.TRIM_HEATER_COLD ? HeaterPowerState.DISABLD :
                                         !heater.powerDevc.isOnline() ? HeaterPowerState.OFFLINE :
                                         heater.powerDevc.isEnabled(heater.powerChan) ? HeaterPowerState.ON : HeaterPowerState.OFF;
                if (state != thermalState.getTrimHeaterState(htr)) {
                    thermalState.setTrimHeaterState(htr, state);
                    setTrimHeaterPowerSwitch(htr);
                    changed = true;
                }
            }
        }
        boolean coldCompsOff = refrigColdState == ColdState.BOTH_OFF || refrigColdState == ColdState.UNKNOWN;
        for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
            AuxHeater heater = auxHeaters[htr];
            if (heater == null) continue;
            synchronized(heater) {
                HeaterPowerState state = (coldHigh || coldCompsOff) ? HeaterPowerState.DISABLD :
                                         !heater.powerDevc.isOnline() ? HeaterPowerState.OFFLINE :
                                         heater.powerDevc.isEnabled(heater.powerChan) ? HeaterPowerState.ON : HeaterPowerState.OFF;
                if (state != thermalState.getAuxHeaterState(htr)) {
                    thermalState.setAuxHeaterState(htr, state);
                    setAuxHeaterPower(htr);
                    changed = true;
                }
            }
        }
        if (changed) {
            publishState();
            if (coldHigh) {
                raiseAlert(ThermalAlert.COLD_TEMP_HIGH, AlertState.ALARM, "At least one cold-plate RTD is above " + coldTempLimit + " C");
            }
            else {
                lowerAlert(ThermalAlert.COLD_TEMP_HIGH, "No cold-plate RTD is above " + coldTempLimit + " C");
            }
            if (coldCompsOff) {
                raiseAlert(ThermalAlert.COLD_COMP_OFF, AlertState.ALARM, "Both cold compressors are turned off");
            }
            else {
                lowerAlert(ThermalAlert.COLD_COMP_OFF, "At least one cold compressor is back on");
            }
        }
    }


    /**
     *  Sets a trim heater power on or off, according to its state.
     */
    private void setTrimHeaterPowerSwitch(int htr)
    {
        TrimHeater heater = trimHeaters[htr];
        if (thermalState.getTrimHeaterState(htr) == HeaterPowerState.ON) {
            heater.powerDevc.enableOutput(heater.powerChan, true);
            if (thermalState.getTrimHeaterCtrlState(htr) == HeaterControlState.TEMP) {
                startTempControl(htr);
            }
        }
        else {
            stopTempControl(htr);
            heater.powerDevc.enableOutput(heater.powerChan, false);
        }
    }
            

    /**
     *  Sets the power for all heaters.
     *
     *  This routine is executed periodically to ensure that the correct
     *  values are maintained.
     */
    private void setHeaterPower()
    {
        for (int htr = 0; htr < ThermalState.NUM_TRIM_HEATERS; htr++) {
            setTrimHeaterPower(htr);
        }
        for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
            setAuxHeaterPower(htr);
        }
    }


    /**
     *  Sets the power for a trim heater.
     */
    private void setTrimHeaterPower(int htr)
    {
        TrimHeater heater = trimHeaters[htr];
        if (heater == null || !heater.powerDevc.isOnline()) return;
        if (thermalState.getTrimHeaterState(htr) != HeaterPowerState.ON) return;
        heater.powerDevc.enableOutput(heater.powerChan, thermalState.getTrimHeaterCtrlState(htr) != HeaterControlState.OFF);
        if (thermalState.getTrimHeaterCtrlState(htr) == HeaterControlState.POWER) {
            double loadPower = thermalState.getTrimHeaterPower(htr);
            heater.powerDevc.setPower(heater.powerChan, loadPower);
            heater.startPower = loadPower;
        }
    }
    

    /**
     *  Sets the power for an aux heater.
     */
    private void setAuxHeaterPower(int htr)
    {
        AuxHeater heater = auxHeaters[htr];
        if (heater == null || !heater.powerDevc.isOnline()) return;
        heater.powerDevc.enableOutput(heater.powerChan, thermalState.getAuxHeaterState(htr) == HeaterPowerState.ON);
        heater.powerDevc.setPower(heater.powerChan, thermalState.getAuxHeaterPower(htr));
    }
    

    /**
     *  Starts a temperature control loop.
     */
    private void startTempControl(int htr)
    {
        TrimHeater heater = trimHeaters[htr];
        if (heater != null) {
            if (heater.startPower == null) {
                heater.tempCtrl.restart();
            }
            else {
                heater.tempCtrl.start(heater.startPower);
                heater.startPower = null;
            }
        }
    }
    

    /**
     *  Stops a temperature control loop.
     */
    private void stopTempControl(int htr)
    {
        TrimHeater heater = trimHeaters[htr];
        if (heater != null) {
            heater.tempCtrl.stop();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The thermal alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     */
    private void raiseAlert(ThermalAlert alert, AlertState state, String cond)
    {
        Boolean active = activeAlertMap.put(alert.getId(), true);
        if (active == null || !active) {
            as.raiseAlert(alert.newAlert(), state, cond);
        }
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The thermal alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(ThermalAlert alert, String cond)
    {
        Boolean active = activeAlertMap.put(alert.getId(), false);
        if (active != null && active) {
            as.raiseAlert(alert.newAlert(), AlertState.NOMINAL, cond);
        }
    }


    /**
     *  Enables an alert to be cleared.
     * 
     *  @param  alert  The alert
     *  @return  Action code
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }
    

    /**
     *  Checks a trim heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  RefrigException if the number is invalid or object is null
     */
    private TrimHeater getTrimHeater(int htr) throws RefrigException
    {
        if (htr < 0 || htr >= ThermalState.NUM_TRIM_HEATERS) {
            throw new RefrigException("Invalid trim heater number: " + htr);
        }
        TrimHeater heater = trimHeaters[htr];
        if (heater == null) {
            throw new RefrigException("Trim heater " + htr + " not defined");
        }
        return heater;
    }
    

    /**
     *  Checks an auxiliary heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  RefrigException if the number is invalid or object is null
     */
    private AuxHeater getAuxHeater(int htr) throws RefrigException
    {
        if (htr < 0 || htr >= ThermalState.NUM_AUX_HEATERS) {
            throw new RefrigException("Invalid auxiliary heater number: " + htr);
        }
        AuxHeater heater = auxHeaters[htr];
        if (heater == null) {
            throw new RefrigException("Auxiliary heater " + htr + " not defined");
        }
        return heater;
    }

}
