package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.ColdState;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the refrigeration compressor subsystem.
 *
 *  @author Owen Saxton
 */
public class RefrigMain extends Subsystem implements HasLifecycle, ClearAlertHandler {

    /**
     *  Constants.
     */
    private static ColdState[] coldStates = {ColdState.BOTH_OFF, ColdState.ONE_ON, ColdState.TWO_ON, ColdState.BOTH_ON};

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService as;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService ass;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, ColdCompressor> coldCompMap = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, CryoCompressor> cryoCompMap = new LinkedHashMap<>();

    private static final Logger LOG = Logger.getLogger(Compressor0.class.getName());
    private final Compressor[] coldComps = new Compressor[RefrigState.MAX_COLD_COMPRESSORS];
    private final Compressor[] cryoComps = new Compressor[RefrigState.MAX_CRYO_COMPRESSORS];
    private final RefrigState state = new RefrigState();


    /**
     *  Constructor.
     */
    public RefrigMain() {
        super("refrig", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        aps.setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the refrigeration state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("compressor-state", () -> updateRefrigState()).withPeriod(Duration.ofMillis(3000));
        apts.scheduleAgentPeriodicTask(pt);
        ass.registerState(ColdState.class, "Overall cold compressor state", this);
        ass.updateAgentState(ColdState.UNKNOWN);
    }


    /**
     *  Initializes the refrigeration subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a compressor subsystem.
        aps.setAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE, RefrigMain.class.getCanonicalName());

        // Initialize component references
        if (coldCompMap.isEmpty() && cryoCompMap.isEmpty()) {
            MonitorLogUtils.reportConfigError(LOG, name, "No compressors", "defined");
        }
        int nComp = coldCompMap.size();
        if (nComp > RefrigState.MAX_COLD_COMPRESSORS) {
            LOG.warning("Too many (" + nComp + ") cold compressors specified: using only the first "
                           + RefrigState.MAX_COLD_COMPRESSORS);
            List<String> keys = new ArrayList(coldCompMap.keySet());
            for (int j = RefrigState.MAX_COLD_COMPRESSORS; j < nComp; j++) {
                coldCompMap.remove(keys.get(j));
            }
        }
        nComp = 0;
        for (Compressor comp : coldCompMap.values()) {
            coldComps[nComp] = comp;
            comp.setIndex(nComp++);
        }
        state.setNumColdComps(nComp);
        nComp = cryoCompMap.size();
        if (nComp > RefrigState.MAX_CRYO_COMPRESSORS) {
            LOG.warning("Too many (" + nComp + ") cryo compressors specified: using only the first "
                           + RefrigState.MAX_CRYO_COMPRESSORS);
            List<String> keys = new ArrayList(cryoCompMap.keySet());
            for (int j = RefrigState.MAX_CRYO_COMPRESSORS; j < nComp; j++) {
                cryoCompMap.remove(keys.get(j));
            }
        }
        nComp = 0;
        for (Compressor comp : cryoCompMap.values()) {
            cryoComps[nComp] = comp;
            comp.setIndex(nComp++);
        }
        state.setNumCryoComps(nComp);
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Refrigeration subsystem started with " + coldCompMap.size() + " cold compressors and "
                    + cryoCompMap.size() + " cryo compressors");
    }


    /**
     *  Gets the state of the refrigeration module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the refrigeration state")
    public RefrigState getSystemState()
    {
        state.clearCompStates();
        state.setTickMillis(getTickPeriod());
        for (Compressor comp : coldCompMap.values()) {
            state.addColdState(comp.getState());
        }
        for (Compressor comp : cryoCompMap.values()) {
            state.addCryoState(comp.getState());
        }
        return state;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        state.clearCompStates();
        publishState();
    }


    /**
     *  Enables an alert to be cleared.
     * 
     *  @param  alert  The alert
     *  @return  Action code
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert)
    {
        String id = alert.getAlertId();
        for (String compName : coldCompMap.keySet()) {
            if (id.startsWith(compName)) {
                return coldCompMap.get(compName).canClearAlert(alert);
            }
        }
        for (String compName : cryoCompMap.keySet()) {
            if (id.startsWith(compName)) {
                return cryoCompMap.get(compName).canClearAlert(alert);
            }
        }
        return ClearAlertCode.UNKNOWN_ALERT;
    }


    /**
     *  Updates the refrigeration state.
     */
    private void updateRefrigState()
    {
        boolean changed = false;
        state.clearCompStates();
        int running = 0;
        for (Compressor comp : coldCompMap.values()) {
            comp.checkLimits();
            if (comp.updateState()) {
                changed = true;
                state.addColdState(comp.getState());
            }
            CompressorState compState = comp.getState().getCompressorState();
            if (compState == CompressorState.RUNNING || compState == CompressorState.WAITING) {
                running |= (1 << comp.getIndex());
            }
        }
        ass.updateAgentState(coldStates[running]);
        for (Compressor comp : cryoCompMap.values()) {
            comp.controlValves();
            comp.checkLimits();
            if (comp.updateState()) {
                changed = true;
                state.addCryoState(comp.getState());
            }
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Publishes the state of the refrigeration module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        state.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(RefrigState.KEY, state));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
