package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Line;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.RefrigState1;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the refrigeration compressor subsystem.
 *
 *  @author Owen Saxton
 */
public class Compressor0 implements HasLifecycle, Monitor.AlarmHandler {

    /**
     *  Constants.
     */
    private final static int
        EVENT_ID_MAIN_POWER = 0;

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Line> allLines = new HashMap<>();

    private String cmprPowerLine;
    private String heaterPowerLine;
    private String resetLine;

    private static final Logger LOG = Logger.getLogger(Compressor0.class.getName());
    private boolean coldStart, running;
    private Line cmprPowerLineL, heaterPowerLineL, resetLineL;
    private final RefrigState1 state = new RefrigState1();


    /**
     *  Initializes the refrigeration subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a compressor subsystem.
        subsys.setAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE, Compressor0.class.getCanonicalName());

        // General initialization
        state.setTickMillis(getTickPeriod());

        // Get cold start option
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");

        // Initialize component references
        if (cmprPowerLine != null) {
            cmprPowerLineL = allLines.get(cmprPowerLine);
        }
        if (cmprPowerLineL == null) {
            LOG.error("Compressor power line not specified or not defined");
        }
        if (heaterPowerLine != null) {
            heaterPowerLineL = allLines.get(heaterPowerLine);
        }
        if (heaterPowerLineL == null) {
            LOG.error("Heater power line not specified or not defined");
        }
        if (resetLine != null) {
            resetLineL = allLines.get(resetLine);
        }
        if (resetLineL == null) {
            LOG.error("Reset line not specified or not defined");
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Set power control lines' warm start option
        if (cmprPowerLineL != null) {
            cmprPowerLineL.setWarm(!coldStart);
        }
        if (heaterPowerLineL != null) {
            heaterPowerLineL.setWarm(!coldStart);
        }

        // Turn off the power control lines' warm start option
        if (cmprPowerLineL != null) {
            cmprPowerLineL.setWarm(false);
        }
        if (heaterPowerLineL != null) {
            heaterPowerLineL.setWarm(false);
        }

        // Make sure state and power control lines are consistent
        setCompPowerEnable(isCompPowerOn() ? 1 : 0);
        setHeaterPowerEnable(isHeaterPowerOn() ? 1 : 0);

        // Announce startup
        running = true;
        LOG.info("Refrigeration subsystem (" + subsys.getName() + ") started");
        //publishState();
    }


    /**
     *  Gets the state of the refrigeration module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the refrigeration state")
    public RefrigState1 getSystemState()
    {
        return state;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        state.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Handles alarm events.
     *
     *  @param  event  The event type
     *  @param  parm   The event parameter
     *  @param  cause  The alarm cause
     *  @param  name   The alarm name
     *  @return  Whether to process further as an alarm
     */
    @Override
    public boolean processAlarm(int event, int parm, String cause, String name)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
                if (state.getCmprPowerState(0) != RefrigState1.POWER_TRIPPED) {
                    state.setCmprPowerState(0, RefrigState1.POWER_TRIPPED);
                    setCompPowerEnable(0);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if (state.getCmprPowerState(0) == RefrigState1.POWER_TRIPPED) {
                    state.setCmprPowerState(0, RefrigState1.POWER_OFF);
                    publishState();
                }
            }
            break;

        default:

        }
        return false;
    }


    /**
     *  Sets the compressor power state.
     *
     *  @param  value    The enabled state value to set: 0 = off, not 0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set the main power enabled state")
    public void setCompPowerEnable(int value)
    {
        if (value != 0) {
            if (state.getCmprPowerState(0) != RefrigState1.POWER_TRIPPED) {
                state.setCmprPowerState(0, RefrigState1.POWER_ON);
            }
        }
        else {
            state.setCmprPowerState(0, RefrigState1.POWER_OFF);
        }
        setCompPower();
        publishState();
    }


    /**
     *  Sets the heater power state.
     *
     *  @param  value  The heater power state to set: 0 = off; not 0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set the heater power enabled state")
    public void setHeaterPowerEnable(int value)
    {
        if (value != 0) {
            state.setHeaterPowerState(RefrigState1.POWER_ON);
        }
        else {
            state.setHeaterPowerState(RefrigState1.POWER_OFF);
        }
        setHeaterPower();
        publishState();
    }


    /**
     *  Resets the latches.
     */
    @Command(type=CommandType.ACTION, description="Reset the latches")
    public void reset()
    {
        if (resetLineL != null) {
            resetLineL.set(true);
            try {
                Thread.sleep(200);
            }
            catch (InterruptedException e) {
            }
            resetLineL.set(false);
        }
    }


    /**
     *  Publishes the state of the refrigeration module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(RefrigState1.KEY, state));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Sets the compressor power on or off.
     */
    private void setCompPower()
    {
        if (cmprPowerLineL != null) {
            cmprPowerLineL.set(state.getCmprPowerState(0) == RefrigState1.POWER_ON);
        }
    }
            

    /**
     *  Sets the heater power on or off.
     */
    private void setHeaterPower()
    {
        if (heaterPowerLineL != null) {
            heaterPowerLineL.set(state.getHeaterPowerState() == RefrigState1.POWER_ON);
        }
    }
    

    /**
     *  Gets the hardware compressor power state
     */
    private boolean isCompPowerOn()
    {
        return cmprPowerLineL != null && cmprPowerLineL.isSet();
    }
            

    /**
     *  Gets the hardware heater power state.
     */
    private boolean isHeaterPowerOn()
    {
        return heaterPowerLineL != null && heaterPowerLineL.isSet();
    }

}
