package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the refrigeration thermal control subsystem.
 *
 *  @author Owen Saxton
 */
public class ThermalMain implements HasLifecycle, Monitor.AlarmHandler {

    /**
     *  Inner class for holding heater parameters.
     */
    static class Heater {

        TempControl tempCtrl;
        PowerDevice powerDevc;
        int powerChan;
        Double startPower = 0.0;

        Heater(TempControl tempCtrl, PowerDevice powerDevc, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Constants.
     */
    public static final int
        ALARM_COLD_HEATER = 0,
        ALARM_CRYO_HEATER = 1;
    private static final int
        POWER_SET_INTVL = 5000;

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, TempControl> tempControls = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, Channel> allChannels = new HashMap<>();

    private static final String REFRIG = "Refrig";
    private static final Logger LOG = Logger.getLogger(ThermalMain.class.getName());

    private String[] alarmDsabChans = new String[0];  // Names of the channels whose alarms can be disabled
    private String coldTempCtrl;          // Name of the cold plate temperature control object
    private String cryoTempCtrl;          // Name of the cryo plate temperature control object

    private final List<Channel> alarmDsabChansL = new ArrayList<>();

    private final Heater[] heaters = new Heater[ThermalState.NUM_HEATERS];
    private final ThermalState state = new ThermalState();


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the refrigeration state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("heater-power", () -> setHeaterPower()).withPeriod(Duration.ofMillis(POWER_SET_INTVL));
        pts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the thermal subsystem.
     */
    @Override
    public void postInit()
    {
        //Set a property to define that this Agent is a thermal subsystem.
        subsys.setAgentProperty(RefrigAgentProperties.THERMAL_TYPE, ThermalMain.class.getCanonicalName());

        // General initialization
        state.setTickMillis(getTickPeriod());

        // Initialize component references
        for (String cName : alarmDsabChans) {
            Channel chan = allChannels.get(cName);
            if (chan != null) {
                alarmDsabChansL.add(chan);
            }
        }
        if (alarmDsabChansL.size() != alarmDsabChans.length) {
            LOG.error("Some alarm disable channels are invalid");
        }

        TempControl ctrl = tempControls.get(coldTempCtrl);
        if (ctrl == null) {
            LOG.error("No coldplate temperature controller (or heater control) specified");
            state.setHeaterPowerState(ThermalState.HEATER_COLD, ThermalState.POWER_NONE);   // Indicate no heater power
            state.setHeaterControlState(ThermalState.HEATER_COLD, ThermalState.CONTROL_NONE);   // Indicate no heater control
        }
        else {
            heaters[ThermalState.HEATER_COLD] = new Heater(ctrl, ctrl.getPowerDevice(), ctrl.getPowerChannel());
        }
        ctrl = tempControls.get(cryoTempCtrl);
        if (ctrl == null) {
            LOG.error("No cryoplate temperature controller (or heater control) specified");
            state.setHeaterPowerState(ThermalState.HEATER_CRYO, ThermalState.POWER_NONE);   // Indicate no heater power
            state.setHeaterControlState(ThermalState.HEATER_CRYO, ThermalState.CONTROL_NONE);   // Indicate no heater control
        }
        else {
            heaters[ThermalState.HEATER_CRYO] = new Heater(ctrl, ctrl.getPowerDevice(), ctrl.getPowerChannel());
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Thermal control system (" + name + ") started");
        //publishState();
    }


    /**
     *  Gets the state of the thermal control system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=CommandType.QUERY, description="Get the thermal control state")
    public ThermalState getSystemState()
    {
        return state;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        state.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Handles alarm events.
     *
     *  @param  event  The event type
     *  @param  parm   The event parameter
     *  @param  cause  The alarm cause
     *  @param  name   The alarm name
     *  @return  Whether to process as an alarm
     */
    @Override
    public boolean processAlarm(int event, int parm, String cause, String name)
    {
        if (parm == ALARM_COLD_HEATER || parm == ALARM_CRYO_HEATER) {
            int heater = parm == ALARM_COLD_HEATER ? ThermalState.HEATER_COLD : ThermalState.HEATER_CRYO;
            if (event == Alarm.EVENT_TRIP) {
                if (state.getHeaterPowerState(heater) != ThermalState.POWER_TRIPPED) {
                    state.setHeaterPowerState(heater, ThermalState.POWER_TRIPPED);
                    setHeaterPowerSwitch(heater);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if (state.getHeaterPowerState(heater) == ThermalState.POWER_TRIPPED) {
                    state.setHeaterPowerState(heater, ThermalState.POWER_OFF);
                    publishState();
                }
            }
        }
        return false;
    }


    /**
     *  Sets the heater power enable state.
     *
     *  @param  heater  The heater number
     *  @param  value   The enabled state value to set: 0 = off, ~0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set the heater power enabled state")
    public void setHeaterPowerEnable(int heater, int value)
    {
        int oldState = state.getHeaterPowerState(heater);
        if (oldState != ThermalState.POWER_NONE) {
            int newState = (oldState == ThermalState.POWER_TRIPPED) ? oldState
                             : (value == 0) ? ThermalState.POWER_OFF : ThermalState.POWER_ON;
            if (newState != oldState) {
                state.setHeaterPowerState(heater, newState);
                setHeaterPowerSwitch(heater);
            }
        }
        publishState();
    }


    /**
     *  Sets the heater control state.
     *
     *  @param  heater  The heater number
     *  @param  value  The heater power state to set: 0 = off; &gt; 0 = manual;
     *                 &lt; 0 = automatic (temp loop).
     */
    @Command(type=CommandType.ACTION, description="Set the heater control state")
    public void setHeaterControl(int heater, int value)
    {
        int oldState = state.getHeaterControlState(heater);
        if (oldState != ThermalState.CONTROL_NONE) {
            Heater htr = heaters[heater];
            int newState = (value == 0) ? ThermalState.CONTROL_OFF
                            : (value > 0) ? ThermalState.CONTROL_ON
                            : htr != null ? ThermalState.CONTROL_AUTO
                            : ThermalState.CONTROL_ON;
            if (newState != oldState) {
                state.setHeaterControlState(heater, newState);
                if (newState == ThermalState.CONTROL_AUTO) {
                    htr.tempCtrl.setTemp(state.getPlateTemp(heater));
                    startTempControl(heater);
                }
                else {
                    stopTempControl(heater);
                    setHeaterPower(heater);
                }
            }
        }
        publishState();
    }


    /**
     *  Sets the alarm disabled state on or off.
     *
     *  @param  value  The alarm disabled state value to set: 0 = off; ~0 = on.
     */
    @Command(type=CommandType.ACTION, description="Set the load trip enabled state")
    public void setAlarmDisabled(int value)
    {
        boolean disable = (value != 0);
        state.setAlarmDisabled(disable);
        for (Channel chan : alarmDsabChansL) {
            chan.enableAlarm(false, !disable);
        }
        publishState();
    }


    /**
     *  Sets the heater power value.
     *
     *  @param  heater  The heater number
     *  @param  value  The load power set point.
     */
    @Command(type=CommandType.ACTION, description="Set the heater power set point")
    public void setHeaterPower(int heater, double value)
    {
        Heater htr = heaters[heater];
        if (htr != null) {
            state.setHeaterPower(heater, value);
            if (htr.startPower != null) {
                htr.startPower = value;
            }
            setHeaterPower(heater);
        }
        publishState();
    }


    /**
     *  Sets the plate temperature value.
     *
     *  @param  heater  The heater number
     *  @param  value  The plate temperature set point.
     */
    @Command(type=CommandType.ACTION, description="Set the load temperature set point")
    public void setPlateTemp(int heater, double value)
    {
        Heater htr = heaters[heater];
        if (htr != null) {
            state.setPlateTemp(heater, value);
            htr.tempCtrl.setTemp(value);
        }
        publishState();
    }


    /**
     *  Saves a named refrigeration configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        sce.saveChangesForCategoriesAs(REFRIG + ":" + name);
    }


    /**
     *  Loads a named refrigeration configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Load a named configuration")
    public void loadNamedConfig(String name) throws IOException
    {
        sce.loadCategories(REFRIG + ":" + name);
    }

  
    /**
     *  Publishes the state of the refrigeration module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ThermalState.KEY, state));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Sets a heater power on or off, according to the state.
     */
    private void setHeaterPowerSwitch(int heater)
    {
        Heater htr = heaters[heater];
        if (htr != null) {
            boolean on = state.getHeaterPowerState(heater) == ThermalState.POWER_ON;
            if (on) {
                htr.powerDevc.enableOutput(htr.powerChan, true);
                if (state.getHeaterControlState(heater) == ThermalState.CONTROL_AUTO) {
                    startTempControl(heater);
                }
            }
            else {
                stopTempControl(heater);
                htr.powerDevc.enableOutput(htr.powerChan, false);
            }
        }
    }
            

    /**
     *  Sets the power for all heaters.
     */
    private void setHeaterPower()
    {
        setHeaterPower(ThermalState.HEATER_COLD);
        setHeaterPower(ThermalState.HEATER_CRYO);
    }


    /**
     *  Sets the power for a heater.
     */
    private void setHeaterPower(int heater)
    {
        Heater htr = heaters[heater];
        if (htr == null || !htr.powerDevc.isOnline()) return;
        if (state.getHeaterPowerState(heater) != ThermalState.POWER_ON) return;
        htr.powerDevc.enableOutput(htr.powerChan, state.getHeaterControlState(heater) != ThermalState.CONTROL_OFF);
        if (state.getHeaterControlState(heater) == ThermalState.CONTROL_ON) {
            double loadPower = state.getHeaterPower(heater);
            htr.powerDevc.setPower(htr.powerChan, loadPower);
            htr.startPower = loadPower;
        }
    }
    

    /**
     *  Starts a temperature control loop.
     */
    private void startTempControl(int heater)
    {
        Heater htr = heaters[heater];
        if (htr != null) {
            if (htr.startPower == null) {
                htr.tempCtrl.restart();
            }
            else {
                htr.tempCtrl.start(htr.startPower);
                htr.startPower = null;
            }
        }
    }
    

    /**
     *  Stops a temperature control loop.
     */
    private void stopTempControl(int heater)
    {
        Heater htr = heaters[heater];
        if (htr != null) {
            htr.tempCtrl.stop();
        }
    }

}
