package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the refrigeration compressor subsystem.
 *
 *  @author Owen Saxton
 */
public class RefrigMain extends Subsystem implements HasLifecycle, Monitor.AlarmHandler {

    /**
     *  Constants.
     */
    private final static int
        EVENT_ID_MAIN_POWER = 0;

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService as;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService ass;
    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, ColdCompressor> coldComps = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, CryoCompressor> cryoComps = new LinkedHashMap<>();

    private static final Logger LOG = Logger.getLogger(Compressor0.class.getName());
    private final RefrigState state = new RefrigState();


    /**
     *  Constructor.
     */
    public RefrigMain() {
        super("refrig", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("compressor-state",
                                   () -> updateRefrigState()).withPeriod(Duration.ofMillis(3000));
        apts.scheduleAgentPeriodicTask(pt);
        ass.registerState(CompressorState.class, "Compressor state", this);
        ass.updateAgentState(CompressorState.OFFLINE);
    }


    /**
     *  Initializes the refrigeration subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a compressor subsystem.
        subsys.setAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE, RefrigMain.class.getCanonicalName());

        // Initialize component references
        if (coldComps.isEmpty() && cryoComps.isEmpty()) {
            MonitorLogUtils.reportConfigError(LOG, name, "No compressors", "defined");
        }
        int nComp = coldComps.size();
        if (nComp > RefrigState.MAX_COLD_COMPRESSORS) {
            LOG.warning("Too many (" + nComp + ") cold compressors specified: using only the first "
                           + RefrigState.MAX_COLD_COMPRESSORS);
            List<String> keys = new ArrayList(coldComps.keySet());
            for (int j = RefrigState.MAX_COLD_COMPRESSORS; j < nComp; j++) {
                coldComps.remove(keys.get(j));
            }
        }
        nComp = 0;
        for (Compressor comp : coldComps.values()) {
            comp.setIndex(nComp++);
        }
        state.setNumColdComps(nComp);
        nComp = cryoComps.size();
        if (nComp > RefrigState.MAX_CRYO_COMPRESSORS) {
            LOG.warning("Too many (" + nComp + ") cryo compressors specified: using only the first "
                           + RefrigState.MAX_CRYO_COMPRESSORS);
            List<String> keys = new ArrayList(cryoComps.keySet());
            for (int j = RefrigState.MAX_CRYO_COMPRESSORS; j < nComp; j++) {
                cryoComps.remove(keys.get(j));
            }
        }
        nComp = 0;
        for (Compressor comp : cryoComps.values()) {
            comp.setIndex(nComp++);
        }
        state.setNumCryoComps(nComp);
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Refrigeration subsystem started with " + coldComps.size() + " cold compressors and "
                    + cryoComps.size() + " cryo compressors");
    }


    /**
     *  Gets the state of the refrigeration module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the refrigeration state")
    public RefrigState getSystemState()
    {
        state.clearCompStates();
        state.setTickMillis(getTickPeriod());
        for (Compressor comp : coldComps.values()) {
            state.addColdState(comp.getState());
        }
        for (Compressor comp : cryoComps.values()) {
            state.addCryoState(comp.getState());
        }
        return state;
    }    


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(int value)
    {
        setTickPeriod(value);
        state.clearCompStates();
        publishState();
    }


    /**
     *  Handles alarm events.
     *
     *  @param  event  The event type
     *  @param  parm   The event parameter
     *  @param  cause  The alarm cause
     *  @param  name   The alarm name
     *  @return  Whether to process further as an alarm
     */
    @Override
    public boolean processAlarm(int event, int parm, String cause, String name)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
            }
            else if (event == Alarm.EVENT_RESET) {
            }
            break;

        default:

        }
        return false;
    }


    /**
     *  Updates the refrigeration state.
     */
    private void updateRefrigState()
    {
        boolean changed = false;
        state.clearCompStates();
        for (Compressor comp : coldComps.values()) {
            if (comp.updateState()) {
                changed = true;
                state.addColdState(comp.getState());
            }
        }
        for (Compressor comp : cryoComps.values()) {
            if (comp.updateState()) {
                changed = true;
                state.addCryoState(comp.getState());
            }
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Publishes the state of the refrigeration module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        state.setTickMillis(getTickPeriod());
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(RefrigState.KEY, state));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
