package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.auxelex.HeaterPS;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a SLAC heater power supply.
 *
 *  @author Owen Saxton
 */
public class HeaterPsDevice extends PowerDevice {

    /**
     *  Constants.
     */
    public static final int
        CHANSET_COLD     = 0,
        CHANSET_CRYO     = 1,

        TYPE_VOLTAGE     = 0,
        TYPE_CURRENT     = 1,
        TYPE_POWER       = 2,
        TYPE_TEMP        = 3,
        TYPE_MAIN_VOLTS  = 4,
        TYPE_MAIN_CURR   = 5,
        TYPE_MAIN_POWER  = 6,
        TYPE_MAIN_TEMP   = 7,
        TYPE_MAIN_STATUS = 8;

    private static final int
        N_HW_CHANS = HeaterPS.NUM_HEATERS;
    private static final String
        NODE   = "node";

    /**
     *  Private lookup maps.
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("VOLTAGE", TYPE_VOLTAGE);
        typeMap.put("CURRENT", TYPE_CURRENT);
        typeMap.put("POWER", TYPE_POWER);
        typeMap.put("TEMP", TYPE_TEMP);
        typeMap.put("MAINVOLTS", TYPE_MAIN_VOLTS);
        typeMap.put("MAINCURR", TYPE_MAIN_CURR);
        typeMap.put("MAINPOWER", TYPE_MAIN_POWER);
        typeMap.put("MAINTEMP", TYPE_MAIN_TEMP);
        typeMap.put("MAINSTAT", TYPE_MAIN_STATUS);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category="Refrig", isFinal=true)
    private Integer node;      // Heater PS node number
    private int[] cryoChannels = {0, 1, 2, 3, 4, 5};
    private int[] coldChannels = {6, 7, 8, 9, 10, 11};
    private double heaterOhms = 12.5;

    private static final Logger LOG = Logger.getLogger(HeaterPsDevice.class.getName());
    private final HeaterPS htr = new HeaterPS();  // Associated heater PS object
    private final Set<Integer> coldChannelSet = new HashSet<>();
    private final Set<Integer> cryoChannelSet = new HashSet<>();
    private final Set<Integer>[] channelSets = new Set[2];
    private final double[] currents = new double[N_HW_CHANS];  // Read currents 


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "node", "is missing");
        }
        for (int chan : coldChannels) {
            if (chan < 0 || chan >= N_HW_CHANS) {
                MonitorLogUtils.reportConfigError(LOG, getName(), "coldChannel " + chan, " is invalid");
            }
            if (!coldChannelSet.add(chan)) {
                MonitorLogUtils.reportConfigError(LOG, getName(), "coldChannel " + chan, " is a duplicate");
            }
        }
        for (int chan : cryoChannels) {
            if (chan < 0 || chan >= N_HW_CHANS) {
                MonitorLogUtils.reportConfigError(LOG, getName(), "cryoChannel " + chan, " is invalid");
            }
            if (coldChannelSet.contains(chan)) {
                MonitorLogUtils.reportConfigError(LOG, getName(), "cryoChannel " + chan, " is a coldChannel");
            }
            if (!cryoChannelSet.add(chan)) {
                MonitorLogUtils.reportConfigError(LOG, getName(), "cryoChannel " + chan, " is a duplicate");
            }
        }
        channelSets[CHANSET_COLD] = coldChannelSet;
        channelSets[CHANSET_CRYO] = cryoChannelSet;
        fullName = "Heater power supply (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            htr.open(node);
            htr.setMainPowerOn(true);
            setOnline(true);
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            if (htr != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            htr.close();
        }
        catch (DriverException e) {
        }
        for (int j = 0; j < N_HW_CHANS; j++) {
            currents[j] = 0;
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string (not used)
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(LOG, name, "type", type);
        }
        if (iType == TYPE_VOLTAGE || iType == TYPE_CURRENT || iType == TYPE_POWER) {
            if (hwChan < 0 || hwChan >= N_HW_CHANS) {
                MonitorLogUtils.reportError(LOG, name, "HW channel", hwChan);
            }
        }
        return new int[]{iType, 0};
    }


    /**
     *  Reads all heater channels as a group.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            for (int chan = 0; chan < N_HW_CHANS; chan++) {
                currents[chan] = htr.readCurrent(chan);
            }
        }
        catch (DriverException e) {
            LOG.error("Error reading currents from " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (online) {
            try {
                switch (type) {
                case TYPE_VOLTAGE:
                    value = heaterOhms * currents[hwChan];
                    break;
                case TYPE_CURRENT:
                    value = currents[hwChan];
                    break;
                case TYPE_POWER:
                    value = heaterOhms * currents[hwChan] * currents[hwChan];
                    break;
                case TYPE_TEMP:
                    value = htr.readBoardTemperature();
                    break;
                case TYPE_MAIN_VOLTS:
                    value = htr.readMainVoltage();
                    break;
                case TYPE_MAIN_CURR:
                    value = htr.readMainCurrent();
                    break;
                case TYPE_MAIN_POWER:
                    value = htr.readMainVoltage() * htr.readMainCurrent();
                    break;
                case TYPE_MAIN_TEMP:
                    value = htr.readMainTemperature();
                    break;
                case TYPE_MAIN_STATUS:
                    value = htr.readMainStatus();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.error("Error reading value from " + fullName + ": " + e);
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Sets the output state of a set of channels.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @param  value    The output state to set, true or false
     */
    @Override
    public void setOutput(int chanSet, boolean value)
    {
        if (!online) return;
        try {
            for (int chan : channelSets[chanSet]) {
                htr.setOutput(chan, value);
            }
        }
        catch (DriverException e) {
            LOG.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Gets the output state.
     *
     *  @param  chanSet   The hardware channel set (cold or cryo)
     *  @return  The output state (true if any channel has output true)
     */
    @Override
    public boolean getOutput(int chanSet)
    {
        if (!online) return false;
        try {
            boolean state = false;
            for (int chan : channelSets[chanSet]) {
                state |= htr.getOutput(chan);
            }
            return state;
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return false;
        }
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @param  value    The voltage to set
     */
    @Command (description="Set the voltage", type=CommandType.ACTION)
    @Override
    public void setVoltage(int chanSet, double value)
    {
        if (!online) return;
        try {
            for (int chan : channelSets[chanSet]) {
                htr.setVoltage(chan, value);
            }
        }
        catch (DriverException e) {
            LOG.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @return  The read voltage (average of all channels in the set)
     */
    @Override
    public double readVoltage(int chanSet)
    {
        if (!online) return -1;
        try {
            double current = 0;
            for (int chan : channelSets[chanSet]) {
                current += htr.readCurrent(chan);
            }
            return heaterOhms * current / channelSets[chanSet].size();
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }


    /**
     *  Sets the current.
     *
     *  This is a no-op.
     * 
     *  @param  chan   The hardware channel (not used)
     *  @param  value  The current to set
     */
    @Override
    public void setCurrent(int chan, double value)
    {
    }


    /**
     *  Reads the current.
     *
     *  @param  chanSet  The hardware channel set (cold or cryo)
     *  @return  The read current (the sum over all channels in the set)
     */
    @Override
    public double readCurrent(int chanSet)
    {
        if (!online) return -1;
        try {
            double current = 0;
            for (int chan : channelSets[chanSet]) {
                current += htr.readCurrent(chan);
            }
            return current;
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }

}
