package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.sorensen.Dcs;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Sorensen DCS series power supply.
 *
 *  @author Owen Saxton
 */
public class DcsDevice extends PowerDevice {

    /**
     *  Constants.
     */
    static final int
        CHAN_POWER   = 0,
        CHAN_VOLTAGE = 1,
        CHAN_CURRENT = 2,
        N_HW_CHANS   = 3;

    /**
     *  Private lookup maps.
     */
    private static final Map<String, Dcs.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Dcs.ConnType.SERIAL);
        cTypeMap.put("FTDI",   Dcs.ConnType.FTDI);
    }

    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POWER", Channel.TYPE_POWER);
    }


    /**
     *  Data fields.
     */
    private String connType;          // DCS connection type string
    private String ident;             // Serial device name or FTDI serial number
    private double maxCurrent = 3.0;  // Maximum current limit

    private static final Logger LOG = Logger.getLogger(DcsDevice.class.getName());
    private Dcs.ConnType connTypeC;   // DCS connection type
    private Dcs dcs;                  // Associated DCS object
    private final double[] values = new double[N_HW_CHANS];  // Read values 


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        super.configure(mon);
        if (connType == null) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "connType", "is missing");
        }
        connTypeC = cTypeMap.get(connType);
        if (connTypeC == null) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "connType", "is invalid: " + connType);
        }
        if (ident == null) {
            MonitorLogUtils.reportConfigError(LOG, getName(), "ident", "is missing");
        }
        fullName = "Sorensen DCS module (" + ident + ")";
        disabled = true;
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        if (connTypeC == null) return;
        try {
            if (!inited || dcs == null) {
                dcs = new Dcs();
            }
            dcs.open(connTypeC, ident);
            dcs.setCurrent(maxCurrent);
            setOnline(true);
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            if (dcs != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            dcs.close();
        }
        catch (DriverException e) {
        }
        for (int j = 0; j < N_HW_CHANS; j++) {
            values[j] = 0;
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_HW_CHANS) {
            MonitorLogUtils.reportError(LOG, name, "HW channel", hwChan);
        }
        Integer iSubtype = 0, iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(LOG, name, "type", type);
        }

        return new int[]{iType, iSubtype};
    }


    /**
     *  Reads all channels as a group.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            values[CHAN_VOLTAGE] = dcs.readVoltage();
            values[CHAN_CURRENT] = dcs.readCurrent();
            values[CHAN_POWER] = values[CHAN_VOLTAGE] * values[CHAN_CURRENT];
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return online ? values[hwChan] : super.readChannel(hwChan, type);
    }


    /**
     *  Sets the output state.
     *
     *  @param  chan   The hardware channel (not used)
     *  @param  value  The output state to set, true or false
     */
    @Override
    public void setOutput(int chan, boolean value)
    {
        if (!testOnline()) return;
        try {
            dcs.setOutput(value);
        }
        catch (DriverException e) {
            LOG.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Gets the output state.
     *
     *  @param  chan   The hardware channel (not used)
     *  @return  The output state
     */
    @Override
    public boolean getOutput(int chan)
    {
        if (!testOnline()) return false;
        try {
            return dcs.getOutput();
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return false;
        }
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chan   The hardware channel (not used)
     *  @param  value  The voltage to set
     */
    @Override
    public void setVoltage(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            dcs.setVoltage(value);
        }
        catch (DriverException e) {
            LOG.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chan   The hardware channel (not used)
     *  @return  The read voltage
     */
    @Override
    public double readVoltage(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return dcs.readVoltage();
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }


    /**
     *  Sets the current.
     *
     *  @param  chan   The hardware channel (not used)
     *  @param  value  The current to set
     */
    @Override
    public void setCurrent(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            dcs.setCurrent(value);
        }
        catch (DriverException e) {
            LOG.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


    /**
     *  Reads the current.
     *
     *  @param  chan   The hardware channel (not used)
     *  @return  The read current
     */
    @Override
    public double readCurrent(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return dcs.readCurrent();
        }
        catch (DriverException e) {
            LOG.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }

}
