package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Controls a refrigeration compressor.
 *
 *  @author Owen Saxton
 */
public class Compressor implements HasLifecycle, Monitor.AlarmHandler {

    public static interface SwitchDevice {

        public void setSwitchOn(int chan, boolean on) throws RefrigException;

        public Boolean isSwitchOn(int chan);

    }

    /**
     *  Constants.
     */
    private final static int
        EVENT_ID_MAIN_POWER = 0;

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService as;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService ass;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private CompPlutoDevice plutoDevc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private CompMaq20Device maq20Devc;

    private static final Logger LOG = Logger.getLogger(Compressor.class.getName());
    private final CompState state;
    private final SwitchDevice[] switchDevices;
    private final int[] switchChannels;
    private final int type;


    /**
     *  Constructor.
     *
     *  @param  state  The compressor state object
     */
    public Compressor(CompState state)
    {
        this.state = state;
        type = state.getType();
        switchDevices = new SwitchDevice[CompState.NUM_SWITCHES];
        switchChannels = new int[CompState.NUM_SWITCHES];
    }


    /**
     *  Initializes the compressor control.
     */
    @Override
    public void postInit()
    {
        if (plutoDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "plutoDevc", "not specified");
        }
        if (maq20Devc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "maq20Devc", "not specified");
        }
        if (type == CompState.TYPE_COLD) {
            switchDevices[CompState.SW_ENABLE] = plutoDevc;
            switchChannels[CompState.SW_ENABLE] = CompPlutoDevice.SW_ENABLE;
            switchDevices[CompState.SW_LIGHTS] = plutoDevc;
            switchChannels[CompState.SW_LIGHTS] = CompPlutoDevice.SW_LIGHTS;
        }
        else {
            switchDevices[CompState.SW_ENABLE] = plutoDevc;
            switchChannels[CompState.SW_ENABLE] = CompPlutoDevice.SW_ENABLE;
            switchDevices[CompState.SW_LIGHTS] = plutoDevc;
            switchChannels[CompState.SW_LIGHTS] = CompPlutoDevice.SW_LIGHTS;
            switchDevices[CompState.SW_HEATER] = plutoDevc;
            switchChannels[CompState.SW_HEATER] = CompPlutoDevice.SW_HEATER;
            switchDevices[CompState.SW_NORMAL_OP_VALVE] = maq20Devc;
            switchChannels[CompState.SW_NORMAL_OP_VALVE] = CompMaq20Device.SW_NORMAL_OP_VALVE;
            switchDevices[CompState.SW_COOLANT_VALVE] = maq20Devc;
            switchChannels[CompState.SW_COOLANT_VALVE] = CompMaq20Device.SW_COOLANT_VALVE;
            switchDevices[CompState.SW_EVAC_VALVE] = maq20Devc;
            switchChannels[CompState.SW_EVAC_VALVE] = CompMaq20Device.SW_EVAC_VALVE;
            switchDevices[CompState.SW_SURGE_HEATER] = maq20Devc;
            switchChannels[CompState.SW_SURGE_HEATER] = CompMaq20Device.SW_SURGE_HEATER;
        }
        state.setName(name);
        plutoDevc.setType(type);
        maq20Devc.setType(type);
    }


    /**
     *  Sets the compressor index.
     *
     *  @param  index  The index value
     */
    public void setIndex(int index)
    {
        state.setIndex(index);
    }


    /**
     *  Gets the compressor index.
     *
     *  @return  The index value
     */
    public int getIndex()
    {
        return state.getIndex();
    }


    /**
     *  Turns a switch on or off..
     *
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a compressor's switch state")
    public void setSwitchOn(@Argument(description="Switch number") int sw,
                            @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        if (!state.getValidSwitches().contains(sw)) {
            throw new RefrigException("Invalid switch number: " + sw);
        }
        switchDevices[sw].setSwitchOn(switchChannels[sw], on);
    }


    /**
     *  Resets the latches.
     */
    @Command(type=CommandType.ACTION, description="Reset a compressor's latches")
    public void resetLatches()
    {
        plutoDevc.resetLatches();
    }


    /**
     *  Updates the compressor state.
     *
     *  @return  Whether any value changed
     */
    public boolean updateState()
    {
        boolean changed = false;
        CompressorState compState = null;
        for (int sw : state.getValidSwitches()) {
            Boolean on = switchDevices[sw].isSwitchOn(switchChannels[sw]);
            SwitchState swState = on == null ? SwitchState.OFFLINE : on ? SwitchState.ON : SwitchState.OFF;
            if (swState != state.getSwitchState(sw)) {
                state.setSwitchState(sw, swState);
                changed = true;
            }
        }
        for (int cond : state.getValidLatches()) {
            LatchState latchState = plutoDevc.getLatchState(cond);
            if (latchState != state.getLatchState(cond)) {
                state.setLatchState(cond, latchState);
                changed = true;
            }
        }
        for (int cond : state.getValidConditions()) {
            ConditionState condState = plutoDevc.getConditionState(cond);
            if (condState == ConditionState.OFF) {
                compState = CompressorState.OFFLINE;
            }
            if (condState != state.getConditionState(cond)) {
                state.setConditionState(cond, condState);
                changed = true;
            }
        }
        if (compState == null) {
            compState = state.getConditionState(CompState.COND_CMP_POWERED) == ConditionState.YES ? CompressorState.RUNNING :
                        state.getConditionState(CompState.COND_CMP_WAITING) == ConditionState.YES ? CompressorState.WAITING :
                        state.getConditionState(CompState.COND_LATCHES_CLEAR) == ConditionState.NO ? CompressorState.HW_DSAB :
                        CompressorState.STOPPED;
        }
        if (compState != state.getCompressorState()) {
            state.setCompressorState(compState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Gets the compressor state.
     *
     *  @return  The compressor state
     */
    public CompState getState()
    {
        return state;
    }    


    /**
     *  Handles alarm events.
     *
     *  @param  event  The event type
     *  @param  parm   The event parameter
     *  @param  cause  The alarm cause
     *  @param  name   The alarm name
     *  @return  Whether to process further as an alarm
     */
    @Override
    public boolean processAlarm(int event, int parm, String cause, String name)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
            }
            else if (event == Alarm.EVENT_RESET) {
            }
            break;

        default:

        }
        return false;
    }
            
}
