package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.MonitorCommands;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ********************************************************************
 *
 *  Implements the refrigeration subscale camera modular subsystem.
 *
 *  @author Owen Saxton
 *
 ********************************************************************
 */
public class Subscale extends Module implements HardwareController, Monitor.AlarmHandler {

   /**
    *  Constants.
    */
    private static final int
        TICK_INTERVAL = 10000,
        POWER_SET_INTERVAL = 5000,
        ALARM_HEATER = RefrigState.MAX_COMPRESSORS;

   /**
    *  Data fields.
    */
    private static final String REFRIG = "Refrig";
    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.refrig");

    private int monMillis = 1000;     // Monitoring period
    private int checkMillis = 0;      // Device check period (0 = default)
    private String loadPowerDevc;     // Name of the device controlling the load power
    private String[][] cmprPowerLines = new String[0][0];  // Names of the lines controlling the compressor power
    private String[] loadPowerLines = new String[0];  // Names of the lines controlling the load power
    private String[] alarmDsabChans = new String[0];  // Names of the channels whose alarms can be disabled
    private String[] loadTempChans = new String[0];   // Names of the load temperatures controlling the feedback
    private String tempCtrl;          // Name of the temperature control onject

    private boolean coldStart, running;
    private final List<List<Line>> cmprPowerLinesL = new ArrayList<>();
    private final List<Line> loadPowerLinesL = new ArrayList<>();
    private final List<Channel> alarmDsabChansL = new ArrayList<>();
    private final List<Channel> loadTempChansL = new ArrayList<>();

    private ConfigurableSubsystem subsys;
    private PowerDevice loadPowerDevcD;
    private TempControl tempCtrlC;
    private Monitor mon;
    private final RefrigState state = new RefrigState();
    private Double startPower = 0.0;


   /**
    *  Initializes the refrigeration subsystem.
    */
    @Override
    public void initModule()
    {
        /*
        *  General initialization
        */
        setName("main");
        if (getTickMillis() <= 0) {
            setTickMillis(TICK_INTERVAL);
        }
        state.setTickMillis(getTickMillis());
        subsys = (ConfigurableSubsystem)getSubsystem();

        /*
        *  Get cold start option
        */
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");

        /*
        *  Initialize component references
        */
        loadPowerDevcD = (PowerDevice)getComponent(loadPowerDevc,
                                                   PowerDevice.class);
        for (String[] chans : cmprPowerLines) {
            List<Line> lines = new ArrayList<>();
            cmprPowerLinesL.add(lines);
            for (String cName : chans) {
                Object cmpt = getComponent(cName, Line.class);
                if (cmpt != null) {
                    lines.add((Line)cmpt);
                }
            }
        }
        state.setNumCmprs(Math.min(RefrigState.MAX_COMPRESSORS, cmprPowerLinesL.size()));
        for (String cName : loadPowerLines) {
            Object cmpt = getComponent(cName, Line.class);
            if (cmpt != null) {
                loadPowerLinesL.add((Line)cmpt);
            }
        }
        for (String cName : alarmDsabChans) {
            Object cmpt = getComponent(cName, Channel.class);
            if (cmpt != null) {
                alarmDsabChansL.add((Channel)cmpt);
            }
        }
        for (String cName : loadTempChans) {
            Object cmpt = getComponent(cName, Channel.class);
            if (cmpt != null) {
                loadTempChansL.add((Channel)cmpt);
            }
        }
        tempCtrlC = (TempControl)getComponent(tempCtrl, TempControl.class);

        if (cmprPowerLinesL.isEmpty()) {
            sLog.error("No valid compressors specified");
        }
        else if (state.getNumCmprs() != cmprPowerLines.length) {
            sLog.error("Some compressors are invalid");
        }
        for (int j = 0; j < state.getNumCmprs(); j++) {
            List<Line> lines = cmprPowerLinesL.get(j);
            if (lines.size() != cmprPowerLines[j].length) {
                sLog.error("Some (or all) control lines are invalid for compressor " + j);
            }
            else if (lines.isEmpty()) {
                sLog.error("No control lines specified for compressor " + j);
            }
        }
        if (loadPowerLinesL.size() != loadPowerLines.length) {
            sLog.error("Some (or all) heater PS control lines are invalid");
        }
        else if (loadPowerLinesL.isEmpty()) {
            sLog.warn("No heater PS control lines specified: assume always on");
        }
        if (loadPowerDevcD == null) {
            sLog.warn("No heater PS device specified: assume no heater control");
            state.setHeaterControlState(RefrigState.CONTROL_NONE);   // Indicate no heater control
        }
        if (alarmDsabChansL.size() != alarmDsabChans.length) {
            sLog.error("Some alarm disable channels are invalid");
        }
        if (loadTempChansL.size() != loadTempChans.length) {
            sLog.error("Some load temperature channels are invalid");
        }
        if (tempCtrlC == null) {
            sLog.error("No valid temperature controller specified");
        }

       /*
        *  Initialize all monitoring configuration data
        */
        mon = new Monitor(this, this, sLog);
        mon.initConfiguration();

       /*
        *  Add monitoring commands
        */
        subsys.addCommandsFromObject(new MonitorCommands(mon), "");

       /*
        *  Initialize the temperature controller
        */
        if (tempCtrlC != null) {
            tempCtrlC.initialize(loadTempChansL, loadPowerDevcD, mon);
        }
    }


   /**
    *  Initializes the hardware.
    *
    *  @return  GO
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        /*
        *  Set power control lines' warm start option
        */
        for (List<Line> lines : cmprPowerLinesL) {
            for (Line line : lines) {
                line.setWarm(!coldStart);
            }
        }
        for (Line line : loadPowerLinesL) {
            line.setWarm(!coldStart);
        }

        /*
        *  Initialize the hardware
        */
        mon.initSensors(checkMillis);

        /*
        *  Turn off the power control lines' warm start option
        */
        for (List<Line> lines : cmprPowerLinesL) {
            for (Line line : lines) {
                line.setWarm(false);
            }
        }
        for (Line line : loadPowerLinesL) {
            line.setWarm(false);
        }

        /*
        *  Make sure state and power control lines are consistent
        */
        for (int cmpr = 0; cmpr < state.getNumCmprs(); cmpr++) {
            setCmprPowerEnable(cmpr, isCmprPowerOn(cmpr) ? 1 : 0);
        }
        setHeaterPowerEnable(isHeaterPowerOn() ? 1 : 0);

        return TreeWalkerDiag.GO;
    }


   /**
    *  Checks whether hardware is started.
    */
    @Override
    public void checkStarted()
    {
        sLog.info("checkStarted was called");
    }


   /**
    *  Checks whether hardware is stopped.
    */
    @Override
    public void checkStopped()
    {
        sLog.info("checkStopped was called");
    }


   /**
    *  Starts the subsystem.
    *
    *  @throws  HardwareException
    */
    @Override
    public void postStart() throws HardwareException
    {
        running = true;
        mon.start(monMillis);
        sLog.info("Refrigeration subsystem (" + subsys.getName() + ") started");
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        (new Timer("SetPower", true)).scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                setHeaterPower();     // Make sure it's correct
            }
        }, POWER_SET_INTERVAL, POWER_SET_INTERVAL);
        super.postStart();
    }


   /**
    *  Performs periodic trending data broadcast.
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    *  Sets the tick period.
    *
    *  Overrides the method in Module in order to publish a status update.
    *
    *  @param  value  The tick period (milliseconds) to set.
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        state.setTickMillis(value);
        if (running) {
            publishState();
        }
    }


   /**
    *  Handles alarm events.
    *
    *  @param  event  The event type
    *
    *  @param  parm   The event parameter
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        if (parm < state.getNumCmprs()) {
            int first = (parm < 0) ? 0 : parm;
            int last = (parm < 0) ? state.getNumCmprs() : parm + 1;
            for (int j = first; j < last; j++) {
                if (event == Alarm.EVENT_TRIP) {
                    if (state.getCmprPowerState(j) != RefrigState.POWER_TRIPPED) {
                        state.setCmprPowerState(j, RefrigState.POWER_TRIPPED);
                        setCmprPowerSwitch(j);
                        publishState();
                    }
                }
                else if (event == Alarm.EVENT_RESET) {
                    if (state.getCmprPowerState(j) == RefrigState.POWER_TRIPPED) {
                        state.setCmprPowerState(j, RefrigState.POWER_OFF);
                        publishState();
                    }
                }
            }
        }
        else if (parm == ALARM_HEATER) {
            if (event == Alarm.EVENT_TRIP) {
                if (state.getHeaterPowerState() != RefrigState.POWER_TRIPPED) {
                    state.setHeaterPowerState(RefrigState.POWER_TRIPPED);
                    setHeaterPowerSwitch();
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if (state.getHeaterPowerState() == RefrigState.POWER_TRIPPED) {
                    state.setHeaterPowerState(RefrigState.POWER_OFF);
                    publishState();
                }
            }
        }
    }


   /**
    *  Sets the power state of a compressor.
    *
    *  @param  cmpr   The compressor number
    *
    *  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    */
    @Command(type=CommandType.ACTION, description="Set compressor power enabled state")
    public void setCmprPowerEnable(int cmpr, int value)
    {
        if (cmpr < 0 || cmpr >= state.getNumCmprs()) {
            sLog.error("Invalid compressor number");
            return;
        }
        int oldState = state.getCmprPowerState(cmpr);
        int newState = (oldState == RefrigState.POWER_TRIPPED) ? oldState
                         : (value == 0) ? RefrigState.POWER_OFF : RefrigState.POWER_ON;
        if (newState != oldState) {
            state.setCmprPowerState(cmpr, newState);
            setCmprPowerSwitch(cmpr);
            publishState();
        }
    }


   /**
    *  Sets the heater power enable state.
    *
    *  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    */
    @Command(type=CommandType.ACTION, description="Set the heater power enabled state")
    public void setHeaterPowerEnable(int value)
    {
        int oldState = state.getHeaterPowerState();
        int newState = (oldState == RefrigState.POWER_TRIPPED) ? oldState
                         : (value == 0) ? RefrigState.POWER_OFF : RefrigState.POWER_ON;
        if (newState != oldState) {
            state.setHeaterPowerState(newState);
            setHeaterPowerSwitch();
            publishState();
        }
    }


   /**
    *  Sets the heater control state.
    *
    *  @param  value  The heater power state to set: 0 = off; &gt; 0 = manual;
    *                 &lt; 0 = automatic (temp loop).
    */
    @Command(type=CommandType.ACTION, description="Set the heater control state")
    public void setHeaterControl(int value)
    {
        int oldState = state.getHeaterControlState();
        if (oldState != RefrigState.CONTROL_NONE) {
            int newState = (value == 0) ? RefrigState.CONTROL_OFF
                            : (value > 0) ? RefrigState.CONTROL_ON
                            : tempCtrlC != null ? RefrigState.CONTROL_AUTO
                            : RefrigState.CONTROL_ON;
            if (newState != oldState) {
                state.setHeaterControlState(newState);
                if (newState == RefrigState.CONTROL_AUTO) {
                    tempCtrlC.setTemp(state.getLoadTemp());
                    startTempControl();
                }
                else {
                    stopTempControl();
                    setHeaterPower();
                }
                publishState();
            }
        }
    }


   /**
    *  Sets the alarm disabled state on or off.
    *
    *  @param  value  The alarm disabled state value to set: 0 = off;
    *                 ~0 = on.
    */
    @Command(type=CommandType.ACTION, description="Set the load trip enabled state")
    public void setAlarmDisabled(int value)
    {
        boolean disable = (value != 0);
        state.setAlarmDisabled(disable);
        for (Channel chan : alarmDsabChansL) {
            chan.enableAlarm(false, !disable);
        }
        publishState();
    }


   /**
    *  Sets the heater power value.
    *
    *  @param  value  The load power set point.
    */
    @Command(type=CommandType.ACTION, description="Set the heater power set point")
    public void setHeaterPower(double value)
    {
        state.setHeaterPower(value);
        if (startPower != null) {
            startPower = value;
        }
        setHeaterPower();
        publishState();
    }


   /**
    *  Sets the load temperature value.
    *
    *  @param  value  The load temperature set point.
    */
    @Command(type=CommandType.ACTION, description="Set the load temperature set point")
    public void setLoadTemp(double value)
    {
        state.setLoadTemp(value);
        if (tempCtrlC != null) {
            tempCtrlC.setTemp(value);
        }
        publishState();
    }


   /**
    *  Saves a named refrigeration configuration.
    *
    *  @param  name  The configuration name
    *
    *  @throws  IOException
    */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        subsys.saveChangesForCategoriesAs(REFRIG + ":" + name);
    }


   /**
    *  Loads a named refrigeration configuration.
    *
    *  @param  name  The configuration name
    *
    *  @throws  IOException
    */
    @Command(type=CommandType.ACTION, description="Load a named configuration")
    public void loadNamedConfig(String name) throws IOException
    {
        subsys.loadCategories(REFRIG + ":" + name);
    }

  
   /**
    *  Gets the full state of the refrigeration module.
    *
    *  This is intended to be called by GUIs during initialization
    *
    *  @return  The full refrigeration state
    */
    @Command(type=CommandType.QUERY, description="Get the full refrigeration state")
    public RefrigFullState getFullState()
    {
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(state, monState);
    }    


   /**
    *  Publishes the state of the refrigeration module.
    *
    *  This is intended to be called whenever any element of the state is
    *  changed.
    */
    private void publishState()
    {
        KeyValueData data = new KeyValueData(RefrigState.KEY, state);
        subsys.publishSubsystemDataOnStatusBus(data);
    }    


   /**
    *  Powers a compressor on or off, according to its state.
    */
    private void setCmprPowerSwitch(int cmpr)
    {
        boolean on = state.getCmprPowerState(cmpr) == RefrigState.POWER_ON;
        for (Line line : cmprPowerLinesL.get(cmpr)) {
            line.set(on);
        }
    }


   /**
    *  Sets the heater power on or off, according to the state.
    */
    private void setHeaterPowerSwitch()
    {
        boolean on = state.getHeaterPowerState() == RefrigState.POWER_ON;
        for (Line line : loadPowerLinesL) {
            line.set(on);
        }
        if (loadPowerDevcD != null) {
            if (on) {
                loadPowerDevcD.enable();
                if (state.getHeaterControlState() == RefrigState.CONTROL_AUTO) {
                    startTempControl();
                }
            }
            else {
                stopTempControl();
                loadPowerDevcD.disable();
            }
        }
    }
            

   /**
    *  Sets the heater power.
    */
    private void setHeaterPower()
    {
        if (loadPowerDevcD == null || !loadPowerDevcD.isOnline()) return;
        loadPowerDevcD.enableOutput(0, state.getHeaterControlState() != RefrigState.CONTROL_OFF);
        if (state.getHeaterControlState() == RefrigState.CONTROL_ON) {
            double loadPower = state.getHeaterPower();
            loadPowerDevcD.setPower(0, loadPower);
            startPower = loadPower;
        }
    }
    

   /**
    *  Gets the power state of a hardware compressor
    *
    *  @param  cmpr  The compressor number
    */
    private boolean isCmprPowerOn(int cmpr)
    {
        if (cmprPowerLinesL.isEmpty() || cmprPowerLinesL.get(cmpr).isEmpty()) return false;
        for (Line line : cmprPowerLinesL.get(cmpr)) {
            if (!line.isSet()) return false;
        }
        return true;
    }
            

   /**
    *  Gets the hardware heater power state.
    */
    private boolean isHeaterPowerOn()
    {
        if (loadPowerLinesL.isEmpty()) return true;
        for (Line line : loadPowerLinesL) {
            if (!line.isSet()) return false;
        }
        return true;
    }
    

   /**
    *  Starts the temperature control loop.
    */
    private void startTempControl()
    {
        if (tempCtrlC != null) {
            if (startPower == null) {
                tempCtrlC.restart();
            }
            else {
                tempCtrlC.start(startPower);
                startPower = null;
            }
        }
    }
    

   /**
    *  Stops the temperature control loop.
    */
    private void stopTempControl()
    {
        if (tempCtrlC != null) {
            tempCtrlC.stop();
        }
    }


   /**
    *  Gets the named component of a specified class.
    */
    private Object getComponent(String name, Class cls)
    {
        Object cmpt = getComponentByName(name);
        if (cmpt != null && !cls.isInstance(cmpt)) {
            cmpt = null;
        }
        return cmpt;
    }

}
