package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.monitor.Device;

/**
 **************************************
 *
 *  Defines a power supply interface.
 *
 *  @author Owen Saxton
 *
 **************************************
 */
public abstract class PowerDevice extends Device {

    private final static double
        MIN_VOLTAGE = 10;

    private final Map<Integer, PowerState> states = new HashMap<>();

   /**
    *  Inner class for saving the power state of a channel.
    */
    public static class PowerState {
        
        private double resistance, power;
        private boolean enabled;

    }


   /**
    *  Enables the output.
    *
    *  @param  chan    The hardware channel
    *
    *  @param  enable  The enabled state to set
    */
    public void enableOutput(int chan, boolean enable)
    {
        getPowerState(chan).enabled = enable;
        setOutput(chan, enable);
    }


   /**
    *  Sets the power.
    *
    *  @param  chan   The hardware channel
    *
    *  @param  value  The power to set
    */
    public void setPower(int chan, double value)
    {
        PowerState state = getPowerState(chan);
        if (!state.enabled) return;
        StringBuilder msg = new StringBuilder("Setting power to ");
        msg.append((float)value);
        List<Float> voltages = new ArrayList<>();
        state.power = value;
        if (state.resistance == 0) {
            boolean okay = false;
            double ohms = 0; 
            for (int j = 0; j < 5 && !okay; j++) {
                double prevOhms = ohms;
                setVoltage(chan, MIN_VOLTAGE);
                double amps = readCurrent(chan);
                ohms = (amps <= 0) ? 0 : readVoltage(chan) / amps;
                okay = areClose(ohms, prevOhms);
            }
            if (okay && ohms != 0) {
                state.resistance = ohms;
            }
            else {
                log.error("Unable to determine load resistance");
                return;
            }
        }
        boolean okay = false;
        double ohms = 0;
        for (int j = 0; j < 5 && !okay; j++) {
            double prevOhms = ohms;
            double volts = Math.sqrt(state.power * state.resistance);
            setVoltage(chan, volts);
            voltages.add((float)volts);
            double amps = readCurrent(chan);
            ohms = (amps <= 0) ? 0 : readVoltage(chan) / amps;
            okay = areClose(ohms, state.resistance) || areClose(ohms, prevOhms);
        }
        if (okay) {
            if (ohms != 0) {
                state.resistance = ohms;
                double volts = Math.sqrt(state.power * state.resistance);
                setVoltage(chan, volts);
                voltages.add((float)volts);
            }
        }
        else {
            log.error("Unable to set power consistently");
        }
        msg.append(": voltages =");
        for (Float v : voltages) {
            msg.append(" ").append(v);
        }
        log.debug(msg);
    }


    private boolean areClose(double val1, double val2)
    {
        if (val2 == 0) return val1 == 0;
        double error = (val1 - val2) / val2;
        return error > -0.05 && error < 0.05;
    }

   
   /**
    *  Sets the online state.
    *
    *  If coming on line, set previously-cached value.
    *
    *  @param  online  The online state to set: true or false
    */
    @Override
    protected void setOnline(boolean online)
    {
        super.setOnline(online);
        if (online) {
            for (Integer chan : states.keySet()) {
                setPower(chan, states.get(chan).power);
            }
        }
    }


   /**
    *  Sets the voltage.
    *
    *  @param  chan   The hardware channel
    *
    *  @param  value  The voltage to set
    */
    public abstract void setVoltage(int chan, double value);


   /**
    *  Sets the current.
    *
    *  @param  chan   The hardware channel
    *
    *  @param  value  The current to set
    */
    public abstract void setCurrent(int chan, double value);


   /**
    *  Sets the output state.
    *
    *  @param  chan   The hardware channel
    *
    *  @param  value  The output state to set
    */
    public abstract void setOutput(int chan, boolean value);


   /**
    *  Reads the voltage.
    *
    *  @param  chan  The hardware channel
    *
    *  @return  The voltage
    */
    public abstract double readVoltage(int chan);


   /**
    *  Reads the current.
    *
    *  @param  chan  The hardware channel
    *
    *  @return  The current
    */
    public abstract double readCurrent(int chan);


   /**
    *  Gets the output state.
    *
    *  @param  chan   The hardware channel
    *
    *  @return  The output state
    */
    public abstract boolean getOutput(int chan);


   /**
    *  Gets the power state for a channel.
    *
    *  @param  chan   The hardware channel
    *
    *  @return  The power state
    */
    private PowerState getPowerState(int chan)
    {
        PowerState state = states.get(chan);
        if (state == null) {
            state = new PowerState();
            states.put(chan, state);
        }

        return state;
    }

}
