package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.subsystem.monitor.Control;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ************************************************
 *
 *  Handles a Pluto PLC system control channel.
 *
 *  @author Owen Saxton
 *
 ************************************************
 */
public class PlutoControl extends Control {

   /**
    *  Constants.
    */
    private final static int
        TYPE_BIT  = 0,
        TYPE_REG  = 1;

   /**
    *  Private lookup maps.
    */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("BIT", TYPE_BIT);
        typeMap.put("REG", TYPE_REG);
    }

   /**
    *  Data fields.
    */
    private final PlutoDevice pDevc;
    private final String sType;
    private final double offset, scale;
    private int type, area;


   /**
    *  Constructor.
    *
    *  @param  desc
    *
    *  @param  type
    *
    *  @param  hwChan
    *
    *  @param  offset
    *
    *  @param  scale
    */
    public PlutoControl(String desc, int hwChan, String type, double offset,
                        double scale)
    {
        super(desc, hwChan);
        this.sType = type;
        this.offset = offset;
        this.scale = scale;
        pDevc = (PlutoDevice)devc;
    }


   /**
    *  Performs configuration.
    *
    *  @param  mon   The associated monitor
    *
    *  @param  devc  The associated device
    */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        Integer iType = null;
        String[] typeFields = sType.split(":", -1);
        if (typeFields.length == 2) {
            iType = typeMap.get(typeFields[0].toUpperCase());
        }
        try {
            area = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            area = -1;
        }
        if (iType == null || area < 0 || area >= Pluto.NUM_DTP_AREAS) {
            mon.reportConfigError(getName(), "type", "has invalid value");
        }
        type = iType;
        int maxChan = type == TYPE_BIT ? 16 : 2;
        if (hwChan >= maxChan) {
            mon.reportConfigError(getName(), "hwchan", "has invalid value");
        }
    }


   /**
    *  Adds to a map of configured values.
    *
    *  @param  map  The map of configured values to be updated
    */
    @Override
    public void addConfigValues(Map<String, Object> map)
    {
    }


   /**
    *  Writes a value.
    *
    *  @param  value  The value
    */
    public void writeValue(double value)
    {
        int ival = (int)((value - offset) / scale);
        if (type == TYPE_BIT) {
            pDevc.writeBit(area, hwChan, ival);
        }
        else {
            pDevc.writeRegister(area, hwChan, ival);
        }
    }

}
