package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.GP390;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ***************************************
 *
 *  Handles an MKS GP390 vacuum gauge.
 *
 *  @author Owen Saxton
 *
 ***************************************
 */
public class GP390Device extends Device {

   /**
    *  Data fields.
    */
    private final static int BAUD_RATE = 19200;

    private String  serial;   // Serial number of FTDI adapter
    private int     address;  // RS-485 address of gauge

    private GP390   gp;       // Associated GP390 object


   /**
    *  Constructor.
    *
    *  @param  serialNo  The serial number of the FTDI adapter
    *
    *  @param  addr      The RS-485 node address
    */
    public GP390Device(String serialNo, int addr)
    {
        serial = serialNo;
        address = addr;
    }


    public GP390Device() {
    }


   /**
    *  Performs configuration.
    *
    *  @param  mon  The monitor object
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        if (serial == null) {
            mon.reportConfigError(getName(), "serial", "is missing");
        }
        fullName = "GP390 vacuum gauge (" + serial + ":" + address + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            if (!inited || gp == null) {
                gp = new GP390();
            }
            gp.open(GP390.ConnType.FTDI, serial, BAUD_RATE, address);
            gp.setLock(false);
            gp.setPressureUnit(GP390.UNIT_TORR);
            gp.setLock(true);
            setOnline(true);
            initSensors();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (gp != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            gp.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *
    *  @param  type     The encoded channel type returned by checkChannel.
    *
    *  @return  The value read from the channel
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (online) {
            try {
                value = gp.readPressure();
            }
            catch (DriverException e) {
                log.error("Error reading pressure: " + e);
                setOnline(false);
            }
        }

        return value;
    }

}
