package org.lsst.ccs.subsystem.refrig;

import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Implements a temperature controller for the refrigeration system.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TempControl extends ConfigurableComponent {

    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput;         // maximum PID output (watts)
    private Double  awGain;            // anti-windup gain
    private Double  basePower;         // base power input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 0.0;   // minimum PID output (watts)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)

    private PIController pic;               // The PI controller
    private PowerDevice powerDevc;          // The power device
    private List<Channel>tempChans;         // Temperature channels to use
    private double[] temps;                 // Temperatures to use
    private double lastPower;               // The last power value set
    private boolean running;                // True if loop is running
    private Timer loopTimer;                // Timer for running the loop


   /**
    ***************************************************************************
    **
    **  Inner class for control loop iteration.
    **
    ***************************************************************************
    */
    private class Iterate extends TimerTask {

        @Override
        public void run()
        {
            int index = 0;
            for (Channel tempChan : tempChans) {
                temps[index++] = tempChan.getValue();
            }
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(temps, tod);
            powerDevc.setPower(0, lastPower);
        }
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  gain        The loop gain
    **
    **  @param  timeConst   The time constant (sec)
    **
    **  @param  smoothTime  The smoothing time (sec)
    **
    **  @param  maxOutput   The maximum output power allowed (watts)
    **
    **  @param  awGain      The anti-windup gain
    **
    **  @param  basePower   The base power input (watts)
    **
    **  @param  tolerance   The maximum tolerated error (percent)
    **
    **  @param  updateTime  The update time interval (msec)
    **
    ***************************************************************************
    */
    public TempControl(double gain, double timeConst, double smoothTime,
                       double maxOutput, double awGain, double basePower,
                       double tolerance, int updateTime)
    {
        this.gain = gain;
        this.timeConst = timeConst;
        this.smoothTime = smoothTime;
        this.maxOutput = maxOutput;
        this.awGain    = awGain;
        this.basePower = basePower;
        this.tolerance = tolerance;
        this.updateTime = updateTime;
    }


    public TempControl() {
    }


   /**
    ***************************************************************************
    **
    **  Initializes the parameters.
    **
    **  @param  tempChans   The list of temperature channels used for input
    **
    **  @param  powerDevc   The power device being controlled
    **
    **  @param  mon         The subsystem monitor object
    **
    ***************************************************************************
    */
    public void initialize(List tempChans, PowerDevice powerDevc, Monitor mon)
    {
        if (gain == null) {
            mon.reportConfigError(getName(), "gain", "is missing");
        }
        if (timeConst == null) {
            mon.reportConfigError(getName(), "timeConst", "is missing");
        }
        if (smoothTime == null) {
            mon.reportConfigError(getName(), "smoothTime", "is missing");
        }
        if (maxOutput == null) {
            mon.reportConfigError(getName(), "maxOutput", "is missing");
        }
        if (awGain == null) {
            mon.reportConfigError(getName(), "awGain", "is missing");
        }
        if (basePower == null) {
            mon.reportConfigError(getName(), "basePower", "is missing");
        }
        if (tolerance == null) {
            mon.reportConfigError(getName(), "tolerance", "is missing");
        }
        if (updateTime == null) {
            mon.reportConfigError(getName(), "updateTime", "is missing");
        }
        this.tempChans = tempChans;
        this.powerDevc = powerDevc;
        temps = new double[tempChans.size()];
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


   /**
    ***************************************************************************
    **
    **  Sets the target temperature.
    **
    **  @param  value  The temperature to set
    **
    ***************************************************************************
    */
    public void setTemp(double value)
    {
        pic.setSetpoint(value);
   }


   /**
    ***************************************************************************
    **
    **  Resets the controller.
    **
    ***************************************************************************
    */
    public void reset()
    {
        pic.reset();
    }


   /**
    ***************************************************************************
    **
    **  Restarts the control loop.
    **
    ***************************************************************************
    */
    public void restart()
    {
        start(lastPower);
    }


   /**
    ***************************************************************************
    **
    **  Starts the control loop.
    **
    **  @param  power  The starting power value
    **
    ***************************************************************************
    */
    public void start(double power)
    {
        if (running) return;
        running = true;
        pic.reset();
        pic.setIntegral(power - basePower);
        loopTimer = new Timer(true);
        loopTimer.schedule(new Iterate(), 0L, updateTime);
    }


   /**
    ***************************************************************************
    **
    **  Stops the control loop.
    **
    ***************************************************************************
    */
    public void stop()
    {
        if (!running) return;
        running = false;
        loopTimer.cancel();
    }

}
