package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.MonitorCommands;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration subscale camera modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Subscale extends Module
                      implements HardwareController, Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        TICK_INTERVAL = 10000,
        EVENT_ID_MAIN_POWER = 0,
        POWER_SET_INTERVAL = 5000;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private static final String REFRIG = "Refrig";
    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.refrig");

    private int monMillis = 1000;
    private int checkMillis = 0;
    private String loadPowerDevc;
    private String[] cmprPowerLines;
    private String[] loadPowerLines;
    private String[] alarmDsabChans;
    private String[] loadTempChans;
    private String tempCtrl;

    private boolean coldStart, running;
    private final List<Line> cmprPowerLinesL = new ArrayList<>();
    private final List<Line> loadPowerLinesL = new ArrayList<>();
    private final List<Channel> alarmDsabChansL = new ArrayList<>();
    private final List<Channel> loadTempChansL = new ArrayList<>();

    private ConfigurableSubsystem subsys;
    private PowerDevice loadPowerDevcD;
    private TempControl tempCtrlC;
    private Monitor mon;
    private int state = RefrigState.LOAD_TRIP_ENAB_STATE;
    private double loadTemp, loadPower;
    private Double startPower = 0.0;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name            The subsystem name.
    **
    **  @param  tickMillis      The tick period (ms).
    **
    **  @param  monMillis       The monitor period (ms).
    **
    **  @param  checkMillis     The hardware check period (ms).
    **
    **  @param  cmprPowerLines  The names of the compressor power lines.
    **
    **  @param  loadPowerLines  The names of the load power lines.
    **
    **  @param  loadPowerDevc   The name of the load power device.
    **
    **  @param  alarmDsabChans  The names of the channels whose alarms are to
    **                          be disabled during cool-down.
    **
    **  @param  loadTempChans   The names of the load temperature channels.
    **
    **  @param  tempCtrl        The name of the temperature controller.
    **
    ***************************************************************************
    */
    public Subscale(String name, int tickMillis, int monMillis, int checkMillis,
                    String[] cmprPowerLines, String[] loadPowerLines, String loadPowerDevc,
                    String[] alarmDsabChans, String[] loadTempChans, String tempCtrl)
    {
        super(name, tickMillis);
        this.monMillis = monMillis;
        this.checkMillis = checkMillis;
        this.cmprPowerLines = cmprPowerLines;
        this.loadPowerLines = loadPowerLines;
        this.loadPowerDevc = loadPowerDevc;
        this.alarmDsabChans = alarmDsabChans;
        this.loadTempChans = loadTempChans;
        this.tempCtrl = tempCtrl;
    }


    public Subscale() {
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  General initialization
        */
        setName("main");
        if (getTickMillis() <= 0) {
            setTickMillis(TICK_INTERVAL);
        }
        subsys = (ConfigurableSubsystem)getSubsystem();

        /*
        **  Get cold start option
        */
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");

        /*
        **  Initialize component references
        */
        loadPowerDevcD = (PowerDevice)getComponent(loadPowerDevc,
                                                   PowerDevice.class);
        for (String chanS : cmprPowerLines) {
            Object cmpt = getComponent((String)chanS, Line.class);
            if (cmpt != null) {
                cmprPowerLinesL.add((Line)cmpt);
            }
        }
        for (String chanS : loadPowerLines) {
            Object cmpt = getComponent((String)chanS, Line.class);
            if (cmpt != null) {
                loadPowerLinesL.add((Line)cmpt);
            }
        }
        for (String chanS : alarmDsabChans) {
            Object cmpt = getComponent((String)chanS, Channel.class);
            if (cmpt != null) {
                alarmDsabChansL.add((Channel)cmpt);
            }
        }
        for (String chanS : loadTempChans) {
            Object cmpt = getComponent((String)chanS, Channel.class);
            if (cmpt != null) {
                loadTempChansL.add((Channel)cmpt);
            }
        }
        tempCtrlC = (TempControl)getComponent(tempCtrl, TempControl.class);

        if (cmprPowerLinesL.isEmpty()) {
            sLog.error("No valid compressor power lines specified");
        }
        else if (cmprPowerLinesL.size() != cmprPowerLines.length) {
            sLog.error("Some compressor power lines are invalid");
        }
        if (loadPowerLinesL.isEmpty()) {
            sLog.error("No valid load power lines specified");
        }
        else if (loadPowerLinesL.size() != loadPowerLines.length) {
            sLog.error("Some load power lines are invalid");
        }
        if (loadPowerDevcD == null) {
            sLog.error("No valid load power device specified");
            state |= RefrigState.LOAD_TEMP_LOOP_STATE;   // Indicate no power control
        }
        if (alarmDsabChansL.size() != alarmDsabChans.length) {
            sLog.error("Some alarm disable channels are invalid");
        }
        if (loadTempChansL.size() != loadTempChans.length) {
            sLog.error("Some load temperature channels are invalid");
        }
        if (tempCtrlC == null) {
            sLog.error("No valid temperature controller specified");
        }
        
        /*
        **  Initialize all monitoring configuration data
        */
        mon = new Monitor(this, this, sLog);
        mon.initConfiguration();
        
        /*
        **  Add monitoring commands
        */
        subsys.addCommandsFromObject(new MonitorCommands(mon), "");

        /*
        **  Initialize the temperature controller
        */
        if (tempCtrlC != null) {
            tempCtrlC.initialize(loadTempChansL, loadPowerDevcD, mon);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        /*
        **  Set power control lines' warm start option
        */
        for (Line line : cmprPowerLinesL) {
            line.setWarm(!coldStart);
        }
        for (Line line : loadPowerLinesL) {
            line.setWarm(!coldStart);
        }

        /*
        **  Initialize the hardware
        */
        mon.initSensors(checkMillis);

        /*
        **  Turn off the power control lines' warm start option
        */
        for (Line line : cmprPowerLinesL) {
            line.setWarm(false);
        }
        for (Line line : loadPowerLinesL) {
            line.setWarm(false);
        }

        /*
        **  Make sure state and power control lines are consistent
        */
        setMainPowerEnable(isCmprPowerOn() ? 1 : isLoadPowerOn() ? -1 : 0);

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
        System.out.println("checkStarted was called");
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
        System.out.println("checkStopped was called");
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    **  @throws  HardwareException
    **
    ***************************************************************************
    */
    @Override
    public void postStart() throws HardwareException
    {
        running = true;
        mon.start(monMillis);
        sLog.info("Refrigeration subsystem (" + subsys.getName() + ") started");
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        (new Timer("SetPower", true)).scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                setLoadPower();     // Make sure it's correct
            }
        }, POWER_SET_INTERVAL, POWER_SET_INTERVAL);
        super.postStart();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    **  @param  event  The event type
    **
    **  @param  parm   The event parameter
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setMainPowerEnable(0);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
                }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power state.
    **
    **  @param  value    The enabled state value to set: 0 = off,
    **                   &gt; 0 = fully on, &lt; 0 = heat load only.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the main power enabled state")
    public void setMainPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
                if (value > 0) {
                    state &= ~RefrigState.MAIN_LOAD_ONLY_STATE;
                }
                else {
                    state |= RefrigState.MAIN_LOAD_ONLY_STATE;
                }
            }
        }
        else {
            state &= ~(RefrigState.MAIN_POWER_STATE
                        | RefrigState.MAIN_LOAD_ONLY_STATE);
        }
        setMainPower();
        setLoadPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power state.
    **
    **  @param  value  The load power state to set: 0 = off; &gt; 0 = manual;
    **                 &lt; 0 = automatic (temp loop).
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the load power enabled state")
    public void setLoadPowerEnable(int value)
    {
        if (loadPowerDevcD != null) {
            if (value != 0) {
                state |= RefrigState.LOAD_POWER_STATE;
            }
            else {
                state &= ~RefrigState.LOAD_POWER_STATE;
            }
            if (value < 0) {
                if (tempCtrlC != null) {
                    state |= RefrigState.LOAD_TEMP_LOOP_STATE;
                    tempCtrlC.setTemp(loadTemp);
                    if ((state & RefrigState.MAIN_POWER_STATE) != 0) {
                        startTempCtrl();
                    }
                }
            }
            else {
                state &= ~RefrigState.LOAD_TEMP_LOOP_STATE;
                stopTempCtrl();
                setLoadPower();
            }
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load trip enabled state on or off.
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the load trip enabled state")
    public void setLoadTripEnable(int value)
    {
        boolean enable = (value != 0);
        if (enable) {
            state |= RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        else {
            state &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        for (Channel chan : alarmDsabChansL) {
            chan.enableAlarm(false, enable);
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power value.
    **
    **  @param  value  The load power set point.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the load power set point")
    public void setLoadPower(double value)
    {
        loadPower = value;
        if (startPower != null) {
            startPower = loadPower;
        }
        setLoadPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load temperature value.
    **
    **  @param  value  The load temperature set point.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the load temperature set point")
    public void setLoadTemp(double value)
    {
        loadTemp = value;
        if (tempCtrlC != null) {
            tempCtrlC.setTemp(loadTemp);
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Saves a named refrigeration configuration.
    **
    **  @param  name  The configuration name
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        subsys.saveChangesForCategoriesAs(REFRIG + ":" + name);
    }


   /**
    ***************************************************************************
    **
    **  Loads a named refrigeration configuration.
    **
    **  @param  name  The configuration name
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Load a named configuration")
    public void loadNamedConfig(String name) throws IOException
    {
        subsys.loadCategories(REFRIG + ":" + name);
    }

  
   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    **  @return  The full refrigeration state
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Get the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis(),
                                                loadPower, loadTemp);
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    private void publishState()
    {
        RefrigState rState = new RefrigState(state, getTickMillis(), loadPower, loadTemp);
        KeyValueData data = new KeyValueData(RefrigState.KEY, rState);
        subsys.publishSubsystemDataOnStatusBus(data);
    }    


   /**
    ***************************************************************************
    **
    **  Sets the compressor and load power on or off, according to the state.
    **
    ***************************************************************************
    */
    private void setMainPower()
    {
        boolean on = (state & RefrigState.MAIN_POWER_STATE) != 0;
        boolean loadOnly = (state & RefrigState.MAIN_LOAD_ONLY_STATE) != 0;
        for (Line line : cmprPowerLinesL) {
            line.set(on & !loadOnly);
        }
        for (Line line : loadPowerLinesL) {
            line.set(on);
        }
        if (loadPowerDevcD != null) {
            if (on) {
                loadPowerDevcD.enable();
                if ((state & RefrigState.LOAD_TEMP_LOOP_STATE) != 0) {
                    startTempCtrl();
                }
            }
            else {
                stopTempCtrl();
                loadPowerDevcD.disable();
            }
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the load power.
    **
    ***************************************************************************
    */
    private void setLoadPower()
    {
        if (loadPowerDevcD == null || !loadPowerDevcD.isOnline()) return;
        loadPowerDevcD.enableOutput(0, (state & RefrigState.LOAD_POWER_STATE) != 0);
        if ((state & RefrigState.LOAD_TEMP_LOOP_STATE) == 0) {
            loadPowerDevcD.setPower(0, loadPower);
            startPower = loadPower;
        }
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the hardware compressor power state
    **
    ***************************************************************************
    */
    private boolean isCmprPowerOn()
    {
        if (cmprPowerLinesL.isEmpty()) return false;
        for (Line line : cmprPowerLinesL) {
            if (!line.isSet()) return false;
        }
        return true;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state.
    **
    ***************************************************************************
    */
    private boolean isLoadPowerOn()
    {
        if (loadPowerLinesL.isEmpty()) return false;
        for (Line line : loadPowerLinesL) {
            if (!line.isSet()) return false;
        }
        return true;
    }
    

   /**
    ***************************************************************************
    **
    **  Starts the temperature control loop.
    **
    ***************************************************************************
    */
    private void startTempCtrl()
    {
        if (tempCtrlC != null) {
            if (startPower == null) {
                tempCtrlC.restart();
            }
            else {
                tempCtrlC.start(startPower);
                startPower = null;
            }
        }
    }
    

   /**
    ***************************************************************************
    **
    **  Stops the temperature control loop.
    **
    ***************************************************************************
    */
    private void stopTempCtrl()
    {
        if (tempCtrlC != null) {
            tempCtrlC.stop();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the named component of a specified class.
    **
    ***************************************************************************
    */
    private Object getComponent(String name, Class cls)
    {
        Object cmpt = getComponentByName(name);
        if (cmpt != null && !cls.isInstance(cmpt)) {
            cmpt = null;
        }
        return cmpt;
    }

}
