package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles a Dataforth Maq20 DAQ system.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Maq20Device extends Device {

   /**
    ***************************************************************************
    **
    **  Inner class for holding static module data.
    **
    ***************************************************************************
    */
    private static class ModuleDef {

        int type;
        Map rangeMap;

        ModuleDef(int type, Map rangeMap) {
            this.type = type;
            this.rangeMap = rangeMap;
        }

    }

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String, Integer> dummyRangeMap = new HashMap<>();
    private final static Map<String, Integer> ttcRangeMap = new HashMap<>();
    static {
        ttcRangeMap.put("T400", Maq20.RANGE_TC_T_400);
        ttcRangeMap.put("T220", Maq20.RANGE_TC_T_220);
    }
    private final static Map<String, Integer> jtcRangeMap = new HashMap<>();
    static {
        jtcRangeMap.put("J760", Maq20.RANGE_TC_J_760);
        jtcRangeMap.put("J393", Maq20.RANGE_TC_J_393);
        jtcRangeMap.put("J199", Maq20.RANGE_TC_J_199);
    }
    private final static Map<String, Integer> ktcRangeMap = new HashMap<>();
    static {
        ktcRangeMap.put("K1350", Maq20.RANGE_TC_K_1350);
        ktcRangeMap.put("K651", Maq20.RANGE_TC_K_651);
        ktcRangeMap.put("K332", Maq20.RANGE_TC_K_332);
    }
    private final static Map<String, Integer> rstcRangeMap = new HashMap<>();
    static {
        rstcRangeMap.put("R1750", Maq20.RANGE_TC_R_1750);
        rstcRangeMap.put("R990", Maq20.RANGE_TC_R_990);
        rstcRangeMap.put("S1750", Maq20.RANGE_TC_S_1750);
        rstcRangeMap.put("S970", Maq20.RANGE_TC_S_970);
    }
    private final static Map<String, Integer> rtdRangeMap = new HashMap<>();
    static {
        rtdRangeMap.put("R850", Maq20.RANGE_RTD_850);
        rtdRangeMap.put("R200", Maq20.RANGE_RTD_200);
        rtdRangeMap.put("R100", Maq20.RANGE_RTD_100);
    }
    private final static Map<String, Integer> voltRangeMap = new HashMap<>();
    static {
        voltRangeMap.put("V60", Maq20.RANGE_VOLT_60);
        voltRangeMap.put("V40", Maq20.RANGE_VOLT_40);
        voltRangeMap.put("V20", Maq20.RANGE_VOLT_20);
        voltRangeMap.put("V10", Maq20.RANGE_VOLT_10);
        voltRangeMap.put("V5", Maq20.RANGE_VOLT_5);
    }
    private final static Map<String, Integer> mvoltRangeMap = new HashMap<>();
    static {
        mvoltRangeMap.put("V2", Maq20.RANGE_MVOLT_2000);
        mvoltRangeMap.put("V1", Maq20.RANGE_MVOLT_1000);
        mvoltRangeMap.put("MV250", Maq20.RANGE_MVOLT_250);
        mvoltRangeMap.put("MV100", Maq20.RANGE_MVOLT_100);
        mvoltRangeMap.put("MV50", Maq20.RANGE_MVOLT_50);
    }
    private final static Map<String, Integer> ampRangeMap = new HashMap<>();
    static {
        ampRangeMap.put("MA0_20", Maq20.RANGE_MAMP_0_20);
        ampRangeMap.put("MA4_20", Maq20.RANGE_MAMP_4_20);
    }
    private final static Map<String, ModuleDef> typeMap = new HashMap<>();
    static {
        typeMap.put("JTC", new ModuleDef(Maq20.MOD_TYPE_JTC, jtcRangeMap));
        typeMap.put("KTC", new ModuleDef(Maq20.MOD_TYPE_KTC, ktcRangeMap));
        typeMap.put("TTC", new ModuleDef(Maq20.MOD_TYPE_TTC, ttcRangeMap));
        typeMap.put("RSTC", new ModuleDef(Maq20.MOD_TYPE_RSTC, rstcRangeMap));
        typeMap.put("RTD", new ModuleDef(Maq20.MOD_TYPE_RTD, rtdRangeMap));
        typeMap.put("VOLT", new ModuleDef(Maq20.MOD_TYPE_VD, voltRangeMap));
        typeMap.put("MVOLT", new ModuleDef(Maq20.MOD_TYPE_MVD, mvoltRangeMap));
        typeMap.put("AMP", new ModuleDef(Maq20.MOD_TYPE_ID, ampRangeMap));
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private String    node;     // Network node name
    private String[]  serial;   // Serial numbers of DAQ modules

    private Maq20     maq;      // Associated Maq20 object


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  nodeName  The node name of the MAQ20 system
    **
    **  @param  serialNo  The array of serial numbers of the MAQ20 modules in
    **                    the system, in device ID order
    **
    ***************************************************************************
    */
    public Maq20Device(String nodeName, String[] serialNo)
    {
        node = nodeName;
        serial = serialNo;
    }


    public Maq20Device() {
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    **  @param  mon  The associated monitor
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        if (node == null) {
            mon.reportConfigError(getName(), "node", "is missing");
        }
        if (serial == null) {
            mon.reportConfigError(getName(), "serial", "is missing");
        }
        fullName = "Maq20 system (" + node + ")";
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        try {
            if (!inited || maq == null) {
                maq = new Maq20();
            }
            maq.open(Maq20.CONN_TYPE_NETWORK, node);
            maq.register(serial);
            setOnline(true);
            initSensors();
            setOutputLines();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (maq != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            maq.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    **  @param  name     The name of the channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The channel type string.
    **
    **  @param  subtype  The channel subtype string.
    **
    **  @return  A two-element array containing the encoded type [0] and
    **           subtype [1] values.
    **
    **  @throws  Exception if any errors found in the parameters.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype)
        throws Exception
    {
        Integer iType = null;
        int modId;
        ModuleDef mDef;
        Map<String, Integer> rMap = dummyRangeMap;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            mDef = typeMap.get(typeFields[0].toUpperCase());
            if (mDef != null) {
                iType = mDef.type;
                rMap = mDef.rangeMap;
            }
        }
        try {
            modId = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            modId = -1;
        }
        if (iType == null || modId < 0 || modId >= Maq20.NUM_MODULES) {
            mon.reportError(name, "type", type);
        }
        Integer iRange = rMap.get(subtype.toUpperCase());
        if (iRange == null) {
            mon.reportError(name, "range", subtype);
        }

        return new int[]{(iType << 8) | (modId + 1), iRange};
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    **  Much of the testing that is often done in checkChannel() has to be
    **  done here because the device has to be opened first.
    **
    **  @param  name     The channel name
    **
    **  @param  id       The channel ID
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @param  subtype  The channel subtype returned by checkChannel.
    **
    ***************************************************************************
    */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type,
                               int subtype)
    {
        if (!online) return;

        int mod = type & 0xff, modType = type >> 8, range = subtype;
        String error = null;
        try {
            if (!maq.moduleExists(mod)) {
                mon.reportError(name, "module ID", mod - 1);
            }
            int mType = maq.getModuleType(mod);
            if (modType != mType
                  && !(modType == Maq20.MOD_TYPE_VD && mType == Maq20.MOD_TYPE_VS)
                  && !(modType == Maq20.MOD_TYPE_ID && mType == Maq20.MOD_TYPE_IS)) {
                mon.reportError(name, "module type", modType);
            }
            if (hwChan < 0 || hwChan >= maq.getNumInputs(mod)) {
                mon.reportError(name, "channel", hwChan);
            }
            if (range < 0 || range >= maq.getNumRanges(mod)) {
                mon.reportError(name, "range", range);
            }
            maq.enable(mod, hwChan, true);
            maq.setRange(mod, hwChan, range);
        }
        catch (DriverException e) {
            log.error("Error configuring " + fullName + ": " + e);
            setOnline(false);
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  Channel value
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (online) {
            try {
                value = maq.readValue(type & 0xff, hwChan);
            }
            catch (DriverException e) {
                log.error("Error reading channel: " + e);
                setOnline(false);
            }
        }

        return value;
    }

}
