package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.mmr.Mmr;
import org.lsst.ccs.drivers.scpi.Scpi;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles an MMR controller.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MMRDevice extends Device implements Mmr.Listener {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        CHAN_T0  = 0,
        CHAN_T1  = 1,
        CHAN_P0  = 2,
        CHAN_P1  = 3,
        CHAN_D0  = 4,
        CHAN_D1  = 5,
        CHAN_I   = 6,
        N_DEV_CHANS = 7;

    final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("FTDI", Scpi.CONN_TYPE_FTDI);
        typeMap.put("SERIAL", Scpi.CONN_TYPE_SERIAL);
    }
        
   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private String   connType;    // Connection type string (FTDI or SERIAL)
    private String   ident;       // Identification (Serial number or device name)

    private int      connTypeC;   // Connection type code
    private Mmr      mmr;         // MMR controller hardware object
    private final double[] value = new double[N_DEV_CHANS]; // Current values


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  cType  The connection type: ftdi or serial
    **
    **  @param  id     The device ID: USB serial no. or serial device name
    **
    ***************************************************************************
    */
    public MMRDevice(String cType, String id)
    {
        connType = cType;
        ident = id;
    }


    public MMRDevice() {
    }


   /**
    ***************************************************************************
    **
    **  Configures the device parameters.
    **
    **  @param  mon  The associated monitor
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        if (connType == null) {
            mon.reportConfigError(getName(), "connType", "is missing");
        }
        Integer iConnType = typeMap.get(connType.toUpperCase());
        if (iConnType == null) {
            mon.reportConfigError(getName(), "connType", "is invalid: " + connType);
        }
        if (ident == null) {
            mon.reportConfigError(getName(), "ident", "is missing");
        }
        connTypeC = iConnType;
        fullName = "MMR controller (" + ident + ")";
    }

    
   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (connTypeC < 0) return;
        try {
            if (!inited || mmr == null) {
                mmr = new Mmr();
                mmr.addListener(this);
            }
            mmr.open(connTypeC, ident);
            setOnline(true);
            log.info("Connected to " + fullName);
        }
        catch (Exception e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    **  This is a no-op
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    **  @param  name     The name of the channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The channel type string.
    **
    **  @param  subtype  The channel subtype string.
    **
    **  @return  A two-element array containing the encoded type [0] and
    **           subtype [1] values.
    **
    **  @throws  Exception if any errors found in the parameters.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            mon.reportError(name, "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  Channel value
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return online ? value[hwChan] : super.readChannel(hwChan, type);
    }


   /**
    ***************************************************************************
    **
    **  Changes the MMR controller powered state.
    **
    **  Zeroes values upon power-off.
    **
    **  @param  on  Whether the controller is powered on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (!on) {
            for (int j = 0; j < N_DEV_CHANS; j++) {
                value[j] = 0;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the MMR controller open state.
    **
    **  @param  e  The exception causing closure
    **
    ***************************************************************************
    */
    @Override
    public void setClosed(Exception e)
    {
        log.error("MMR controller error: " + e);
        setOnline(false);
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the MMR controller.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(double[] data)
    {
        value[CHAN_T0] = data[Mmr.FLD_T0];
        value[CHAN_T1] = data[Mmr.FLD_T1];
        value[CHAN_P0] = data[Mmr.FLD_P0];
        value[CHAN_P1] = data[Mmr.FLD_P1];
        value[CHAN_D0] = data[Mmr.FLD_D0];
        value[CHAN_D1] = data[Mmr.FLD_D1];
        value[CHAN_I]  = data[Mmr.FLD_I];
    }

}
