package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.MonitorCommands;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class LongLines extends Module
                       implements HardwareController, Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        EVENT_ID_MAIN_POWER = 0,
        EVENT_ID_LOAD_POWER = 1;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.refrig");
    private final boolean coldStart;
    private final int monMillis, checkMillis;
    private final String mainPower, loadPower;
    private ConfigurableSubsystem subsys;
    private Monitor mon;
    private boolean loadAlarm;
    private int state = RefrigState.LOAD_POWER_STATE;
    private Line mainPowerLine, loadPowerLine;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name         The subsystem name
    **
    **  @param  tickMillis   The tick period (ms)
    **
    **  @param  monMillis    The monitor period (ms).
    **
    **  @param  checkMillis  The hardware check period (ms).
    **
    **  @param  mainPower    The name of the main power line
    **
    **  @param  loadPower    The name of the load power line
    **
    ***************************************************************************
    */
    public LongLines(String name, int tickMillis, int monMillis, int checkMillis,
                     String mainPower, String loadPower)
    {
        super(name, tickMillis);
        this.monMillis = monMillis;
        this.checkMillis = checkMillis;
        this.mainPower = mainPower;
        this.loadPower = loadPower;

        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  General initialization
        */
        subsys = (ConfigurableSubsystem)getSubsystem();

        /*
        **  Initialize all configuration data
        */
        mon = new Monitor(this, this, sLog);
        mon.initConfiguration();
        
        /*
        **  Add monitoring commands
        */
        subsys.addCommandsFromObject(new MonitorCommands(mon), "");

        /*
        **  Get output lines to be used
        */
        mainPowerLine = mon.getLine(mainPower);
        if (mainPowerLine == null) {
            sLog.error("Main power line not defined");
        }
        loadPowerLine = mon.getLine(loadPower);
        if (loadPowerLine == null) {
            sLog.error("Load power line not defined");
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        System.out.println("checkHardware was called");

        /*
        **  Initialize the hardware
        */
        if (mainPowerLine != null) {
            mainPowerLine.setWarm(!coldStart);
        }
        mon.initSensors(checkMillis);
        if (mainPowerLine != null) {
            mainPowerLine.setWarm(false);
        }

        /*
        **  Set output lines
        */
        setMainPowerEnable(isMainPowerOn() ? 1 : 0);
        setLoadPower();

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
        System.out.println("checkStarted was called");
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
        System.out.println("checkStopped was called");
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    **  @throws  HardwareException
    **
    ***************************************************************************
    */
    @Override
    public void postStart() throws HardwareException
    {
        mon.start(monMillis);
        sLog.info("Refrigeration long lines test system started");
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        super.postStart();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    **  @param  event  The event type
    **
    **  @param  parm   The event parameter
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setMainPowerEnable(0);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
                }
            }
            break;
            
        case EVENT_ID_LOAD_POWER:
            if (event == Alarm.EVENT_TRIP) {
                loadAlarm = true;
                if ((state & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    state &= ~RefrigState.LOAD_POWER_STATE;
                    setLoadPower();
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                loadAlarm = false;
                if ((state & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    state |= RefrigState.LOAD_POWER_STATE;
                    setLoadPower();
                    publishState();
                }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the main power enabled state")
    public void setMainPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
            }
        }
        else {
            state &= ~RefrigState.MAIN_POWER_STATE;
        }
        setMainPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power trip enabled state on or off.
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the load power trip enabled state")
    public void setLoadTripEnable(int value)
    {
        if (value != 0) {
            state |= RefrigState.LOAD_TRIP_ENAB_STATE;
            if (loadAlarm) {
                state &= ~RefrigState.LOAD_POWER_STATE;
                setLoadPower();
            }
        }
        else {
            state &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
            if ((state & RefrigState.LOAD_POWER_STATE) == 0) {
                state |= RefrigState.LOAD_POWER_STATE;
                setLoadPower();
            }
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    **  @return  The full refrigeration state
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Get the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis(), 0, 0);
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Gets the operating state word.
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    int getState()
    {
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        RefrigState rState = new RefrigState(state, getTickMillis(), 0, 0);
        KeyValueData data = new KeyValueData(RefrigState.KEY, rState);
        subsys.publishSubsystemDataOnStatusBus(data);
    }    


   /**
    ***************************************************************************
    **
    **  Sets the main power on or off, according to its state.
    **
    ***************************************************************************
    */
    void setMainPower()
    {
        if (mainPowerLine != null) {
            mainPowerLine.set((state & RefrigState.MAIN_POWER_STATE) != 0);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the load power on or off, according to its state.
    **
    ***************************************************************************
    */
    void setLoadPower()
    {
        if (loadPowerLine != null) {
            loadPowerLine.set((state & RefrigState.LOAD_POWER_STATE) != 0);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the hardware main power state
    **
    ***************************************************************************
    */
    boolean isMainPowerOn()
    {
        return mainPowerLine != null && mainPowerLine.isSet();
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state
    **
    ***************************************************************************
    */
    boolean isLoadPowerOn()
    {
        return loadPowerLine != null && loadPowerLine.isSet();
    }
    
}
