package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.MonitorCommands;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration compressor modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Compressor extends Module
                        implements HardwareController, Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        TICK_INTERVAL = 10000,
        EVENT_ID_MAIN_POWER = 0;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.refrig");

    private int monMillis = 1000;
    private int checkMillis = 0;
    private String cmprPowerLine;
    private String heaterPowerLine;
    private String resetLine;

    private boolean coldStart, running;
    private ConfigurableSubsystem subsys;
    private Line cmprPowerLineL, heaterPowerLineL, resetLineL;
    private Monitor mon;
    private int state = 0;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name             The subsystem name.
    **
    **  @param  tickMillis       The tick period (ms).
    **
    **  @param  monMillis       The monitor period (ms).
    **
    **  @param  checkMillis      The hardware check period (ms).
    **
    **  @param  compPowerLine    The name of the compressor power line.
    **
    **  @param  heaterPowerLine  The name of the heater power line.
    **
    **  @param  resetLine        The name of the PLC reset line
    **
    ***************************************************************************
    */
    public Compressor(String name, int tickMillis, int monMillis, int checkMillis,
                      String compPowerLine, String heaterPowerLine, String resetLine)
    {
        super(name, tickMillis);
        this.monMillis = monMillis;
        this.checkMillis = checkMillis;
        cmprPowerLine = compPowerLine;
        this.heaterPowerLine = heaterPowerLine;
        this.resetLine = resetLine;
    }


    public Compressor() {
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  General initialization
        */
        setName("main");
        if (getTickMillis() <= 0) {
            setTickMillis(TICK_INTERVAL);
        }
        subsys = (ConfigurableSubsystem)getSubsystem();

        /*
        **  Get cold start option
        */
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");

        /*
        **  Initialize component references
        */
        if (cmprPowerLine != null) {
            cmprPowerLineL = (Line)getComponent(cmprPowerLine, Line.class);
        }
        if (heaterPowerLine != null) {
            heaterPowerLineL = (Line)getComponent(heaterPowerLine, Line.class);
        }
        if (resetLine != null) {
            resetLineL = (Line)getComponent(resetLine, Line.class);
        }

        if (cmprPowerLineL == null) {
            sLog.error("Compressor power line not specified");
        }
        if (heaterPowerLineL == null) {
            sLog.error("Heater power line not specified");
        }
        if (resetLineL == null) {
            sLog.error("Reset line not specified");
        }
        
        /*
        **  Initialize all monitoring configuration data
        */
        mon = new Monitor(this, this, sLog);
        mon.initConfiguration();

        /*
        **  Add monitoring commands
        */
        subsys.addCommandsFromObject(new MonitorCommands(mon), "");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        /*
        **  Set power control lines' warm start option
        */
        if (cmprPowerLineL != null) {
            cmprPowerLineL.setWarm(!coldStart);
        }
        if (heaterPowerLineL != null) {
            heaterPowerLineL.setWarm(!coldStart);
        }

        /*
        **  Initialize the hardware
        */
        mon.initSensors(checkMillis);

        /*
        **  Turn off the power control lines' warm start option
        */
        if (cmprPowerLineL != null) {
            cmprPowerLineL.setWarm(false);
        }
        if (heaterPowerLineL != null) {
            heaterPowerLineL.setWarm(false);
        }

        /*
        **  Make sure state and power control lines are consistent
        */
        setCompPowerEnable(isCompPowerOn() ? 1 : 0);
        setHeaterPowerEnable(isHeaterPowerOn() ? 1 : 0);

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
        System.out.println("checkStarted was called");
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
        System.out.println("checkStopped was called");
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    **  @throws  HardwareException
    **
    ***************************************************************************
    */
    @Override
    public void postStart() throws HardwareException
    {
        running = true;
        mon.start(monMillis);
        sLog.info("Refrigeration subsystem (" + subsys.getName() + ") started");
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        super.postStart();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    **  @param  event  The event type
    **
    **  @param  parm   The event parameter
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setCompPowerEnable(0);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
                }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the compressor power state.
    **
    **  @param  value    The enabled state value to set: 0 = off,
    **                   not 0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the main power enabled state")
    public void setCompPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
            }
        }
        else {
            state &= ~(RefrigState.MAIN_POWER_STATE
                        | RefrigState.MAIN_LOAD_ONLY_STATE);
        }
        setCompPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the heater power state.
    **
    **  @param  value  The heater power state to set: 0 = off; not 0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the heater power enabled state")
    public void setHeaterPowerEnable(int value)
    {
        if (value != 0) {
            state |= RefrigState.LOAD_POWER_STATE;
        }
        else {
            state &= ~RefrigState.LOAD_POWER_STATE;
        }
        setHeaterPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Resets the latches.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Reset the latches")
    public void reset()
    {
        if (resetLineL != null) {
            resetLineL.set(true);
            try {
                Thread.sleep(200);
            }
            catch (InterruptedException e) {
            }
            resetLineL.set(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    **  @return  The full refrigeration state
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Get the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis(), 0, 0);
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    private void publishState()
    {
        RefrigState rState = new RefrigState(state, getTickMillis(), 0, 0);
        KeyValueData data = new KeyValueData(RefrigState.KEY, rState);
        subsys.publishSubsystemDataOnStatusBus(data);
    }    


   /**
    ***************************************************************************
    **
    **  Sets the compressor power on or off.
    **
    ***************************************************************************
    */
    private void setCompPower()
    {
        boolean on = (state & RefrigState.MAIN_POWER_STATE) != 0;
        if (cmprPowerLineL != null) {
            cmprPowerLineL.set(on);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the heater power on or off.
    **
    ***************************************************************************
    */
    private void setHeaterPower()
    {
        boolean on = (state & RefrigState.LOAD_POWER_STATE) != 0;
        if (heaterPowerLineL != null) {
            heaterPowerLineL.set(on);
        }
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the hardware compressor power state
    **
    ***************************************************************************
    */
    private boolean isCompPowerOn()
    {
        return cmprPowerLineL != null && cmprPowerLineL.isSet();
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware heater power state.
    **
    ***************************************************************************
    */
    private boolean isHeaterPowerOn()
    {
        return heaterPowerLineL != null && heaterPowerLineL.isSet();
    }


   /**
    ***************************************************************************
    **
    **  Gets the named component of a specified class.
    **
    ***************************************************************************
    */
    private Object getComponent(String name, Class cls)
    {
        Object cmpt = getComponentByName(name);
        return cls.isInstance(cmpt) ? cmpt : null;
    }

}
