package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles a Pluto PLC system.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class PlutoDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private static final int
        TYPE_GLOBAL  = 0,
        TYPE_ADD_BIT = 1,
        TYPE_ADD_REG = 2;

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("GLOBAL", TYPE_GLOBAL);
        typeMap.put("ADDBIT", TYPE_ADD_BIT);
        typeMap.put("ADDREG", TYPE_ADD_REG);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final String node;          // Network node name
    private final List<Integer> areas;  // Area configuration data
    private Pluto plu;                  // Associated Pluto object
    private final int[] globData = new int[Pluto.NUM_MODULES];
    private final int[] addData = new int[Pluto.NUM_ADD_AREAS];
    private final int nArea;
    private final Set<Integer> modsUsed = new HashSet<>();
    private final Set<Integer> areasUsed = new HashSet<>();

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  nodeName  The node name of the MAQ20 system
    **
    **  @param  areaCfg   The additional area configuration data
    **
    ***************************************************************************
    */
    public PlutoDevice(String nodeName, List areaCfg)
    {
        node = nodeName;
        areas = areaCfg;
        nArea = areas.size() / 2;
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        fullName = "Pluto PLC system (" + node + ")";
        if ((areas.size() & 1) != 0) {
            mon.reportConfigError(fullName, "area data",
                                  "has odd number of elements");
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        try {
            if (!inited || plu == null) {
                plu = new Pluto();
            }
            plu.open(node);
            plu.configStart(-1, 15, 0, 100);
            for (int j = 0; j < areas.size(); j += 2) {
                plu.configDataArea(j / 2, areas.get(j), areas.get(j + 1));
            }
            plu.configWrite();
            setOnline(true);
            initSensors();
            setOutputLines();
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (plu != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            plu.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype)
        throws Exception
    {
        Integer iType = null;
        int id;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            iType = typeMap.get(typeFields[0].toUpperCase());
        }
        try {
            id = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            id = -1;
        }
        int maxId = 0;
        if (iType != null) {
            maxId = iType == TYPE_GLOBAL ? Pluto.NUM_MODULES : nArea;
            int maxChan = iType == TYPE_GLOBAL ? 32
                                               : iType == TYPE_ADD_BIT ? 16 : 2;
            if (hwChan < 0 || hwChan >= maxChan) {
                mon.reportError(name, "hwchan", hwChan);
            }
        }
        if (iType == null || id < 0 || id >= maxId) {
            mon.reportError(name, "type", type);
        }
        Set used = iType == TYPE_GLOBAL ? modsUsed : areasUsed;
        used.add(id);

        return new int[]{(iType << 8) | id, 0};
    }


   /**
    ***************************************************************************
    **
    **  Reads all referenced channels.
    **
    ***************************************************************************
    */
    @Override
    protected void readChannelGroup()
    {
        try {
            for (int id : modsUsed) {
                globData[id] = plu.readGlobalData(id);
            }
            for (int id : areasUsed) {
                addData[id] = plu.readAdditionalData(id);
            }
        }
        catch (DriverException e) {
            log.error("Error reading " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        int index = type & 0xff;

        switch (type >> 8) {

        case TYPE_GLOBAL:
            value = (globData[index] >> hwChan) & 1;
            break;

        case TYPE_ADD_BIT:
            value = (addData[index] >> hwChan) & 1;
            break;

        case TYPE_ADD_REG:
            value = (addData[index] >> (16 * hwChan)) & 0xffff;
            break;
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Checks an output line number.
    **
    ***************************************************************************
    */
    @Override
    protected void checkHwLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= 16 * Pluto.NUM_DTP_AREAS) {
            mon.reportError(name, "line", line);
        }
        addLine(line);
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    ***************************************************************************
    */
    @Override
    protected void setHwLine(int id, boolean on)
    {
        writeBit(id / 16, id & 0x0f, on ? 1 : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set state of an output line.
    **
    ***************************************************************************
    */
    @Override
    protected boolean isHwLineSet(int id)
    {
        return readBit(id / 16, id & 0x0f) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Writes a bit value.
    **
    **  @param  area   The area number
    **
    **  @param  bit    The bit number
    **
    **  @param  value  The bit value
    **
    ***************************************************************************
    */
    public void writeBit(int area, int bit, int value)
    {
        try {
            plu.writeAreaBit(area, bit, value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a bit value.
    **
    **  @param  area   The area number
    **
    **  @param  bit    The bit number
    **
    **  @return  The bit value
    **
    ***************************************************************************
    */
    public int readBit(int area, int bit)
    {
        try {
            return plu.readAreaBit(area, bit);
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes a register value.
    **
    **  @param  area   The area number
    **
    **  @param  reg    The register number
    **
    **  @param  value  The register value
    **
    ***************************************************************************
    */
    public void writeRegister(int area, int reg, int value)
    {
        try {
            plu.writeAreaRegister(area, reg, value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }

}
