package org.lsst.ccs.subsystem.refrig;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

/**
 * File: PITest.java. 
 * <p> Description: This set of classes implements a PI
 * controller, simulates a load to be cooled, and a refrigerator to cool the
 * load.
 *
 * @author Innes
 * @since 3 APR 2014
 */


/**
 * Class: PITest. This class runs a simulation and writes the result.
 */
public class PITest {

   /**
    * This main program runs a simulation and records the result.
    *
    * @param args The output file name
    * @throws IOException
    */
    public static void main(String[] args) throws IOException {

        if (args.length < 1) {
            System.out.println("No output file specified");
            System.exit(1);
        }
        FileWriter out = new FileWriter(new File(args[0]));
        int steps = 400;  // number of iterations
        double sTmp = 290.; // initial load tmeperature (K)
        double sTa = 290.;  // ambient temperature (K)
        double sGn = 10.;    // PI loop gain 
        double sTc = 5000.; // integration time constant (s)
        double sSm = 200.;  // input smoothing time constant (s)
        double sTime = 0.;  // starting time (s)
        Refrigerator myFrig = new Refrigerator();
        LoadSim myLoad = new LoadSim(sTmp, sTa, sTime, myFrig);
        PIController myPI = new PIController(sGn, sTc);
        myPI.setSmoothTime(sSm);
        double[] mTmp = new double[1];
        mTmp[0] = sTmp;
        String outText =
                "Time, Temperature, Smoothed, Heat, Integral, Rout \r\n";
        out.write(outText);
        for (int j = 0; j < steps; j++) {
            double tm = 100. * j +sTime;
            double hout = myPI.performPI(mTmp, tm);
            mTmp[0] = myLoad.newT(hout, tm);
            double smT = myPI.getSmooth();
            double intE = myPI.getIntegral();
            double outR = myFrig.getRout();
            outText =
                    tm + ", " + mTmp[0] + "," + smT + ", " + hout + ", " 
                    + intE + ", " + outR + "\r\n";
            out.write(outText);
        }
    }

   /**
    * Class: LoadSim.
    *
    * Description: This LoadSim class simulates the behavior of a refrigerated
    * mass in a vacuum. It includes heat conduction, radiative heat
    * absorption, refrigeration, and heat capacity. It uses an instance of the
    * class Refrigerator to determine the refrigeration level. The setting of a
    * heater is provided in the call and derives from an instance of the
    * PIController class.
    *
    * @author Innes
    */
    static class LoadSim {

        private double C = 9608.;     // heat capacity (J/K)
        private double k = 0.1;        // heat leak impedance (J/K)
        private double A = 0.3;       // area (m^2)
        private final double sigma = 5.67E-8; // Stefan-Boltzman const. (W m−2 K−4)
        private double epsl = 0.06;    // emissivity of load (default for Cu)
        private double epsc = 0.09;   // emissivity of the cryostat (default for Al)
        private double T = 281.;        // temperature (K)
        private double Ta = 281.;      // ambient temperature (K)
        private double t = 0.;         // time (s)
        private double tMax = 10.;     // maximum time integration step (s)
        private Refrigerator refrig;   // the refrigerator instance  

       /**
        * Allocate a LoadSim object with the initial values for temperature and
        * the ambient temperature
        *
        * @param init_T the initial load temperature (K)
        * @param init_Ta the ambient temperature (K)
        */
        public LoadSim(double init_T, double init_Ta,
                double init_t, Refrigerator frig) {
            T = init_T;
            Ta = init_Ta;
            // P = init_P;
            t = init_t;
            refrig = frig;
        }

       /**
        * Return a value for the load temperature at time tNew given the heater
        * power P.
        *
        * @param P the heater power (W)
        * @param tNew the time (s)
        */
        public double newT(double P, double tNew) {
        //        Told = T ;
            int n = (int) ((tNew - t) / tMax);
            double dt = (tNew - t) / (n + 1);
            int i = 0;
            do {
                double Rt = refrig.Refrig(T, Ta);
                //         System.out.println(dt+", "+P+", "+T+", "+Ta+", "+Rt);
                T = T + (dt / C) * (P
                        + k * (Ta - T)
                        + ((epsl * epsc)/(epsl+epsc-epsl*epsc)) * sigma * A 
                        * (Ta * Ta * Ta * Ta - T * T * T * T)
                        + refrig.Refrig(T, Ta));
                i++;
            } while (i < n);
            t = tNew;
            return T;
        }

       /**
        * Set the heat capacity of the load (J/K)
        *
        * @param sC the heat capacity of the load (J/K)
        */
        public void setC(double sC) {
            C = sC;
        }

       /**
        * Set the heat conduction coefficient (J/K)
        *
        * @param sk the heat conduction coefficient (J/K)
        */
        public void setk(double sk) {
            k = sk;
        }

       /**
        * Set the surface area of the load (m*m)
        *
        * @param sA the surface area of the load (m*m)
        */
        public void setA(double sA) {
            A = sA;
        }

       /**
        * Set the ambient temperature (K)
        *
        * @param sTa the ambient temperature (K)
        */
        public void setTa(double sTa) {
            Ta = sTa;
        }

       /**
        * Set the maximum time interval for integration (s)
        *
        * @param stMax the maximum time interval for integration (s)
        */
        public void settMax(double stMax) {
            tMax = stMax;
        }

       /**
        * Set the emissivity of the load
        *
        * @param sepsl the emissivity of the load
        */
        public void setepsl(double sepsl) {
            epsl = sepsl;
        }

     
       /**
        * Set the emissivity of the cryostat
        *
        * @param sepsc the emissivity of the cryostat
        */
        public void setepsc(double sepsc) {
            epsc = sepsc;
        }

       /**
        * Return C the heat capacity of the load (J/K)
        *
        * @return C the heat capacity of the load (J/K)
        */
        public double getC() {
            return C;
        }

       /**
        * Return k the heat conduction coefficient ((J/K)
        *
        * @return k the heat conduction coefficient ((J/K)
        */
        public double getk() {
            return k;
        }

       /**
        * Return Ta the ambient temperature (K)
        *
        * @return tMax the ambient temperature (K)
        */
        public double getA() {
            return A;
        }

       /**
        * Return Ta the ambient temperature (K)
        *
        * @return tMax the ambient temperature (K)
        */
        public double getTa() {
            return Ta;
        }

       /**
        * Return tMax the maximum integration time step (s)
        *
        * @return tMax the maximum integration time step (s)
        */
        public double gettMax() {
            return tMax;
        }

       /**
        * Return the load emissivity
        *
        * @return epsl the load emissivity
        */
        public double getepsl() {
            return epsl;
        }

       /**
        * Return the cryostat emissivity
        *
        * @return epsc the cryostat emissivity
        */
        public double getepsc() {
            return epsc;
        }

    }


   /**
    * Class: Refrigerator.
    *
    * Description: This Refrigerator class simulates the behavior of a
    * refrigerator. It uses a lookup table and interpolation to determine
    * the rate of heat removal from the load given the load temperature.
    * As currently implemented, it does not take into account the temperature of
    * the refrigerator itself. The table assumes 290 K for this.
    *
    * @author Innes
    */
    static class Refrigerator {

        private double R[][] = {
            {0., 105.},
            {13., 106.8},
            {32., 109.3},
            {51., 111.8},
            {72., 115.8},
            {86., 120.7},
            {92., 123.},
            {96., 126.7},
            {101., 130.7},
            {104., 135.7},
            {106., 143.7},
            {109., 153.6},
            {110., 154.8},
            {111., 164.4},
            {112., 180.},
            {113., 188.8},
            {116., 194.8},
            {126., 202.1},
            {146., 210.},
            {155., 213.8},
            {165., 220.},
            {400., 289.}
        }; // refrigerator performance: {heat removal (W), temperature (K)}
        private double Rout; // output heat removal (W)
        private double Tanom = 290.; // ambient temperature for the table (K)
        private int len; // number of entry pairs in the table less 1.

       /**
        * Allocate a Refrigerator object.
        */
        public Refrigerator() {
            len = R.length - 1;
        }

       /**
        * Return a value for the refrigerator heat removal rate (W).
        *
        * @param Tl the load temperature (W)
        * @param Th the refrigerator temperature (W) (ignored for now)
        */
        public double Refrig(double Tl, double Th) {
            if (Tl <= R[0][1]) { // temperature too low (use lowest value)
                Rout = -R[0][0];
            } else if (Tl >= R[len][1]) {// temperature too high (use highest value)
                Rout = -R[len][0];
            } else {              // temperature in range (use linear interpolation)
                for (int j = 1; j <= len; j++) {
                    if ((Tl > R[j - 1][1]) && (Tl <= R[j][1])) {
                        Rout = -((Tl - R[j - 1][1]) * R[j][0]
                                + (R[j][1] - Tl) * R[j - 1][0])
                                / (R[j][1] - R[j - 1][1]);
                        break;
                    }
                }
            }
            return Rout;
        }

       /**
        * Returns the current refrigeration value (W)
        *
        * @return Rout the current refrigeration value (W)
        */
        public double getRout() {
            return Rout;
        }

    }

}
