package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.agilent.DL34970;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles an Agilent 34970A data logger.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class DL34970Device extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private static final int BAUD_RATE = 9600;
    private static final String tcTypes = "JKTERSBN";

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private static final Map<String,Integer> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", DL34970.CONN_TYPE_SERIAL);
        cTypeMap.put("FTDI",   DL34970.CONN_TYPE_FTDI);
    }

    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   Channel.TYPE_TEMP);
    }

    private final static Map<String,DL34970. TcType> tcMap = new HashMap<>();
    static {
        tcMap.put("J", DL34970.TcType.J);
        tcMap.put("K", DL34970.TcType.K);
        tcMap.put("T", DL34970.TcType.T);
        tcMap.put("E", DL34970.TcType.E);
        tcMap.put("R", DL34970.TcType.R);
        tcMap.put("S", DL34970.TcType.S);
        tcMap.put("B", DL34970.TcType.B);
        tcMap.put("N", DL34970.TcType.N);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final String connTypeS;  // Connection type string
    private final String devcId;     // Device parameter
    private final int baudRate;      // Baud rate
    private int connType;            // Connection type
    private DL34970 dl;              // Associated DL34970 object
    private int firstChan = 1000;    // First  channel to be read
    private int lastChan = -1;       // Last channel to be read
    private double chanValues[];     // Read channel data


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  connType   The device type (FTDI or serial)
    **
    **  @param  devcId   The device parameter (serial no. or device name)
    **
    **  @param  baudRate  The baud rate (0 for default)
    **
    ***************************************************************************
    */
    public DL34970Device(String connType, String devcId, int baudRate)
    {
        this.connTypeS = connType;
        this.devcId = devcId;
        this.baudRate = (baudRate == 0) ? BAUD_RATE : baudRate;
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    **  @param  mon  The monitor object
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        fullName = "DL34970 Logger (" + devcId + ")";
        Integer iConnType = cTypeMap.get(connTypeS);
        if (iConnType == null) {
            connType = -1;
            try {
                mon.reportError(getName(), "connection type", connTypeS);
            }
            catch (Exception e) {
            }
        }
        else {
            connType = iConnType;
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (connType < 0) return;
        try {
            if (!inited || dl == null) {
                dl = new DL34970();
            }
            dl.open(connType, devcId, baudRate);
            setOnline(true);
            initSensors();
            if (lastChan >= firstChan) {
                if (chanValues == null) {
                    chanValues = new double[lastChan + 1 - firstChan];
                }
                int[] chans = new int[chanValues.length];
                for (int j = 0; j < chans.length; j++) {
                    chans[j] = j + firstChan;
                }
                dl.setScanList(chans);
            }
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (dl != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            dl.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    **  This is called before the device has been initialized.
    **
    **  @param  name     The channel name
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @param  type     The channel type string
    **
    **  @param  subtype  The channel subtype string
    **
    **  @return  A two-element array containing the encoded type [0] and
    **           subtype [1] values.
    **
    **  @throws  Exception if any errors found in the parameters.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            mon.reportError(name, "type", type);
        }
        int iSubtype = subtype.length() == 1 ? tcTypes.indexOf(subtype) : -1;
        if (iSubtype < 0) {
            mon.reportError(name, "subtype", subtype);
        }
        firstChan = Math.min(hwChan, firstChan);
        lastChan = Math.max(hwChan, lastChan);

        return new int[]{iType, iSubtype};
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    **  @param  chan     The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @param  subtype  The encoded channel subtype returned by checkChannel.
    **
    ***************************************************************************
    */
    @Override
    protected void initChannel(int chan, int type, int subtype)
    {
        if (!online) return;
        try {
            if (type == Channel.TYPE_TEMP) {
                dl.configTC(tcMap.get(tcTypes.substring(subtype, subtype + 1)),
                            new int[]{chan});
            }
        }
        catch (DriverException e) {
            log.error("Error configuring " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads grouped channels.
    **
    ***************************************************************************
    */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            if (lastChan >= firstChan) {
                chanValues = dl.readData();
            }
        }
        catch (DriverException e) {
            log.error("Error reading data logger: " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  The value read from the channel
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        if (online) {
            return chanValues[hwChan - firstChan];
        }
        else {
            return super.readChannel(hwChan, type);
        }
    }

}
