package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration compressor modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Compressor extends Module
                        implements HardwareController, Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        EVENT_ID_MAIN_POWER = 0;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final String configName;
    private final boolean coldStart;
    private final int checkMillis;
    private final String compPowerLineS, heaterPowerLineS, resetLineS;

    private Line compPowerLine, heaterPowerLine, resetLine;
    private Monitor mon;
    private int state = 0;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name             The subsystem name.
    **
    **  @param  tickMillis       The tick period (ms).
    **
    **  @param  checkMillis      The hardware check period (ms).
    **
    **  @param  configName       The name of the configuration.
    **
    **  @param  compPowerLine    The name of the compressor power line.
    **
    **  @param  heaterPowerLine  The name of the heater power line.
    **
    **  @param  resetLine        The name of the PLC reset line
    **
    ***************************************************************************
    */
    public Compressor(String name, int tickMillis, int checkMillis,
                      String configName, String compPowerLine,
                      String heaterPowerLine, String resetLine)
    {
        super(name, tickMillis);
        this.checkMillis = checkMillis;
        this.configName = configName != null ? configName : "";
        compPowerLineS = compPowerLine;
        heaterPowerLineS = heaterPowerLine;
        resetLineS = resetLine;

        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize component references
        */
        if (compPowerLineS != null) {
            compPowerLine = (Line)getComponent(compPowerLineS, Line.class);
        }
        if (heaterPowerLineS != null) {
            heaterPowerLine = (Line)getComponent(heaterPowerLineS, Line.class);
        }
        if (resetLineS != null) {
            resetLine = (Line)getComponent(resetLineS, Line.class);
        }

        if (compPowerLine == null) {
            log.error("Compressor power line not specified");
        }
        if (heaterPowerLine == null) {
            log.error("Heater power line not specified");
        }
        if (resetLine == null) {
            log.error("Reset line not specified");
        }
        
        /*
        **  Initialize all monitoring configuration data
        */
        mon = new Monitor(this, this, log);
        mon.initConfiguration();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        /*
        **  Set power control lines' warm start option
        */
        if (compPowerLine != null) {
            compPowerLine.setWarm(!coldStart);
        }
        if (heaterPowerLine != null) {
            heaterPowerLine.setWarm(!coldStart);
        }

        /*
        **  Initialize the hardware
        */
        mon.initSensors(checkMillis);

        /*
        **  Turn off the power control lines' warm start option
        */
        if (compPowerLine != null) {
            compPowerLine.setWarm(false);
        }
        if (heaterPowerLine != null) {
            heaterPowerLine.setWarm(false);
        }

        /*
        **  Make sure state and power control lines are consistent
        */
        setCompPowerEnable(isCompPowerOn() ? 1 : 0);
        setHeaterPowerEnable(isHeaterPowerOn() ? 1 : 0);

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
        System.out.println("checkStarted was called");
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
        System.out.println("checkStopped was called");
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void startTicking()
    {
        mon.start();
        super.startTicking();
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        System.out.println("Refrigeration compressor test started");
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    **  @param  event  The event type
    **
    **  @param  parm   The event parameter
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setCompPowerEnable(0);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
                }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the compressor power state.
    **
    **  @param  value    The enabled state value to set: 0 = off,
    **                   not 0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the main power enabled state")
    public void setCompPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
            }
        }
        else {
            state &= ~(RefrigState.MAIN_POWER_STATE
                        | RefrigState.MAIN_LOAD_ONLY_STATE);
        }
        setCompPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the heater power state.
    **
    **  @param  value  The heater power state to set: 0 = off; not 0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Set the heater power enabled state")
    public void setHeaterPowerEnable(int value)
    {
        if (value != 0) {
            state |= RefrigState.LOAD_POWER_STATE;
        }
        else {
            state &= ~RefrigState.LOAD_POWER_STATE;
        }
        setHeaterPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Resets the latches.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Reset the latches")
    public void reset()
    {
        if (resetLine != null) {
            resetLine.set(true);
            try {
                Thread.sleep(200);
            }
            catch (InterruptedException e) {
            }
            resetLine.set(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Save the current configuration")
    public void saveConfig() throws IOException
    {
        getEnvironment().saveConfiguration();
        mon.clearLimitChanges();
        mon.publishState();     // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    **  @return  The full refrigeration state
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Get the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis(), 0, 0);
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    private void publishState()
    {
        publish(RefrigState.KEY, new RefrigState(state, getTickMillis(), 0, 0));
    }    


   /**
    ***************************************************************************
    **
    **  Sets the compressor power on or off.
    **
    ***************************************************************************
    */
    private void setCompPower()
    {
        boolean on = (state & RefrigState.MAIN_POWER_STATE) != 0;
        if (compPowerLine != null) {
            compPowerLine.set(on);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the heater power on or off.
    **
    ***************************************************************************
    */
    private void setHeaterPower()
    {
        boolean on = (state & RefrigState.LOAD_POWER_STATE) != 0;
        if (heaterPowerLine != null) {
            heaterPowerLine.set(on);
        }
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the hardware compressor power state
    **
    ***************************************************************************
    */
    private boolean isCompPowerOn()
    {
        return compPowerLine != null && compPowerLine.isSet();
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware heater power state.
    **
    ***************************************************************************
    */
    private boolean isHeaterPowerOn()
    {
        return heaterPowerLine != null && heaterPowerLine.isSet();
    }


   /**
    ***************************************************************************
    **
    **  Gets the named component of a specified class.
    **
    ***************************************************************************
    */
    private Object getComponent(String name, Class cls)
    {
        Object cmpt = getComponentByName(name);
        return cls.isInstance(cmpt) ? cmpt : null;
    }

}
