package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.subsystem.monitor.Device;

/**
 ******************************************************************************
 **
 **  Simulates the refrigeration hardware.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class SimDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        MAIN_POWER_LINE = 0,
        LOAD_POWER_LINE = 1,
        N_DEV_CHANS = 64,
        N_DEV_LINES = 8;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final static double LOAD_POWER_FCTR = 0.0006667;
    private final ChannelSim[] chanData = new ChannelSim[N_DEV_CHANS];
    private final String loadTmpChanS, loadPwrChanS;

    private static SimDevice sim;
    private static Power power;
    private ChannelSim loadTmpChan, loadPwrChan;
    private double loadFract = 0.6;
    private final boolean[] lineOn = new boolean[N_DEV_LINES];


   /**
    ***************************************************************************
    **
    **  Inner class to simulate power supply device with attached load.
    **
    ***************************************************************************
    */
    static class Power extends PowerDevice {

        private final static int
            N_DEV_CHANS = 3,
            CHAN_WATTS = 0,
            CHAN_VOLTS = 1,
            CHAN_AMPS  = 2;
        private final static double
            LOAD_OHMS = 50;

        private double setVolts, setAmps, loadVolts, loadAmps, loadWatts;
        private boolean outputOn, powerOn;

       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public Power()
        {
            fullName = "Simulated power supply";
            power = this;
            disabled = true;
        }


       /**
        ***********************************************************************
        **
        **  Performs basic initialization.
        **
        ***********************************************************************
        */
        @Override
        public void initialize()
        {
            if (checkOn()) {
                setCurrent(0, 3.0);
                setOnline(true);
                initSensors();
            }
        }


       /**
        ***********************************************************************
        **
        **  Closes the connection.
        **
        ***********************************************************************
        */
        @Override
        public void close()
        {
        }


       /**
        ***********************************************************************
        **
        **  Checks a channel's parameters for validity.
        **
        ***********************************************************************
        */
        @Override
        public int[] checkChannel(String name, int hwChan, String type,
                                  String subtype) throws Exception
        {
            if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
                mon.reportError(name, "hw channel number", hwChan);
            }
                    
            return new int[]{0, 0};
        }


       /**
        ***********************************************************************
        **
        **  Reads a channel.
        **
        ***********************************************************************
        */
        @Override
        public double readChannel(int hwChan, int type)
        {
            checkOn();
            switch (hwChan) {
            case CHAN_WATTS:
                return loadWatts;
            case CHAN_VOLTS:
                return loadVolts;
            case CHAN_AMPS:
                return loadAmps;
            default:
                return 0;
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets the power supply voltage.
        **
        ***********************************************************************
        */
        @Override
        public void setVoltage(int chan, double value)
        {
            if (checkOn()) {
                setVolts = value;
                setPower();
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets the power supply current.
        **
        ***********************************************************************
        */
        @Override
        public void setCurrent(int chan, double value)
        {
            if (checkOn()) {
                setAmps = value;
                setPower();
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets the power supply output state.
        **
        ***********************************************************************
        */
        @Override
        public void setOutput(int chan, boolean value)
        {
            if (checkOn()) {
                outputOn = value;
                setPower();
            }
        }


       /**
        ***********************************************************************
        **
        **  Gets the power supply voltage.
        **
        ***********************************************************************
        */
        @Override
        public double readVoltage(int chan)
        {
            checkOn();
            return loadVolts;
        }


       /**
        ***********************************************************************
        **
        **  Gets the power supply current.
        **
        ***********************************************************************
        */
        @Override
        public double readCurrent(int chan)
        {
            checkOn();
            return loadAmps;
        }


       /**
        ***********************************************************************
        **
        **  Gets the power supply output state.
        **
        ***********************************************************************
        */
        @Override
        public boolean getOutput(int chan)
        {
            checkOn();
            return outputOn;
        }


       /**
        ***********************************************************************
        **
        **  Turns the power supply on or off.
        **
        ***********************************************************************
        */
        void setOn(boolean on)
        {
            powerOn = on;
            if (!on) {
                outputOn = false;
                setVolts = 0;
                setAmps = 0;
            }
        }


       /**
        ***********************************************************************
        **
        **  Tests whether the power is on.
        **
        ***********************************************************************
        */
        boolean isOn()
        {
            return powerOn;
        }


       /**
        ***********************************************************************
        **
        **  Sets the load power values.
        **
        ***********************************************************************
        */
        private void setPower()
        {
            if (!outputOn) {
                loadVolts = 0;
                loadAmps = 0;
            }
            else {
                double amps = setVolts / LOAD_OHMS;
                if (amps > setAmps) {
                    amps = setAmps;
                }
                loadVolts = amps * LOAD_OHMS;
                loadAmps = amps;
            }
            loadWatts = loadVolts * loadAmps;
            sim.setLoadTmpRate(loadWatts);
        }


       /**
        ***********************************************************************
        **
        **  Checks whether the power is on.
        **
        ***********************************************************************
        */
        private boolean checkOn()
        {
            if (!powerOn) {
                log.error("Simulated power supply is turned off");
                setOnline(false);
            }
            return powerOn;
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner container class for channel simulation data.
    **
    ***************************************************************************
    */
    class ChannelSim {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        int      hwChan;      // HW channel number
        double   onValue;     // On (operating) value
        double   onTime;      // Time (secs) to reach on value
        double   offValue;    // Off (ambient) value
        double   offTime;     // Time (secs) to reach off value

        double   endValue;    // Ending value
        double   rate;        // Rate of change
        boolean  negative;    // True if nominal change is negative
        double   value;       // Current value
        long     currTime;    // Current system time


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelSim(int hwChan, double onValue, double onTime,
                          double offValue, double offTime)
        {
            this.hwChan   = hwChan;
            this.onValue  = onValue;
            this.onTime   = onTime;
            this.offValue = offValue;
            this.offTime  = offTime;
            currTime      = System.currentTimeMillis();
        }


       /**
        ***********************************************************************
        **
        **  Initializes the channel.
        **
        ***********************************************************************
        */
        void initialize()
        {
            if (this == loadPwrChan) {
                onTime = 0;
                offValue = 0;
                offTime = 0;
            }
            endValue = offValue;
            value = offValue;
        }


       /**
        ***********************************************************************
        **
        **  Reads the channel.
        **
        ***********************************************************************
        */
        void read()
        {
            if (this == loadPwrChan && power != null) {
                if (power.isOn()) {
                    value = power.readVoltage(0) * power.readCurrent(0);
                }
                else {
                    value = Double.NaN;
                }
            }
            else {
                long thisTime = System.currentTimeMillis();
                double intvl = (thisTime - currTime) / 1000.0;
                currTime = thisTime;
                value += rate * intvl;
                if (negative) {
                    if (value < endValue) {
                        value = endValue;
                    }
                }
                else {
                    if (value > endValue) {
                        value = endValue;
                    }
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets channel to power-on or power-off value.
        **
        ***********************************************************************
        */
        void powerOn()
        {
            currTime = System.currentTimeMillis();
            rate = 0;
            if (lineOn[MAIN_POWER_LINE]) {
                endValue = onValue;
                if (onTime != 0) {
                    rate = (onValue - offValue) / onTime;
                }
            }
            else {
                endValue = offValue;
                if (offTime != 0) {
                    rate = (offValue - onValue) / offTime;
                }
            }
            if (rate == 0) {
                value = endValue;
            }
            negative = rate < 0;
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  loadTmpChan  The name of the load temperature channel
    **
    **  @param  loadPwrChan  The name of the load power channel
    **
    ***************************************************************************
    */
    public SimDevice(String loadTmpChan, String loadPwrChan)
    {
        loadTmpChanS = loadTmpChan;
        loadPwrChanS = loadPwrChan;
        fullName = "Simulated devices";
        sim = this;
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (loadTmpChan == null) {
            log.error("No load temperature channel defined");
        }
        if (power == null && loadPwrChan == null) {
            log.error("No load power channel defined");
        }
        setOnline(true);
        initSensors();
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    **  @param  name     The name of the channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The channel type string.
    **
    **  @param  subtype  The channel subtype string.
    **
    **  @return  A two-element array containing the encoded type [0] and
    **           subtype [1].
    **
    **  @throws  Exception if any errors found in the parameters
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            mon.reportError(name, "hw channel number", hwChan);
        }

        String flds[] = subtype.split(":");
        try {
            double onValue = Double.valueOf(flds[0]);
            double onTime = Double.valueOf(flds[1]);
            double offValue = Double.valueOf(flds[2]);
            double offTime = Double.valueOf(flds[3]);
            ChannelSim chan = new ChannelSim(hwChan, onValue, onTime,
                                             offValue, offTime);
            chanData[hwChan] = chan;
            if (name.equals(loadTmpChanS)) {
                loadTmpChan = chan;
            }
            else if (name.equals(loadPwrChanS)) {
                loadPwrChan = chan;
            }
        }
        catch (Exception e) {
            mon.reportError(name, "subtype", subtype);
        }
                
        return new int[]{0, 0};
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @param  subtype  The channel subtype returned by checkChannel.
    **
    ***************************************************************************
    */
    @Override
    protected void initChannel(int hwChan, int type, int subtype)
    {
        chanData[hwChan].initialize();
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  The value read from the channel
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        ChannelSim chan = chanData[hwChan];
        chan.read();

        return chan.value;
    }


   /**
    ***************************************************************************
    **
    **  Checks an output line number.
    **
    **  @param  name  The name of the output line.
    **
    **  @param  line  The hardware line number of the output line.
    **
    **  @throws  Exception if the line number is invalid.
    **
    ***************************************************************************
    */
    @Override
    protected void checkHwLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= N_DEV_LINES) {
            mon.reportError(name, "line number", line);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    **  @param  line  The output line number.
    **
    **  @param  on    The on state to set: true or false
    **
    ***************************************************************************
    */
    @Override
    protected void setHwLine(int line, boolean on)
    {
        lineOn[line] = on;
        System.out.print("Line states:");
        for (boolean lon : lineOn) {
            System.out.print(lon ? " on" : " off");
        }
        System.out.println();
        if (line == MAIN_POWER_LINE) {
            setMainPower();
        }
        else if (line == LOAD_POWER_LINE) {
            setLoadPower();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the set state of an output line.
    **
    **  @param  line  The output line number.
    **
    **  @return  Whether or not the line is set.
    ** 
    ***************************************************************************
    */
    @Override
    protected boolean isHwLineSet(int line)
    {
        return lineOn[line];
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power fraction
    **
    **  @param  fract  The fraction of the maximum load power to use
    **
    ***************************************************************************
    */
    public void setLoadFract(double fract)
    {
        loadFract = fract;
        setLoadPower();
    }


   /**
    ***************************************************************************
    **
    **  Adjusts the load temperature rate from the load power.
    **
    ***************************************************************************
    */
    void setLoadTmpRate(double loadPower)
    {
        ChannelSim tc = loadTmpChan;
        if (tc != null) {
            tc.powerOn();
            tc.rate += LOAD_POWER_FCTR * loadPower;
        }
    }

        
   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    private void setMainPower()
    {
        for (ChannelSim chan : chanData) {
            if (chan != null && chan != loadPwrChan) {
                chan.powerOn();
            }
        }
        setLoadPower();
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, setting the correct level.
    **
    ***************************************************************************
    */
    private void setLoadPower()
    {
        if (power != null) {
            power.setOn(lineOn[LOAD_POWER_LINE]);
        }
        else {
            if (loadPwrChan == null) return;
            double value = 0;
            if (lineOn[MAIN_POWER_LINE]) {
                if (lineOn[LOAD_POWER_LINE]) {
                    value = loadFract * loadPwrChan.onValue;
                }
            }
            setLoadTmpRate(value);
            loadPwrChan.endValue = value;
            loadPwrChan.value = value;
        }
    }

}
