package org.lsst.ccs.subsystem.refrig;

import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements a temperature controller for the refrigeration system.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TempControl extends ConfigurableComponent {

    final private double gain,              // loop gain
                         timeConst,         // integration time constant (secs)
                         smoothTime,        // input smoothing time (secs)   
                         maxOutput,         // maximum PID output (watts)
                         minOutput = 0.0,   // minimum PID output (watts)
                         maxInput = 100.0,   // maximum input
                         minInput = -200.0, // minimum input
                         awGain,            // anti-windup gain
                         basePower,         // base power input
                         tolerance;         // maximum on-target error (%)
    final private PIController pic;         // The PI controller
    final private int updateTime;           // The update time interval (msec)

    private PowerDevice powerDevc;          // The power device
    private List<Channel>tempChans;         // Temperature channels to use
    private double[] temps;                 // Temperatures to use
    private double lastPower;               // The last power value set
    private boolean running;                // True if loop is running
    private Timer loopTimer;                // Timer for running the loop
    private Logger log;                     // The logger


   /**
    ***************************************************************************
    **
    **  Inner class for control loop iteration.
    **
    ***************************************************************************
    */
    private class Iterate extends TimerTask {

        @Override
        public void run()
        {
            int index = 0;
            for (Channel tempChan : tempChans) {
                temps[index++] = tempChan.getValue();
            }
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(temps, tod);
            powerDevc.setPower(0, lastPower);
        }
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  gain        The loop gain
    **
    **  @param  timeConst   The time constant (sec)
    **
    **  @param  smoothTime  The smoothing time (sec)
    **
    **  @param  maxOutput   The maximum output power allowed (watts)
    **
    **  @param  awGain      The anti-windup gain
    **
    **  @param  basePower   The base power input (watts)
    **
    **  @param  tolerance   The maximum tolerated error (percent)
    **
    **  @param  updateTime  The update time interval (msec)
    **
    ***************************************************************************
    */
    public TempControl(double gain, double timeConst, double smoothTime,
                       double maxOutput, double awGain, double basePower,
                       double tolerance, int updateTime)
    {
        this.gain = gain;
        this.timeConst = timeConst;
        this.smoothTime = smoothTime;
        this.maxOutput = maxOutput;
        this.awGain    = awGain;
        this.basePower = basePower;
        this.tolerance = tolerance;
        this.updateTime = updateTime;

        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
}


   /**
    ***************************************************************************
    **
    **  Initializes the parameters.
    **
    **  @param  tempChans   The list of temperature channels used for input
    **
    **  @param  powerDevc   The power device being controlled
    **
    **  @param  log         The logger to use
    **
    ***************************************************************************
    */
    public void initialize(List tempChans, PowerDevice powerDevc, Logger log)
    {
        this.tempChans = tempChans;
        this.powerDevc = powerDevc;
        this.log = log;
        temps = new double[tempChans.size()];
    }


   /**
    ***************************************************************************
    **
    **  Sets the target temperature.
    **
    **  @param  value  The temperature to set
    **
    ***************************************************************************
    */
    public void setTemp(double value)
    {
        pic.setSetpoint(value);
   }


   /**
    ***************************************************************************
    **
    **  Resets the controller.
    **
    ***************************************************************************
    */
    public void reset()
    {
        pic.reset();
    }


   /**
    ***************************************************************************
    **
    **  Restarts the control loop.
    **
    ***************************************************************************
    */
    public void restart()
    {
        start(lastPower);
    }


   /**
    ***************************************************************************
    **
    **  Starts the control loop.
    **
    **  @param  power  The starting power value
    **
    ***************************************************************************
    */
    public void start(double power)
    {
        if (running) return;
        running = true;
        pic.reset();
        pic.setIntegral(power - basePower);
        loopTimer = new Timer(true);
        loopTimer.schedule(new Iterate(), 0L, updateTime);
    }


   /**
    ***************************************************************************
    **
    **  Stops the control loop.
    **
    ***************************************************************************
    */
    public void stop()
    {
        if (!running) return;
        running = false;
        loopTimer.cancel();
    }

}
