package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration subscale camera modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Subscale extends Module
                      implements HardwareController, Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        EVENT_ID_MAIN_POWER = 0;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final String configName;
    private final boolean coldStart;
    private final int checkMillis;
    private final String loadPowerDevcS;
    private final List cmprPowerLinesS, loadPowerLinesS, alarmDsabChansS,
                       loadTempChansS;
    private final String tempCtrlS;
    private final List<Line> cmprPowerLines = new ArrayList<>();
    private final List<Line> loadPowerLines = new ArrayList<>();
    private final List<Channel> alarmDsabChans = new ArrayList<>();
    private final List<Channel> loadTempChans = new ArrayList<>();

    private PowerDevice loadPowerDevc;
    private TempControl tempCtrl;
    private Monitor mon;
    private int state = RefrigState.LOAD_TRIP_ENAB_STATE;
    private double loadTemp, loadPower;
    private Double startPower = 0.0;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name            The subsystem name.
    **
    **  @param  tickMillis      The tick period (ms).
    **
    **  @param  checkMillis     The hardware check period (ms).
    **
    **  @param  configName      The name of the configuration.
    **
    **  @param  cmprPowerLines  The names of the compressor power lines.
    **
    **  @param  loadPowerLines  The names of the load power lines.
    **
    **  @param  loadPowerDevc   The name of the load power device.
    **
    **  @param  alarmDsabChans  The names of the channels whose alarms are to
    **                          be disabled during cool-down.
    **
    **  @param  loadTempChans   The names of the load temperature channels.
    **
    **  @param  tempCtrl        The name of the temperature controller.
    **
    ***************************************************************************
    */
    public Subscale(String name, int tickMillis, int checkMillis,
                    String configName, List cmprPowerLines,
                    List loadPowerLines, String loadPowerDevc,
                    List alarmDsabChans, List loadTempChans, String tempCtrl)
    {
        super(name, tickMillis);
        this.checkMillis = checkMillis;
        this.configName = configName != null ? configName : "";
        cmprPowerLinesS = cmprPowerLines;
        loadPowerLinesS = loadPowerLines;
        loadPowerDevcS = loadPowerDevc;
        alarmDsabChansS = alarmDsabChans;
        loadTempChansS = loadTempChans;
        tempCtrlS = tempCtrl;

        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize component references
        */
        loadPowerDevc = (PowerDevice)getComponent(loadPowerDevcS,
                                                  PowerDevice.class);
        for (Object chanS : cmprPowerLinesS) {
            Object cmpt = (chanS instanceof String)
                            ? getComponent((String)chanS, Line.class) : null;
            if (cmpt != null) {
                cmprPowerLines.add((Line)cmpt);
            }
        }
        for (Object chanS : loadPowerLinesS) {
            Object cmpt = (chanS instanceof String)
                            ? getComponent((String)chanS, Line.class) : null;
            if (cmpt != null) {
                loadPowerLines.add((Line)cmpt);
            }
        }
        for (Object chanS : alarmDsabChansS) {
            Object cmpt = (chanS instanceof String)
                            ? getComponent((String)chanS, Channel.class) : null;
            if (cmpt != null) {
                alarmDsabChans.add((Channel)cmpt);
            }
        }
        for (Object chanS : loadTempChansS) {
            Object cmpt = (chanS instanceof String)
                            ? getComponent((String)chanS, Channel.class) : null;
            if (cmpt != null) {
                loadTempChans.add((Channel)cmpt);
            }
        }
        tempCtrl = (TempControl)getComponent(tempCtrlS, TempControl.class);

        if (cmprPowerLines.isEmpty()) {
            log.error("No valid compressor power lines specified");
        }
        else if (cmprPowerLines.size() != cmprPowerLinesS.size()) {
            log.error("Some compressor power lines are invalid");
        }
        if (loadPowerLines.isEmpty()) {
            log.error("No valid load power lines specified");
        }
        else if (loadPowerLines.size() != loadPowerLinesS.size()) {
            log.error("Some load power lines are invalid");
        }
        if (loadPowerDevc == null) {
            log.error("No valid load power device specified");
        }
        if (alarmDsabChans.size() != alarmDsabChansS.size()) {
            log.error("Some alarm disable channels are invalid");
        }
        if (loadTempChans.size() != loadTempChansS.size()) {
            log.error("Some load temperature channels are invalid");
        }
        if (tempCtrl == null) {
            log.error("No valid temperature controller specified");
        }
        
        /*
        **  Initialize all monitoring configuration data
        */
        mon = new Monitor(this, this, log);
        mon.initConfiguration();
    
        /*
        **  Initialize the temperature controller
        */
        if (tempCtrl != null) {
            tempCtrl.initialize(loadTempChans, loadPowerDevc, log);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        /*
        **  Set power control lines' warm start option
        */
        for (Line line : cmprPowerLines) {
            line.setWarm(!coldStart);
        }
        for (Line line : loadPowerLines) {
            line.setWarm(!coldStart);
        }

        /*
        **  Initialize the hardware
        */
        mon.initSensors(checkMillis);

        /*
        **  Turn off the power control lines' warm start option
        */
        for (Line line : cmprPowerLines) {
            line.setWarm(false);
        }
        for (Line line : loadPowerLines) {
            line.setWarm(false);
        }

        /*
        **  Make sure state and power control lines are consistent
        */
        setMainPowerEnable(isCmprPowerOn() ? 1 : isLoadPowerOn() ? -1 : 0);

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
        System.out.println("checkStarted was called");
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
        System.out.println("checkStopped was called");
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void startTicking()
    {
        mon.start();
        super.startTicking();
        publishState();
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        System.out.println("Refrigeration subscale camera test started");
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
        setLoadPower();     // Make sure it's correct
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Sets the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    **  @param  event  The event type
    **
    **  @param  parm   The event parameter
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (parm) {

        case EVENT_ID_MAIN_POWER:
            if (event == Alarm.EVENT_TRIP) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setMainPowerEnable(0);
                    publishState();
                }
            }
            else if (event == Alarm.EVENT_RESET) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
                }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power state.
    **
    **  @param  value    The enabled state value to set: 0 = off,
    **                   &gt; 0 = fully on, &lt; 0 = heat load only.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the main power enabled state")
    public void setMainPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
                if (value > 0) {
                    state &= ~RefrigState.MAIN_LOAD_ONLY_STATE;
                }
                else {
                    state |= RefrigState.MAIN_LOAD_ONLY_STATE;
                }
            }
        }
        else {
            state &= ~(RefrigState.MAIN_POWER_STATE
                        | RefrigState.MAIN_LOAD_ONLY_STATE);
        }
        setMainPower();
        setLoadPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power state.
    **
    **  @param  value  The load power state to set: 0 = off; &gt; 0 = manual;
    **                 &lt; 0 = automatic (temp loop).
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load power enabled state")
    public void setLoadPowerEnable(int value)
    {
        if (loadPowerDevc != null) {
            if (value != 0) {
                state |= RefrigState.LOAD_POWER_STATE;
            }
            else {
                state &= ~RefrigState.LOAD_POWER_STATE;
            }
            if (value < 0) {
                if (tempCtrl != null) {
                    state |= RefrigState.LOAD_TEMP_LOOP_STATE;
                    tempCtrl.setTemp(loadTemp);
                    if ((state & RefrigState.MAIN_POWER_STATE) != 0) {
                        startTempCtrl();
                    }
                }
            }
            else {
                state &= ~RefrigState.LOAD_TEMP_LOOP_STATE;
                stopTempCtrl();
                setLoadPower();
            }
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load trip enabled state on or off.
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load trip enabled state")
    public void setLoadTripEnable(int value)
    {
        boolean enable = (value != 0);
        if (enable) {
            state |= RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        else {
            state &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        for (Channel chan : alarmDsabChans) {
            chan.enableAlarm(false, enable);
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power value.
    **
    **  @param  value  The load power set point.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load power set point")
    public void setLoadPower(double value)
    {
        loadPower = value;
        if (startPower != null) {
            startPower = loadPower;
        }
        setLoadPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load temperature value.
    **
    **  @param  value  The load temperature set point.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load temperature set point")
    public void setLoadTemp(double value)
    {
        loadTemp = value;
        if (tempCtrl != null) {
            tempCtrl.setTemp(loadTemp);
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Saves the current configuration")
    public void saveConfiguration() throws IOException
    {
        getEnvironment().register(configName);
        mon.clearLimitChanges();
        mon.publishState();     // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    **  @return  The full refrigeration state
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Gets the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis(),
                                                loadPower, loadTemp);
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    private void publishState()
    {
        publish(RefrigState.KEY,
                new RefrigState(state, getTickMillis(), loadPower, loadTemp));
    }    


   /**
    ***************************************************************************
    **
    **  Sets the compressor and load power on or off, according to the state.
    **
    ***************************************************************************
    */
    private void setMainPower()
    {
        boolean on = (state & RefrigState.MAIN_POWER_STATE) != 0;
        boolean loadOnly = (state & RefrigState.MAIN_LOAD_ONLY_STATE) != 0;
        for (Line line : cmprPowerLines) {
            line.set(on & !loadOnly);
        }
        for (Line line : loadPowerLines) {
            line.set(on);
        }
        if (loadPowerDevc != null) {
            if (on) {
                loadPowerDevc.enable();
                if ((state & RefrigState.LOAD_TEMP_LOOP_STATE) != 0) {
                    startTempCtrl();
                }
            }
            else {
                stopTempCtrl();
                loadPowerDevc.disable();
            }
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the load power.
    **
    ***************************************************************************
    */
    private void setLoadPower()
    {
        if (loadPowerDevc == null || !loadPowerDevc.isOnline()) return;
        loadPowerDevc.enableOutput(0, (state & RefrigState.LOAD_POWER_STATE) != 0);
        if ((state & RefrigState.LOAD_TEMP_LOOP_STATE) == 0) {
            loadPowerDevc.setPower(0, loadPower);
            startPower = loadPower;
        }
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the hardware compressor power state
    **
    ***************************************************************************
    */
    private boolean isCmprPowerOn()
    {
        if (cmprPowerLines.isEmpty()) return false;
        for (Line line : cmprPowerLines) {
            if (!line.isSet()) return false;
        }
        return true;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state.
    **
    ***************************************************************************
    */
    private boolean isLoadPowerOn()
    {
        if (loadPowerLines.isEmpty()) return false;
        for (Line line : loadPowerLines) {
            if (!line.isSet()) return false;
        }
        return true;
    }
    

   /**
    ***************************************************************************
    **
    **  Starts the temperature control loop.
    **
    ***************************************************************************
    */
    private void startTempCtrl()
    {
        if (tempCtrl != null) {
            if (startPower == null) {
                tempCtrl.restart();
            }
            else {
                tempCtrl.start(startPower);
                startPower = null;
            }
        }
    }
    

   /**
    ***************************************************************************
    **
    **  Stops the temperature control loop.
    **
    ***************************************************************************
    */
    private void stopTempCtrl()
    {
        if (tempCtrl != null) {
            tempCtrl.stop();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the named component of a specified class.
    **
    ***************************************************************************
    */
    private Object getComponent(String name, Class cls)
    {
        Object cmpt = getComponentByName(name);
        if (cmpt != null && !cls.isInstance(cmpt)) {
            cmpt = null;
        }
        return cmpt;
    }

}
