package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.bk.Model1696;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles a B&K model 1696 power supply device.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class BK1696Device extends PowerDevice {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static double MAX_CURRENT = 3.0;
    private final static int
        CHAN_POWER   = 0,
        CHAN_VOLTAGE = 1,
        CHAN_CURRENT = 2,
        N_HW_CHANS   = 3,
        BK_CHANNEL   = 0;

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Model1696.CONN_TYPE_SERIAL);
        cTypeMap.put("FTDI",   Model1696.CONN_TYPE_FTDI);
    }

    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POWER", Channel.TYPE_POWER);
    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private final Model1696 bk = new Model1696();  // Power supply object
    private final String    connTypeS;             // PS connection type string
    private final String    ident;                 // Serial device or FTDI serial

    private int             connType;              // PS connection type
    private final double[]  values = new double[N_HW_CHANS];  // Read values 


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  connType  The connection type: ftdi or serial
    **
    **  @param  devcId    The device ID: USB serial no. or serial device name
    **
    ***************************************************************************
    */
    public BK1696Device(String connType, String devcId)
    {
        connTypeS = connType;
        ident = devcId;
        disabled = true;
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    **  @param  mon  The monitor object
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        fullName = "B&K model 1696 PS (" + ident + ")";
        Integer iConnType = cTypeMap.get(connTypeS);
        if (iConnType == null) {
            connType = -1;
            try {
                mon.reportError(getName(), "connection type", connTypeS);
            }
            catch (Exception e) {
            }
        }
        else {
            connType = iConnType;
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (connType < 0) return;
        try {
            bk.open(connType, ident);
            bk.setTimeout(2.0);
            bk.setCurrent(MAX_CURRENT, BK_CHANNEL);
            setOnline(true);
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (bk != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            bk.close();
        }
        catch (DriverException e) {
        }
        for (int j = 0; j < N_HW_CHANS; j++) {
            values[j] = 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    **
    **  @param  name     The channel name
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @param  type     The channel type string
    **
    **  @param  subtype  The channel subtype string
    **
    **  @return  A two-element array containing the encoded type [0] and
    **           subtype [1] values.
    **
    **  @throws  Exception if any errors found in the parameters.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_HW_CHANS) {
            mon.reportError(name, "HW channel", hwChan);
        }
        Integer iSubtype = 0, iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            mon.reportError(name, "type", type);
        }

        return new int[]{iType, iSubtype};
    }


   /**
    ***************************************************************************
    **
    **  Reads all channels as a group.
    **
    ***************************************************************************
    */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            values[CHAN_VOLTAGE] = bk.readVoltage(BK_CHANNEL);
            values[CHAN_CURRENT] = bk.readCurrent(BK_CHANNEL);
            values[CHAN_POWER] = values[CHAN_VOLTAGE] * values[CHAN_CURRENT];
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    **  @param  hwChan   The hardware channel number.
    **
    **  @param  type     The encoded channel type returned by checkChannel.
    **
    **  @return  The value read from the channel
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return online ? values[hwChan] : super.readChannel(hwChan, type);
    }


   /**
    ***************************************************************************
    **
    **  Sets the output state.
    **
    **  @param  chan   The hardware channel (not used)
    **
    **  @param  value  The output state to set, true or false
    **
    ***************************************************************************
    */
    @Override
    public void setOutput(int chan, boolean value)
    {
        if (!testOnline()) return;
        try {
            bk.setOutput(value, BK_CHANNEL);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the output state.
    **
    **  @param  chan   The hardware channel (not used)
    **
    **  @return  The output state
    **
    ***************************************************************************
    */
    @Override
    public boolean getOutput(int chan)
    {
        if (!testOnline()) return false;
        try {
            return bk.getOutput(BK_CHANNEL);
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the voltage.
    **
    **  @param  chan   The hardware channel (not used)
    **
    **  @param  value  The voltage to set
    **
    ***************************************************************************
    */
    @Override
    public void setVoltage(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            bk.setVoltage(value, BK_CHANNEL);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage.
    **
    **  @param  chan   The hardware channel (not used)
    **
    **  @return  The read voltage
    **
    ***************************************************************************
    */
    @Override
    public double readVoltage(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return bk.readVoltage(BK_CHANNEL);
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the current.
    **
    **  @param  chan   The hardware channel (not used)
    **
    **  @param  value  The current to set
    **
    ***************************************************************************
    */
    @Override
    public void setCurrent(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            bk.setCurrent(value, BK_CHANNEL);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the current.
    **
    **  @param  chan   The hardware channel (not used)
    **
    **  @return  The read current
    **
    ***************************************************************************
    */
    @Override
    public double readCurrent(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return bk.readCurrent(BK_CHANNEL);
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }

}
