package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import javax.usb.UsbDisconnectedException;
import javax.usb.UsbEncodingException;
import javax.usb.UsbException;
import javax.usb.UsbNotActiveException;
import javax.usb.UsbNotClaimedException;
import javax.usb.UsbNotOpenException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles a Measurement Computing DAQ box.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MCCDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        N_DEV_CHANS = 8;
    final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> didMap = new HashMap<>();
    static {
        didMap.put("TC_AI", MccUsb.USB_TC_AI_DID);
        didMap.put("TC",    MccUsb.USB_TC_DID);
        didMap.put("TEMP",  MccUsb.USB_TEMP_DID);
    }

    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   Channel.TYPE_TEMP);
        typeMap.put("VOLTS",  Channel.TYPE_VOLTS);
        typeMap.put("SWITCH", Channel.TYPE_SWITCH);
    }

    private final static Map<String,Integer> tSbTypMap = new HashMap<>();
    static {
        tSbTypMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
        tSbTypMap.put("RTD", MccUsb.STP_RTD);
    }

    private final static Map<String,Integer> tcSbTypMap = new HashMap<>();
    static {
        tcSbTypMap.put("J", MccUsb.TC_TYPE_J);
        tcSbTypMap.put("K", MccUsb.TC_TYPE_K);
        tcSbTypMap.put("T", MccUsb.TC_TYPE_T);
        tcSbTypMap.put("E", MccUsb.TC_TYPE_E);
        tcSbTypMap.put("R", MccUsb.TC_TYPE_R);
        tcSbTypMap.put("S", MccUsb.TC_TYPE_S);
        tcSbTypMap.put("B", MccUsb.TC_TYPE_B);
        tcSbTypMap.put("N", MccUsb.TC_TYPE_N);
    }

    private final static Map<String,Integer> rtSbTypMap = new HashMap<>();
    static {
        rtSbTypMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
        rtSbTypMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
        rtSbTypMap.put("3WIRE",  MccUsb.TCT_3WIRE);
        rtSbTypMap.put("4WIRE",  MccUsb.TCT_4WIRE);
    }

    private final static Map<String,Integer> vSbTypMap = new HashMap<>();
    static {
        vSbTypMap.put("10V",   MccUsb.RANGE_10V);
        vSbTypMap.put("5V",    MccUsb.RANGE_5V);
        vSbTypMap.put("2.5V",  MccUsb.RANGE_2_5V);
        vSbTypMap.put("1.25V", MccUsb.RANGE_1_25V);
    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    String  dName;    // MCC device name
    String  serial;   // USB serial number
    int     did;      // MCC device ID
    MccUsb  mcc;      // Associated USB object
    int     tcMask;   // Mask of available thermocouple channels
    int     rtMask;   // Mask of available RT channels
    int     vMask;    // Mask of available voltage channels


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public MCCDevice(String devName, String serialNo)
    {
        dName = devName;
        serial = serialNo;
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        fullName = "MCC " + dName + " module"
                     + (serial == null ? "" : " (" + serial + ")");
        Integer iDid = didMap.get(dName);
        if (iDid == null) {
            did = -1;
            try {
                mon.reportError(getName(), "device name", dName);
            }
            catch (Exception e) {
            }
        }
        else {
            did = iDid;
            if (did == MccUsb.USB_TC_AI_DID) {
                tcMask = 0x0f;
                vMask = 0xf0;
            }
            else if (did == MccUsb.USB_TC_DID) {
                tcMask = 0xff;
            }
            else if (did == MccUsb.USB_TEMP_DID) {
                tcMask = 0xff;
                rtMask = 0xff;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (did < 0) return;
        try {
            if (!inited || mcc == null) {
                mcc = new MccUsb();
            }
            mcc.open(did, serial, true);
            mcc.blink();
            for (int line = 0, mask = lineMask; line < N_DEV_CHANS;
                 line++, mask >>= 1) {
                mcc.dioConfigBit(line, 1 - (mask & 1));
            }
            if (did == MccUsb.USB_TC_AI_DID) {
                for (int j = 0; j < N_DEV_CHANS; j++) {
                    mcc.configAlarm(j, 0, 0, 0f, 0f);
                }
            }
            else if (did == MccUsb.USB_TEMP_DID) {
                for (int j = 0; j < N_DEV_CHANS; j += 2) {
                    mcc.setSensorType(j, MccUsb.STP_DISABLED);
                }
            }
            setOnline(true);
            initSensors();
            setOutputLines();
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (Exception e) {
            checkUsbException(e);
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (mcc != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            mcc.close();
        }
        catch (Exception e) {
            checkUsbException(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype)
        throws Exception
    {
        Integer iSubtype = 0, iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            mon.reportError(name, "type", type);
        }

        String sbtype = subtype.toUpperCase();
        if (iType == Channel.TYPE_TEMP) {
            String[] sbtypes = sbtype.split(":", -1);
            if (sbtypes.length < 2) {
                mon.reportError(name, "subtype", subtype);
            }
            iSubtype = tSbTypMap.get(sbtypes[0]);
            if (iSubtype == null) {
                mon.reportError(name, "subtype", subtype);
            }
            Integer iStp = (iSubtype == MccUsb.STP_RTD)
                             ? rtSbTypMap.get(sbtypes[1])
                             : tcSbTypMap.get(sbtypes[1]);
            if (iStp == null) {
                mon.reportError(name, "subtype", subtype);
            }
            iSubtype |= (iStp << 8);
        }
        else if (iType == Channel.TYPE_VOLTS) {
            iSubtype = vSbTypMap.get(sbtype);
            if (iSubtype == null) {
                mon.reportError(name, "subtype", subtype);
            }
        }

        boolean chanOk = true;
        if (iType == Channel.TYPE_TEMP) {
            int mask = (iSubtype & 0xff) == MccUsb.STP_RTD ? rtMask : tcMask;
            if ((1 << hwChan & mask) == 0) {
                chanOk = false;
            }
        }
        else if (iType == Channel.TYPE_VOLTS) {
            if ((1 << hwChan & vMask) == 0) {
                chanOk = false;
            }
        }
        else {
            if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
                chanOk = false;
            }
        }
        if (!chanOk) {
            mon.reportError(name, "hw channel number", hwChan);
        }

        return new int[]{iType, iSubtype};
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    ***************************************************************************
    */
    @Override
    protected void initChannel(int chan, int type, int subtype)
    {
        if (!online) return;

        try {
            if (type == Channel.TYPE_TEMP) {
                int sType = subtype & 0xff, tType = subtype >> 8;
                mcc.setSensorType(chan, sType);
                if (sType == MccUsb.STP_RTD) {
                    mcc.setThermConnType(chan, tType);
                    mcc.setExcitation(chan, MccUsb.CEX_210UA);
                    mcc.setGain(chan, MccUsb.GAIN_4X);
                    mcc.setGain(chan + 1, MccUsb.GAIN_4X);
                    for (int j = 0; j < RTD_COEFFS.length; j++) {
                        mcc.setCoefficient(chan, j, RTD_COEFFS[j]);
                    }
                }
                else {
                    mcc.setTCType(chan, tType);
                }
            }
            else if (type == Channel.TYPE_VOLTS) {
                mcc.setGain(chan, subtype);
                mcc.setVoltageConnType(chan, MccUsb.VCT_DIFFERENTIAL);
            }
        }
        catch (Exception e) {
            checkUsbException(e);
            log.error("Error configuring " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int chan, int type)
    {
        double value = 0;
        if (!online) return value;

        if (type == Channel.TYPE_SWITCH) {
            try {
                value = mcc.dioInBit(chan);
            }
            catch (Exception e) {
                checkUsbException(e);
                log.error("Error reading DIO line: " + e);
                setOnline(false);
            }
        }
        else {
            try {
                value = mcc.adcIn(chan, 0);
            }
            catch (Exception e) {
                checkUsbException(e);
                log.error("Error reading ADC: " + e);
                setOnline(false);
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Checks an output line number.
    **
    ***************************************************************************
    */
    @Override
    protected void checkHwLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= N_DEV_CHANS) {
            mon.reportError(name, "line", line);
        }
        addLine(line);
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    ***************************************************************************
    */
    @Override
    protected void setHwLine(int id, boolean on)
    {
        try {
            mcc.dioOutBit(id, on ? 1 : 0);
        }
        catch (Exception e) {
            checkUsbException(e);
            log.error("Error setting DIO line: " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the set state of an output line.
    **
    ***************************************************************************
    */
    @Override
    protected boolean isHwLineSet(int id)
    {
        try {
            return mcc.dioInBit(id) != 0;
        }
        catch (Exception e) {
            checkUsbException(e);
            log.error("Error reading DIO line: " + e);
            setOnline(false);
            return false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether an exception is USB-related
    **
    ***************************************************************************
    */
    private static void checkUsbException(Exception e)
    {
        if (e instanceof UsbException
            || e instanceof UsbDisconnectedException
            || e instanceof UsbEncodingException
            || e instanceof UsbNotActiveException
            || e instanceof UsbNotOpenException
            || e instanceof UsbNotClaimedException) return;
        throw (RuntimeException)e;
    }

}
