package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class LongLines extends Module implements Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        EVENT_ID_MAIN_POWER = 0,
        EVENT_ID_LOAD_POWER = 1;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    Monitor mon;
    String configName = "test";
    boolean coldStart, running, loadAlarm;
    int state = RefrigState.LOAD_POWER_STATE;
    Line mainPowerLine, loadPowerLine;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public LongLines(String name, int tickMillis, String configName)
    {
        super(name, tickMillis);
        this.configName = configName;
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        mon = new Monitor(this, this, log);
        mon.initConfiguration();
        mainPowerLine = mon.getLine("MainPower");
        loadPowerLine = mon.getLine("LoadPower");

        /*
        **  Initialize the hardware
        */
        if (mainPowerLine != null) {
            mainPowerLine.setWarm(!coldStart);
        }
        mon.initSensors();
        if (mainPowerLine != null) {
            mainPowerLine.setWarm(false);
        }

        /*
        **  Set output lines
        */
        setMainPowerEnable(isMainPowerOn() ? 1 : 0);
        setLoadPower();

        /*
        **  Start the monitoring
        */
        mon.start();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
//            getSubsystem().switchToEngineeringMode();

            System.out.println("Refrigeration long lines test system started");
            publishState();
            mon.publishState();    // For any GUIs
            mon.publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Broadcast the trending data
        */
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Sets the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (event) {

        case Alarm.EVENT_TRIP:
            if (parm == EVENT_ID_MAIN_POWER) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setMainPowerEnable(0);
                    publishState();
               }
            }
            else if (parm == EVENT_ID_LOAD_POWER) {
                loadAlarm = true;
                if ((state & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    state &= ~RefrigState.LOAD_POWER_STATE;
                    setLoadPower();
                    publishState();
                }
            }
            break;
            
        case Alarm.EVENT_RESET:
            if (parm == EVENT_ID_MAIN_POWER) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
               }
            }
            if (parm == EVENT_ID_LOAD_POWER) {
                loadAlarm = false;
                if ((state & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    state |= RefrigState.LOAD_POWER_STATE;
                    setLoadPower();
                    publishState();
                }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    **  @param  publish  If true, publish the new state on the status bus.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the main power enabled state")
    public void setMainPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
            }
        }
        else {
            state &= ~RefrigState.MAIN_POWER_STATE;
        }
        setMainPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power trip enabled state on or off.
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load power trip enabled state")
    public void setLoadTripEnable(int value)
    {
        if (value != 0) {
            state |= RefrigState.LOAD_TRIP_ENAB_STATE;
            if (loadAlarm) {
                state &= ~RefrigState.LOAD_POWER_STATE;
                setLoadPower();
            }
        }
        else {
            state &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
            if ((state & RefrigState.LOAD_POWER_STATE) == 0) {
                state |= RefrigState.LOAD_POWER_STATE;
                setLoadPower();
            }
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Saves the current configuration")
    public void saveConfiguration() throws IOException
    {
        register(configName, "");
        mon.clearLimitChanges();
        mon.publishState();     // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Gets the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis());
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Gets the operating state word.
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    int getState()
    {
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        if (running) {
            RefrigState refgState = new RefrigState(state, getTickMillis());
            sendToStatus(new RefrigStateStatus(refgState));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Sets the main power on or off, according to its state.
    **
    ***************************************************************************
    */
    void setMainPower()
    {
        if (mainPowerLine != null) {
            mainPowerLine.set((state & RefrigState.MAIN_POWER_STATE) != 0);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the load power on or off, according to its state.
    **
    ***************************************************************************
    */
    void setLoadPower()
    {
        if (loadPowerLine != null) {
            loadPowerLine.set((state & RefrigState.LOAD_POWER_STATE) != 0);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the hardware main power state
    **
    ***************************************************************************
    */
    boolean isMainPowerOn()
    {
        return mainPowerLine != null && mainPowerLine.isSet();
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state
    **
    ***************************************************************************
    */
    boolean isLoadPowerOn()
    {
        return loadPowerLine != null && loadPowerLine.isSet();
    }
    
}
