package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.drivers.wattsup.WattsUp;

/**
 ******************************************************************************
 **
 **  Handles a WattsUp power meter.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class WUDevice extends Device implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        CHAN_WATTS  = 0,
        CHAN_VOLTS  = 1,
        N_DEV_CHANS = 2,
        LOG_PERIOD  = 1;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    String   node;        // Name of the node attached to the meter
    String   serial;      // The serial number of the meter
    WattsUp  wtu;         // WattsUp hardware object
    double[] value;       // Current values: power, voltage


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public WUDevice(String nodeName, String serialNo)
    {
        type = TYPE_WATTS;
        node = nodeName;
        serial = serialNo;
        fullName = "WattsUp meter (" + (node == null ? "local" : node)
                     + (serial == null ? "" : ":" + serial) + ")";
        value = new double[N_DEV_CHANS];
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    @Override
    int[] checkChannel(String name, int hwChan, String type, String subtype)
        throws Exception
    {
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            refg.reportError(name, "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


   /**
    ***************************************************************************
    **
    **  Performs basic initialization.
    **
    ***************************************************************************
    */
    @Override
    void initialize()
    {
        try {
            if (!inited || wtu == null) {
                wtu = new WattsUp();
                wtu.addListener(this);
            }
            wtu.open(node, 0, serial);
            wtu.restart();
            setOnline(true);
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (Exception e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    ***************************************************************************
    */
    @Override
    void initChannel(int chan, int type, int subtype)
    {
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    @Override
    double readChannel(int chan)
    {
        return value[chan];
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    **  This is a no-op
    **
    ***************************************************************************
    */
    @Override
    void close()
    {
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter powered state.
    **
    **  Performs meter setup upon power-on.
    **
    **  @param  on  Whether the meter is powered on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields((1 << WattsUp.FLD_WATTS) |
                                    (1 << WattsUp.FLD_VOLTS));
                wtu.setExternalLogging(LOG_PERIOD);
            }
            catch (Exception e) {
                log.error("Error configuring " + fullName + ": " + e);
            }
        }
        else {
            for (int j = 0; j < N_DEV_CHANS; j++) {
                value[j] = 0F;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter open state.
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        setOnline(false);
        log.error("Disconnected from " + fullName);
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        value[CHAN_WATTS] = data[WattsUp.FLD_WATTS];
        value[CHAN_VOLTS] = data[WattsUp.FLD_VOLTS];
    }

}
